//----------------------------------------------------------------------
// Best viewed with tabs every 4 columns
//----------------------------------------------------------------------

import java.util.Random;
import java.util.Scanner;

/** CombinationPuzzle
 *
 * The objective is to solve a puzzle involving a cycle of combination
 * dials.
 *
 * THE FOLLOWING COMMENT CAN BE REMOVED FROM THE CODE WE GIVE THE STUDENTS
 *
 * You are given a n-element cycle integers, each number in the range
 * from 0 to 9. For i between 0 and n-1, let dial[i] denote the value of
 * the dial in row i and column j. Let us assume that indices are taken
 * modulo n, so dial[n] is equivalent to dial[0]. Let us also assume
 * that dial values are taken modulo 10, so (for example), the dial
 * values -13, -3, 7, 17, 27, and so on are all equivalent to the dial
 * value 7.
 *
 * A single step involves incrementing increasing or decreasing the
 * value of any of the dials by 1. Increasing the value of a dial
 * dial[i] has the effect of decreasing dial[i+1] by one. Symmetrically,
 * decreasing the value of dial[i] has the effect of increasing
 * dial[i+1] by 1.
 *
 * The objective is determine a minimum number of steps needed to set
 * all the dial values to zero. The output is in the form of a string.
 * If it is impossible to do, the string will consist of the single
 * character "X" (capital 'X'). Otherwise, the string will contain a
 * series of steps separated by spaces. Each step has the form {+,-}
 * followed by a dial index, indicating that this dial should be
 * incremented (+) or decremented (-). For example, the entry "+3"
 * increments the value of dial 3 (thus decrementing dial 4). The entry
 * "-17" means that dial 17 should be decremented (thus incrementing
 * dial 18). Decrementing a dial whose value is 0 results in the value
 * 9, and incrementing a dial whose value is 9 results in the value 0.
 *
 * For example, if n = 8, the string "+1 +5 -3 -3 -7" means that dials 1
 * and 5 are each incremented by one (resulting in dials 2 and 6 each
 * being decremented), dial 3 is decremented twice (resulting in dial 4
 * being incremented twice), and dial 7 is decremented (resulting in
 * dial 0 being incremented).
 *
 * The main program reads the input, invokes the function SolvePuzzle to
 * solve the puzzle, and then summarizes the resulting output.
 */

public class CombinationPuzzle {

	/** 
	 *  Main Program - Reads the input and outputs the final answer.
	 */
	public static void main(String[] args) throws Exception {
		Scanner scanner = new Scanner(System.in);
		try {
			int numTrials = scanner.nextInt();	// number of trials to run

			for(int t = 1; t <= numTrials; t++) {
				int n = scanner.nextInt();		// number of dials
				if (n <= 1) {
					System.err.println("Error - Must have at least one dial");
					return;
				}

				int[] dial = new int[n];		// the dials
				int[] dialCopy = new int[n];	// ...and a copy 

				for (int i = 0; i < n; i++) {	// input initial dial settings
					dialCopy[i] = dial[i] = scanner.nextInt();
												// check
					if (dial[i] < 0 || dial[i] > 9) {
						System.err.println("Error - Dial value must be in range 0..9");
						return;
					}
				}
												// summarize the input
				System.out.print("Initial dial settings = [");
				for (int i = 0; i < n; i++) {
					System.out.print(" " + dial[i]);
				}
				System.out.println(" ]");
												// solve it
				String solution = SolvePuzzle(dial);
												// summarize the results
				SummarizeSolution(solution, dialCopy);
			}
		}
		finally {								// close scanner resource
			if (scanner != null) {
				scanner.close();
			}
		}
		System.exit(0); // not sure why needed - but it keeps Eclipse happy
	}

	/**
	 * SummarizeSolution
	 *
	 * Summarize solution and check its basic validity.
	 *
	 * This does NOT check for optimality.
	 *
	 */
	static void SummarizeSolution(String solution, int[] dial) {
		System.out.println("Raw output = \"" + solution + "\"");
		System.out.println(" Analysis:");
		Scanner ssolution = new Scanner(solution);
		try {
												// no solution?
			if (solution.compareToIgnoreCase("X") == 0) {
				System.out.println("  >> No solution");
			}
			else {								// found a solution
				int totalCost = 0;
				while (ssolution.hasNext()) {
												// get the next step
					String step = ssolution.next();

					int di = 0;					// dial index
					int sign;					// increment or decrement
					if (step.charAt(0) == '+') {
						sign = +1;
					}
					else if (step.charAt(0) == '-') {
						sign = -1;
					}
					else {
						System.err.println("  !! Format error - Step must start with '+' or '-'");
						return;
					}

					try {						// get dial index
						di = Integer.parseInt(step);
						if (di < 0) di = -di;	// map +x and -x to x
						if (di > dial.length) {
							System.err.println("  !! Format error - invalid dial index");
							return;
						}
					}
					catch (NumberFormatException e) {
						System.err.println("  !! Format error - dial index is not an integer");
						return;
					}
												// next dial in the sequence
					int dii = (di + 1) % dial.length;
												// increment/decrement mod 10
					dial[di] = (dial[di] + sign + 10) % 10;
					dial[dii] = (dial[dii] - sign + 10) % 10;
					totalCost++;
				}
												// check final dials are zero
				boolean allZero = true;
				for (int i = 0; i < dial.length; i++) {
					if (dial[i] != 0) allZero = false;
				}
				if (!allZero) {					// incorrect final setting?
					System.err.println("  !! Error - Final dial settings must all be zero");
					System.out.print("     Final settings = [");
					for (int i = 0; i < dial.length; i++) {
						System.out.print(" " + dial[i]);
					}
					System.out.println(" ]");
					return;
				}
				System.out.println("  >> Valid solution. Total cost = " + totalCost);
			}
			System.out.println();
		}
		finally {								// close scanner resource
			if (ssolution != null) {
				ssolution.close();
			}
		}
	}

	private static String SolvePuzzle(int[] dial) {

		/* -------------------- INSERT CODE HERE ----------------------*/

		/* --------------------- HOW IT WORKS --------------------------
		 *
		 * Throughout, we will assume that arithmetic on dials is
		 * performed mod 10, and arithmetic on dial indices is performed
		 * mod n, where n is the number of dials.
		 *
		 * It is useful to observe that the final value of dial i
		 * depends on the number of increments and decrement steps 
		 * involving dials i and i-1, but not the order in which they
		 * are applied. Thus, it never profits us to both increment
		 * and decrement the same dial, since these two operations
		 * simply cancel each other out.
		 *
		 * With this in mind, let's define x[i] to be the number of
		 * times that dial i is incremented (if it is incremented) or
		 * the negative of the number of times that it is decremented
		 * (if it is decremented). We have
		 *
		 *   final[i] = (dial[i] + x[i] - x[i-1]) mod 10
		 *
		 * Since we want final[i] = 0 for all i, this yields a system of
		 * linear equations (mod 10):
		 *
		 *   (x[0] - x[1] = dial[1]) mod 10
		 *   (x[1] - x[2] = dial[2]) mod 10
		 *   ...
		 *   (x[n-1] - x[0] = dial[0]) mod 10
		 *
		 * There are n equations and n unknowns, but there is a linear
		 * dependence among the equations (in particular the sum of all
		 * the equations is zero), so a solution doesn't always exist.
		 *
		 * Our quick-and-dirty solution is to guess the value of x[0]
		 * (from among the 10 possible choices), and then use this to
		 * determine the rest:
		 *
		 *   x[1] = (x[0] - dial[1]) mod 10
		 *   x[2] = (x[1] - dial[2]) mod 10
		 *   ...
		 *   x[n-1] = (x[n-2] - dial[n-1]) mod 10
		 *
		 * This is not quite correct since it doesn't distinguish
		 * between positive and negative values, and simply maps all
		 * values to the range 0 to 9. (We'll fix this below.)
		 *
		 * Finally, we check that (x[0] = x[n-1] - dial[0]) mod 10. If
		 * so, we have a valid solution.
		 *
		 * As mentioned above, as computed, all the x[i]'s are
		 * nonnegative. If x[i] <= 5, we will increment the dial x[i]
		 * times, and if x[i] >= 6, we will decrement the dial 10 - x[i]
		 * times towards 10. This is computed in the function addSteps.
		 *
		 * Among the possible guesses for x[0], we return the solution
		 * that resulted from the guessed value for x[0] that yields the
		 * smallest total number of steps.
		 *
		 * Technical note: When taking quantities mod 10, it is
		 * important that the number be nonnegative. So, when negative
		 * values might result, we add 10 before taking the mod.
		 */

		int n = dial.length;				// number of dials
		int x[] = new int[n];				// increment values
		String solution = new String("X");	// best solution so far
		int minCost = (int) Integer.MAX_VALUE; // best cost so far

											// try guesses for x[0]
		for (int guess = 0; guess < 10; guess++) {
			
			x[0] = guess;					// initial guess for x[0]
											// current solution
			StringBuilder thisSolution = new StringBuilder();
											// add steps for dial 0
			int thisCost = addSteps(0, guess, thisSolution);

			for (int i = 1; i < n; i++) {	// get x[i] from x[i-1]
				x[i] = (x[i-1] - dial[i] + 10) % 10;
											// add steps for dial i
				thisCost += addSteps(i, x[i], thisSolution);
			}
											// valid solution?
			if (x[0] == (x[n-1] - dial[0] + 10) % 10) {
				if (thisCost < minCost) {	// best so far?
					minCost = thisCost;
					solution = thisSolution.toString();
				}
			}
		}
		return solution;
	}

											// add nSteps to dial i
	static int addSteps(int i, int nSteps, StringBuilder solution) {
		int cost;
		char sign;
		if (nSteps <= 5) {					// step forward
			cost = nSteps;
			sign = '+';
		}
		else {								// step backward
			cost = 10 - nSteps;
			sign = '-';
		}
		for (int j = 0; j < cost; j++) {	// add steps to solution
			if (solution.length() != 0) solution.append(" ");
			solution.append("" + sign + i);
		}
		return cost;						// return final cost

		/* --------------------- END OF INSERTION ---------------------*/
	}

}
