package cmsc420_s23;

//---------------------------------------------------------------------
// Disclaimer: This program has not been thoroughly tested or
// debugged. If you find any errors, let me know.
//---------------------------------------------------------------------

import java.util.ArrayList;
import java.util.List;

//---------------------------------------------------------------------
// Author: Dave Mount
// For: CMSC 420
// Date: Spring 2023
//
// This is a variant of the scapegoat tree. See Lecture 12 for a
// description of this data structure. This implementation differs from
// the standard scapegoat tree in the following respects:
//
// - This is an extended binary tree. Each internal node stores a
// cutting value and all data resides in the external nodes.
//
// - The selection of the scapegoat node is performed top-down, rather
// than bottom-up.
//
// - Rather than computing subtree sizes on the fly, we instead store
// the size of each subtree inside each internal node.
//
// - Deletion is done lazily, by simply setting the key to null.
//---------------------------------------------------------------------

public class ScapegoatTree<Key extends Comparable<Key>> {

	// -----------------------------------------------------------------
	// Generic Node Type
	// -----------------------------------------------------------------

	private abstract class Node { // generic node type

		// -----------------------------------------------------------------
		// Standard dictionary helpers
		// -----------------------------------------------------------------

		abstract Key find(Key x); // find key in subtree

		abstract Node insert(Key x) throws Exception; // insert a single key
		
		abstract Node delete(Key x) throws Exception; // delete entry with key x

		abstract ArrayList<String> list(); // list nodes in reverse preorder

		// -----------------------------------------------------------------
		// Other utilities
		// -----------------------------------------------------------------
		
		abstract Node rebuildAfterInsert(Key x); // check for rebuild after insertion
		
		abstract void buildKeyList(List<Key> keys); // build a list of keys in this subtree
		
		abstract Node rebuild(); // rebuild subtree rooted at this node
		
		abstract int getSize(); // get number of entries in subtree

		abstract public String toString(); // string representation
	}

	// -----------------------------------------------------------------
	// Internal node
	// -----------------------------------------------------------------

	private class InternalNode extends Node {

		Key cutVal; // cut value
		Node left, right; // children
		int size; // number of entries in this subtree

		/**
		 * Constructor
		 */
		InternalNode(Key cutVal, Node left, Node right) {
			this.cutVal = cutVal;
			this.left = left;
			this.right = right;
			this.size = left.getSize() + right.getSize();
		}

		/**
		 * Find key in this subtree.
		 */
		Key find(Key x) {
			if (x.compareTo(cutVal) < 0) {
				return left.find(x);
			}
			else {
				return right.find(x);
			}
		}

		/**
		 * Insert a single key.
		 */
		Node insert(Key x) throws Exception {
			if (x.compareTo(cutVal) < 0) {
				left = left.insert(x);
			} else {
				right = right.insert(x);
			}
			size += 1; // increment subtree size
			return this;
		}

		/**
		 * Delete a key from this subtree. We recurse on the side
		 * containing the key.
		 */
		Node delete(Key x) throws Exception {
			if (x.compareTo(cutVal) < 0) {
				left = left.delete(x);
			}
			else {
				right = right.delete(x);
			}
			size -= 1; // decrement subtree size
			return this;
		}

		/**
		 * Get a list of the nodes in reverse preorder.
		 */
		ArrayList<String> list() {
			ArrayList<String> list = new ArrayList<String>();
			list.add(toString()); // add this node
			list.addAll(right.list()); // add right
			list.addAll(left.list()); // add left
			return list;
		}
		
		/**
		 * Check and rebuild if needed after insertion. Unlike the
		 * standard scapegoat tree, this is done top-down. A subtree 
		 * is rebuilt if either child has more that 2/3 of the 
		 * total size.
		 */
		Node rebuildAfterInsert(Key x) {
			int leftSize = left.getSize(); // sizes of the subtrees
			int rightSize = right.getSize();
			if (3*leftSize > 2*size || 3*rightSize > 2*size) {
				return rebuild();
			} else if (x.compareTo(cutVal) < 0) {
				left = left.rebuildAfterInsert(x);
			} else {
				right = right.rebuildAfterInsert(x);
			}
			return this;
		}

		/**
		 * Rebuild the subtree rooted at this node. We first compute a sorted
		 * list of keys and then rebuild the subtree recursively.
		 */
		Node rebuild() {
			System.out.println("Rebuilding the subtree rooted at " + this);
			List<Key> keys = new ArrayList<Key>(); // list of keys
			buildKeyList(keys); // collect the non-empty keys in this subtree
			Node result = bulkCreate(keys); // build a new tree
			System.out.println("...The new subtree is rooted at " + result);
			return result;
		}

		/**
		 * Builds a list of keys in this subtree.
		 */
		void buildKeyList(List<Key> keys) {
			left.buildKeyList(keys); // add left
			right.buildKeyList(keys); // add right
		}

		/**
		 * Get subtree size.
		 */
		int getSize() {
			return size;
		}

		/**
		 * String representation.
		 */
		public String toString() {
			return "(" + cutVal + ") size: " + size;
		}
	}

	// -----------------------------------------------------------------
	// External node
	// -----------------------------------------------------------------

	private class ExternalNode extends Node {

		Key key; // the key (null if empty)

		/**
		 * Constructor.
		 */
		ExternalNode(Key x) {
			this.key = x;
		}

		/**
		 * Find key in external node.
		 */
		Key find(Key x) {
			if (x != null && x.compareTo(key) == 0) {
				return key;
			} else {
				return null;
			}
		}

		/**
		 * Insert a single key. If this node is empty, it adds the key
		 * here. Otherwise, we create a list with the current and new
		 * key and invoke bulkCreate to build a new 2-element tree to
		 * store them.
		 */
		Node insert(Key x) throws Exception {
			if (key == null) {
				key = x;
				return this;
			}
			else {
				if (x.compareTo(key) == 0) {
					throw new Exception("Insertion of duplicate key");
				}
				ArrayList<Key> keys = new ArrayList<Key>();
				if (x.compareTo(key) < 0) { // add {x, key} in order
					keys.add(x);
					keys.add(key);
				} else {
					keys.add(key);
					keys.add(x);
				}
				return bulkCreate(keys); // create new tree with both keys
			}
		}

		/**
		 * Delete a key from this node.
		 */
		Node delete(Key x) throws Exception {
			if (key == null || x.compareTo(key) != 0) {
				throw new Exception("Deletion of nonexistent key");
			}
			key = null;
			return this;
		}

		/**
		 * List the contents of this node
		 */
		ArrayList<String> list() {
			ArrayList<String> list = new ArrayList<String>();
			list.add(toString()); // add this node
			return list;
		}

		/**
		 * Rebuild if needed after insert. (No effect on externals.)
		 */
		Node rebuildAfterInsert(Key x) {
			return this;
		}
		
		/**
		 * Rebuild this subtree. (No effect on externals.)
		 */
		Node rebuild() {
			return this;
		}

		/**
		 * Builds a list of (non-null) keys in this subtree.
		 */
		void buildKeyList(List<Key> keys) {
			if (key != null) keys.add(key);
		}

		/**
		 * Get subtree size.
		 */
		int getSize() {
			return key == null ? 0 : 1;
		}

		/**
		 * String representation.
		 */
		public String toString() {
			return "[" + (key == null ? "null" : key.toString()) + "]";
		}
	}
	
	// -----------------------------------------------------------------
	// ScapegoatTree private data
	// -----------------------------------------------------------------

	private Node root; // root of the tree
	private int n; // number of entries in the tree
	private int m; // global insert counter

	// -----------------------------------------------------------------
	// ScapegoatTree public members
	// -----------------------------------------------------------------

	/**
	 * Creates an empty tree.
	 */
	public ScapegoatTree() {
		clear();
	}

	/**
	 * Clear the tree.
	 */
	public void clear() {
		root = new ExternalNode(null);
		n = m = 0;
	}

	/**
	 * Get the current number of keys.
	 */
	public int size() {
		return n;
	}

	/**
	 * Find a key.
	 */
	public Key find(Key x) {
		return root.find(x);
	}

	/**
	 * Insert a key.
	 */
	public void insert(Key x) throws Exception {
		root = root.insert(x); // invoke the insert helper
		root = root.rebuildAfterInsert(x); // rebuild if needed
		n += 1; // update entry count
		m += 1; // increment insert counter
	}
	
	/**
	 * Delete a key.
	 */

	public void delete(Key x) throws Exception {
		root = root.delete(x);
		n -= 1; // update entry count
		if (m > 2*n) { // time to rebuild?
			root = root.rebuild(); // rebuild entire tree
			m = n; // reset the insert counter
		}
	}

	/**
	 * Get a list of nodes in reverse preorder.
	 */
	public ArrayList<String> list() {
		return root.list();
	}

	// -----------------------------------------------------------------
	// ScapegoatTree private utilities
	// -----------------------------------------------------------------
	
	/**
	 * Create a new subtree from a list of keys. If there are 0 or 1
	 * keys, create an external node with this key. Otherwise, split
	 * the keys about the median and build subtrees recursively for
	 * each.
	 */
	private Node bulkCreate(List<Key> keys) {
		int k = keys.size(); // number of keys
		if (k == 0) { // no keys
			return new ExternalNode(null); // empty external node
		} else if (k == 1) { // one key
			return new ExternalNode(keys.get(0)); // node with this key
		} else { // two or more keys
			int j = k/2; // median index of the array
			Key cutVal = keys.get(j); // cutting value of new node
			Node left = bulkCreate(keys.subList(0, j)); // build left subtree keys[0..j-1]
			Node right = bulkCreate(keys.subList(j, k)); // build left subtree keys[j..k-1]
			return new InternalNode(cutVal, left, right); // assemble the final tree
		}
	}
}
