/*  cmds.c: BetaFTPD command handlers
    Copyright (C) 1999 Steinar H. Gunderson

    This program is is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2 if the
    License as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define _GNU_SOURCE

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <crypt.h>

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#include <glob.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_SHADOW_H
#include <shadow.h>
#endif

#include <ftpd.h>
#include <cmds.h>

/*
 * This macro should be used when we don't have a clear image of exacltly
 * *what* could go wrong, but still want to secure ourselves. In other words,
 * it is *not* a debugging macro. Besides, using it saves a few bytes. Oh no,
 * they got lost in this comment. Better end it.
 */ 
#define TRAP_ERROR(x, num, y) \
	if (x) { \
	        numeric(c, num, strerror(errno)); \
	        y; \
	}

extern fd_set master_fds, master_send_fds;
char anon_root_dir[128];
struct handler {
	char cmd_name[6];
	char add_cmlen;		/* =1 if the command takes an argument */
	void (*callback)(struct conn * const);
	char min_auth;
	char do_setuid;		/* =1 if root is not *really* needed */
};

static const struct handler handler_table[] = {
	{ "user ", 1, cmd_user, 0, 0 },
	{ "pass ", 1, cmd_pass, 1, 0 },
	{ "port ", 1, cmd_port, 3, 1 },
	{ "pasv" , 0, cmd_pasv, 3, 1 },
	{ "pwd"  , 0, cmd_pwd,  3, 1 },
	{ "cwd " , 1, cmd_cwd,  3, 1 },
	{ "cdup" , 0, cmd_cdup, 3, 1 },
	{ "rest ", 1, cmd_rest, 3, 1 },
	{ "retr ", 1, cmd_retr, 3, 1 },
	{ "list",  0, cmd_list, 3, 1 },
	{ "nlst",  0, cmd_nlst, 3, 1 },
	{ "type ", 1, cmd_type, 3, 1 },
	{ "mode ", 1, cmd_mode, 3, 1 },
	{ "stru ", 1, cmd_stru, 3, 1 },
	{ "size ", 1, cmd_size, 3, 1 },
	{ "mdtm ", 1, cmd_mdtm, 3, 1 },
	{ "noop" , 0, cmd_noop, 3, 1 },
	{ "syst" , 0, cmd_syst, 0, 1 },
	{ "quit" , 0, cmd_quit, 0, 1 },
	{ "xcup" , 0, cmd_cdup, 3, 1 },
	{ "xcwd" , 0, cmd_cwd,  3, 1 },
	{ "xpwd" , 0, cmd_pwd,  3, 1 },
#if WANT_UPLOAD
	{ "stor ", 1, cmd_stor, 3, 1 },
#endif
	{ ""     , 0, NULL,     0, 0 }
};

int do_chdir(struct conn * const c, const char * const newd)
{
	char chd[512], temp[512];

	TRAP_ERROR(chdir(c->curr_dir) == -1, 550, return -1);

	/* handle 'absolute' paths */
	if (newd[0] == '/') {
		strcpy(temp, c->root_dir);
		if (newd[1] != '\0') {
			temp[strlen(temp) - 1] = 0;
			strcat(temp, newd);
		}
	} else {
		strcpy(temp, newd);
	}

	TRAP_ERROR(chdir(temp) == -1, 550, return -1);

	getcwd(chd, 254);
	if (chd[strlen(chd) - 1] != '/') {
		strcat(chd, "/");
	}

	if (strncmp(chd, c->root_dir, strlen(c->root_dir)) != 0) {
		numeric(c, 550, "No such file or directory.");
		return -1;
	}

	return 0;
}

void cmd_user(struct conn * const c)
{
#if WANT_SHADOW && HAVE_SHADOW_H
	struct spwd *s;
#endif
	struct passwd *p;
	
	strcpy(c->username, c->recv_buf); 

       	p = getpwnam(c->username);
#if WANT_SHADOW && HAVE_SHADOW_H
        getspent();     /* legal? */
        s = getspnam(c->username);
        endspent();
#endif

       	if (p == NULL) {
       		c->auth = 4;
       	} else {
       		c->uid = p->pw_uid;
		strcpy(c->curr_dir, p->pw_dir);
#if WANT_SHADOW && HAVE_SHADOW_H
		/*
		 * this isn't strictly correct, in case a user would have
		 * a shadow entry but still have the password in /etc/shadow
		 */
		if (s != NULL) {
			strcpy(c->pass_want, s->sp_pwdp);
		} else 
#endif
			strcpy(c->pass_want, p->pw_passwd);
		c->auth = 2;
       	}
       	if ((strcasecmp(c->username, "anonymous") == 0) || (strcasecmp(c->username, "ftp") == 0)) {
	  numeric(c, 331, "Login OK, send password (your e-mail).");
	  if (p == NULL) strcpy(c->curr_dir, anon_root_dir);//"/home/ftp");
		if (c->curr_dir[strlen(c->curr_dir) - 1] != '/') {
			strcat(c->curr_dir, "/");
		}
	        strcpy(c->root_dir, c->curr_dir);
		c->auth = 1;
	} else {
		numeric(c, 331, "Password required for %s.", c->username);
		strcpy(c->root_dir, "/");
	}
}

void cmd_pass(struct conn * const c)
{
	if (c->auth == 4) {
		numeric(c, 530, "Login incorrect.");
		c->auth = 0;
	} else if (c->auth == 1) {
		numeric(c, 230, "Guest login OK.");
		c->auth = 3;
	} else {
		/* the first two bytes of c->pass_want is the salt */
		char *pass_got = crypt(c->recv_buf, c->pass_want);
		if (strcmp(pass_got, c->pass_want) == 0) {
			numeric(c, 230, "User logged in.");
			c->auth = 3;
		} else {
			numeric(c, 530, "Login incorrect.");
			c->auth = 0;
		}
	}
}

void cmd_port(struct conn * const c)
{
	short int a0, a1, a2, a3, p0, p1;
	int i, sock;
	struct ftran *f;
	struct sockaddr_in sin;
    
	if ((c->transfer != NULL) && (c->transfer->state == 5)) {
		numeric(c, 500, "Sorry, only one transfer at a time.");
		return;
	}

	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	c->transfer = f = alloc_new_ftran(sock, c);

	i = sscanf(c->recv_buf, "%3hu,%3hu,%3hu,%3hu,%3hu,%3hu", &a0, &a1, &a2, &a3, &p0, &p1);
	if (i < 6) {
		numeric(c, 501, "Parse error.");
	} else {
		int tmp;

		numeric(c, 200, "PORT command OK.");

		/* bind to own address, port 20 (FTP data) */
		tmp = sizeof(sin);
		getsockname(c->sock, (struct sockaddr *)&sin, &tmp);
		sin.sin_port = FTP_PORT - 1;

		/* need root privilegies for a short while */
		seteuid(getuid());
		bind(sock, (struct sockaddr *)&sin, sizeof(sin));
		seteuid(c->uid);

		f->sin.sin_family = AF_INET;
		f->sin.sin_addr.s_addr = htonl(
			((unsigned char)(a0) << 24) +
			((unsigned char)(a1) << 16) +
			((unsigned char)(a2) <<  8) +
			((unsigned char)(a3)      ));
		f->sin.sin_port = htons(
			((unsigned char)(p0) << 8) +
			((unsigned char)(p1)     ));
		f->sock = sock;
		f->state = 3;

		i = 1;		
		ioctl(f->sock, FIONBIO, &i);
	}
}

void cmd_pasv(struct conn * const c)
{
	struct ftran *f;
	int sock, tmp;
	struct sockaddr_in addr;

	if (c->transfer != NULL) {
		numeric(c, 503, "Sorry, only one transfer at once.");
		return;
	}

	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	c->transfer = f = alloc_new_ftran(sock, c);

	/* setup socket */
	tmp = sizeof(addr);
	getsockname(c->sock, (struct sockaddr *)&addr, &tmp);

	addr.sin_port = 0;	/* let the system choose */
	bind(sock, (struct sockaddr *)&addr, sizeof(struct sockaddr));

	tmp = sizeof(addr);
	getsockname(sock, (struct sockaddr *)&addr, &tmp);

	numeric(c, 227, "Entering passive mode (%u,%u,%u,%u,%u,%u)",
		(htonl(addr.sin_addr.s_addr) & 0xff000000) >> 24,
		(htonl(addr.sin_addr.s_addr) & 0x00ff0000) >> 16,
		(htonl(addr.sin_addr.s_addr) & 0x0000ff00) >>  8,
		(htonl(addr.sin_addr.s_addr) & 0x000000ff),
		(htons(addr.sin_port) & 0xff00) >> 8,
		(htons(addr.sin_port) & 0x00ff));

	listen(f->sock, 1);
	f->state = 1;
}

void cmd_pwd(struct conn * const c)
{
	char temp[256], *cdir = NULL;

	strcpy(temp, c->curr_dir);
	if (strncmp(temp, c->root_dir, strlen(c->root_dir)) != 0) {
		numeric(c, 550, "curr_dir() is outside root_dir(), please contact site administrator.");
		return;
	}
	cdir = temp + strlen(c->root_dir) - 1;
	if (cdir[strlen(cdir) - 1] == '/' && strlen(cdir) > 1) {
		cdir[strlen(cdir) - 1] = 0;
	} else if (strlen(cdir) == 0) {
		strcpy(cdir, "/");
	}	

	numeric(c, 257, "\"%s\" is current working directory.", cdir);
}

void cmd_cwd(struct conn * const c)
{
	if (do_chdir(c, c->recv_buf) != -1) {
		int i;

		getcwd(c->curr_dir, 254);
		i = strlen(c->curr_dir);
		if (c->curr_dir[i - 1] != '/') {
			c->curr_dir[i++] = '/';
			c->curr_dir[i] = '\0';
		}
		numeric(c, 250, "CWD successful.");
	}
}

void cmd_cdup(struct conn * const c)
{
 	if (do_chdir(c, "..") != -1) {
               int i;

                getcwd(c->curr_dir, 254);
                i = strlen(c->curr_dir);
                if (c->curr_dir[i - 1] != '/') {
                        c->curr_dir[i++] = '/';
                        c->curr_dir[i] = '\0';
                }
	        numeric(c, 250, "CWD successful.");	/* RFC959 gives two different numbers (250 and 200) */
	}
}

void cmd_rest(struct conn * const c)
{
	/* add better error handler? */
	c->rest_pos = atoi(c->recv_buf);
	numeric(c, 350, "Setting resume at %u bytes.", c->rest_pos);
}

void cmd_retr(struct conn * const c)
{
	struct ftran *f = c->transfer;

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
		numeric(c, 425, "Can't build data connection; please use PASV or PORT.");
		return;
	}

	f->local_file = do_openfile(c, c->recv_buf, f->filename, O_RDONLY);
	f->dir_listing = 0;

	if (f->local_file == -1) {
		numeric(f->owner, 550, strerror(errno));
	} else if (f->local_file > -1) {
#if WANT_UPLOAD
		f->upload = 0;
#endif
		prepare_for_sending(f);
	}
}

#if WANT_UPLOAD
void cmd_stor(struct conn * const c)
{
	struct ftran *f = c->transfer;

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
		numeric(c, 425, "Can't build data connection; please use PASV or PORT.");
		return;
	}

	f->local_file = do_openfile(c, c->recv_buf, f->filename, O_WRONLY | O_CREAT | O_TRUNC);
	f->dir_listing = 0;

        if (f->local_file == -1) {
		numeric(f->owner, 550, strerror(errno));
	} else if (f->local_file > -1) {
		f->upload = 1;
		prepare_for_sending(f);
	}
}
#endif

void cmd_size(struct conn * const c)
{
	int loc_file;
	struct stat buf;

	loc_file = do_openfile(c, c->recv_buf, NULL, O_RDONLY);
	TRAP_ERROR(loc_file == -1, 550, return);
	if (loc_file == -2) return;

	fstat(loc_file, &buf);
 	numeric(c, 213, "%u", buf.st_size);
 	close(loc_file);
}

void cmd_mdtm(struct conn * const c)
{
        int loc_file;
	struct stat buf;
	struct tm *m;

        loc_file = do_openfile(c, c->recv_buf, NULL, O_RDONLY);
        TRAP_ERROR(loc_file == -1, 550, return);
        if (loc_file == -2) return;

        fstat(loc_file, &buf);
	close(loc_file);

	m = gmtime(&(buf.st_mtime));	/* at least wu-ftpd does it in GMT */
        numeric(c, 213, "%u%02u%02u%02u%02u%02u", m->tm_year + 1900,
		m->tm_mon + 1, m->tm_mday, m->tm_hour, m->tm_min, m->tm_sec);
}

/* could use external list */
void cmd_list(struct conn * const c)
{
	int i, n, year;
	char newd[512], temp[1024];
	struct ftran *f = c->transfer;
	struct stat buf;
	unsigned long total = 0;
	char months[][4] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
		 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	struct tm *t;
	time_t now;
	glob_t pglob;
	char *ptr;

	if (prepare_for_listing(c, &ptr) == -1) return;
	/*
	 * this is a bit tricky. we get to know total last, but the client
	 * wants it first. using pipes, we could just create another pipe
	 * and avoid scanning the directory twice, but pipes are out of the
	 * question now (see other comment). therefore, we hope that we are
	 * running 2.1.x or later, which (if i remember right) has some kind
	 * of stat() caching.
	 */
 	TRAP_ERROR(glob(ptr, 0, NULL, &pglob) != 0, 550, {free(ptr); return;}); /* nice!!! */
	free(ptr);

	for (n = 0; n < pglob.gl_pathc; n++) {
		TRAP_ERROR(lstat(pglob.gl_pathv[n], &buf) == -1, 550, return);
	        total += buf.st_blocks;
	}
	i = snprintf(temp, 1024, "total %lu\r\n", total);
	write(f->local_file, temp, i);

	time(&now);
	year = localtime(&now)->tm_year;
	for (n = 0; n < pglob.gl_pathc; n++) {
		struct passwd *p;
		struct group *g;
		char username[16], groupname[16];

		TRAP_ERROR(lstat(pglob.gl_pathv[n], &buf) == -1, 500, return);

		p = getpwuid(buf.st_uid);
		if (p != NULL) {
			strcpy(username, p->pw_name);
		} else {
			snprintf(username, 16, "%u", buf.st_uid);
		}

		g = getgrgid(buf.st_gid);
		if (g != NULL) {
			strcpy(groupname, g->gr_name);
		} else {
			snprintf(groupname, 16, "%u", buf.st_gid);
		}

		/* strftime() doesn't give us the functionality we need */
		t = localtime(&(buf.st_mtime));
		if (t->tm_year == year) {
			snprintf(newd, 512, "%3s %2u %02u:%02u", months[t->tm_mon], t->tm_mday, t->tm_hour, t->tm_min);
		} else {
			snprintf(newd, 512, "%3s %2u %5u", months[t->tm_mon], t->tm_mday, t->tm_year + 1900);
		}
		i = snprintf(temp, 1024, "%c%c%c%c%c%c%c%c%c%c %3u %-8s %-8s %8lu %12s %s\r\n",
			decode_mode(buf.st_mode),
			(buf.st_mode & S_IRUSR) ? 'r' : '-', 
			(buf.st_mode & S_IWUSR) ? 'w' : '-',
			(buf.st_mode & S_IXUSR) ? ((buf.st_mode & S_ISUID) ? 's' : 'x') : '-',
			(buf.st_mode & S_IRGRP) ? 'r' : '-',
			(buf.st_mode & S_IWGRP) ? 'w' : '-',
			(buf.st_mode & S_IXGRP) ? ((buf.st_mode & S_ISGID) ? 's' : 'x') : '-',
			(buf.st_mode & S_IROTH) ? 'r' : '-',
			(buf.st_mode & S_IWOTH) ? 'w' : '-',
			(buf.st_mode & S_IXOTH) ? 'x' : '-',
			buf.st_nlink, username, groupname, buf.st_size, 
			newd, pglob.gl_pathv[n]);
		write(f->local_file, temp, i);
	}
	lseek(f->local_file, 0, SEEK_SET);
	globfree(&pglob);

	prepare_for_sending(f);
}

void cmd_nlst(struct conn * const c)
{
	int i, j;
	struct ftran *f = c->transfer;
	char *ptr;
	glob_t pglob;

	j = prepare_for_listing(c, &ptr);
	if (j == -1) return;

	TRAP_ERROR(glob(ptr, GLOB_MARK, NULL, &pglob) != 0, 550, return);

	for (i = 0; i < pglob.gl_pathc; i++) {
		char *temp = pglob.gl_pathv[i];
		write(f->local_file, temp, strlen(temp));
	        write(f->local_file, "\r\n", 2);		                
	}
	lseek(f->local_file, 0, SEEK_SET);

	globfree(&pglob);
	prepare_for_sending(f);
}

void cmd_noop(struct conn * const c)
{
	numeric(c, 200, "NOOP command successful.");
}

void cmd_syst(struct conn * const c)
{
	numeric(c, 215, "UNIX Type: L%u", NBBY);
}

void cmd_type(struct conn * const c)
{
	numeric(c, 200, "TYPE ignored (always I)");
}

void cmd_mode(struct conn * const c)
{
	numeric(c, 200, "MODE ignored (always S)");
}

void cmd_stru(struct conn * const c)
{
	numeric(c, 200, "STRU ignored (always F)");
}

void cmd_quit(struct conn * const c)
{
	numeric(c, 221, "Have a nice day!");
	destroy_conn(c);
}

void parse_command(struct conn *c)
{
	int cmlen;
	const struct handler *h = handler_table;  	/* first entry */

	if (c == NULL) return;

	/* strip any leading CR/LFs */
	while (c->buf_len > 0 && (c->recv_buf[0] == 10 || c->recv_buf[0] == 13)) {
		remove_bytes(c, 1);
	}

	/* scan, searching for CR or LF */	
	cmlen = strcspn(c->recv_buf, "\r\n");
	if (cmlen >= c->buf_len) return;

        strncpy(c->last_cmd, c->recv_buf, cmlen);
        c->last_cmd[cmlen] = 0;

	do {
		if ((cmlen >= (strlen(h->cmd_name) + h->add_cmlen)) &&
		    (strncasecmp(c->recv_buf, h->cmd_name, strlen(h->cmd_name)) == 0)) {
			if (c->auth < h->min_auth) {
				numeric(c, 503, "Please login with USER and PASS.");
				while (c->recv_buf[0] != '\n') remove_bytes(c, 1);
			} else {
				char schar;

				if (h->do_setuid) {
					seteuid(c->uid);
				} else {
					seteuid(0);
				}

				remove_bytes(c, strlen(h->cmd_name));
				cmlen -= strlen(h->cmd_name);
				while (c->recv_buf[0] == ' ') {
					remove_bytes(c, 1);
					cmlen--;
				}

				schar = c->recv_buf[cmlen];
				c->recv_buf[cmlen] = 0;
				h->callback(c);
				c->recv_buf[cmlen] = schar; 

				if (c->free_me) {
					free(c);
				} else {
					if (h->do_setuid) seteuid(getuid());
					remove_bytes(c, cmlen);
				}
			}
			return;
		}
	} while ((++h)->callback != NULL);

	numeric(c, 500, "Sorry, no such command.");
	remove_bytes(c, cmlen); 
}

/*
 * Used for both upload and download, despite the name
 */ 
void prepare_for_sending(struct ftran *f)
{
#if WANT_UPLOAD
	if (f->upload == 0)
#endif
		f->size = lseek(f->local_file, 0, SEEK_END);
	lseek(f->local_file, 0, SEEK_SET);
	
	if (f->state == 1) {		/* PASV connection */
		f->state = 2;		/* waiting */
		FD_SET(f->sock, &master_fds);
	} else if (f->state == 3) {	/* PORT connection */
		f->state = 4;
		connect(f->sock, (struct sockaddr *)&f->sin, sizeof(f->sin));
		FD_SET(f->sock, &master_send_fds);
	}
	time(&(f->tran_start));
	f->size = 0;
}

char decode_mode(mode_t mode) {
	/*
	 * S_IFLINK seems to be broken? or perhaps I just have missed
	 * something (S_IFLINK is always set for all *files* on my 
	 * glibc 2.0.111 system)
	 */

	/* ordered for speed */
	if (mode & S_IFREG)  return '-';
	if (mode & S_IFDIR)  return 'd';
	if (mode & S_IFLNK)  return 'l';
	if (mode & S_IFBLK)  return 'b';
	if (mode & S_IFCHR)  return 'c';
	if (mode & S_IFSOCK) return 's';
	if (mode & S_IFIFO)  return 'f';

	/* hmmm, i'm a unix newbie -- what is the sticky bit? */
	return '-';
}

int do_openfile(struct conn * const c, char * const path, char * const filename, const int flags)
{
	char *ptr = NULL;
	struct stat buf;

	/* chdir to the right dir, then chop it off */
	chdir(c->curr_dir);

	ptr = strrchr(path, '/');
	if (ptr != NULL) {
	        char save_char = ptr[0];
	        ptr[0] = 0;

	        if (do_chdir(c, path) == -1) {
	                return -2;
	        }
	        ptr[0] = save_char;
	        ptr++;
	} else {
	        ptr = path;
	}

#if WANT_UPLOAD
	if ((flags & O_CREAT) == 0) {
#endif
	stat(ptr, &buf);
 	if (!S_ISREG(buf.st_mode)) {
		numeric(c, 550, "%s: Not a plain file.", ptr);
		return -2;
 	}
#if WANT_UPLOAD
	}
#endif

	if (filename != NULL) {
	        strcpy(filename, ptr);
	}
	return open(ptr, flags);
}

int prepare_for_listing(struct conn * const c, char ** const ptr)
{
	char *tfname;
	struct ftran *f = c->transfer;
	char *tmp, *temp = (char *)(malloc(512));

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
	        numeric(c, 425, "Please use PASV or PORT.");
		free(temp);
	        return -1;
	}

	if (c->recv_buf[0] != '-' && c->recv_buf[0] != '\0') {
		/*
		 * the tricky part here is determining if the last part
		 * of the path is a directory name or not. we leave this
		 * work to scandir() :-) But first we have to cut at the
		 * first " -" we see, to remove any parameters.
		 *
		 * kludgy and big code here; suggestions, anybody?
		 */
	        strcpy(temp, c->recv_buf);
	
		tmp = strrchr(temp, '/');
		if (tmp == NULL) {
			*ptr = temp;
			TRAP_ERROR(chdir(c->curr_dir) == -1, 550, return -1);
		} else {
			tmp[0] = 0;

	        	if (do_chdir(c, temp) == -1) {
				free(temp);
	        	        return -1;
	        	}

			*ptr = ++tmp;
		}
	} else {
	        TRAP_ERROR(chdir(c->curr_dir) == -1, 550, return -1);
		strcpy(temp, "*");
		*ptr = temp;
	}

	/*
	 * Using a pipe would be really, really great here. And I've done it.
	 * BUT... you can only put so much data into a pipe before it blocks
	 * :-( So I'm afraid we'll have to use a normal file here. Oh, well,
	 * we can probably cache it...
	 */
	tfname = tempnam(NULL, "ftp");
	TRAP_ERROR(tfname == NULL, 550, return -1);
	strcpy(f->filename, tfname);
	free(tfname);

	f->local_file = open(f->filename, O_RDWR | O_CREAT | O_TRUNC);
	TRAP_ERROR(f->local_file == -1, 550, return -1);
	f->dir_listing = 1;

	return 0;
}
