/*  ftpd.c: BetaFTPD main
    Copyright (C) 1999 Steinar H. Gunderson

    This program is is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2 if the
    License as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
 * Special note: this file has been overwritten by another (0-byte) file, been
 * through the dead, and restored (with the help of dd, grep, gpm, vi and less)
 * with a sucess rate of 99.9%. Show it a little respect -- don't add junk
 * to it. :-)
 */

#define _GNU_SOURCE

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include <netinet/ip.h>
#include <netinet/tcp.h>

#if HAVE_LINUX_SOCKET_H
#include <linux/socket.h>
#endif

#if HAVE_MMAP
#include <sys/mman.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <signal.h>
#include <malloc.h>
#include <ftpd.h>
#include <cmds.h>

extern struct mallinfo mallinfo();

struct conn *first_conn = NULL, *last_conn = NULL;
struct ftran *first_ftran = NULL, *last_ftran = NULL;

fd_set master_fds, master_send_fds;
unsigned int connected_clients = 0;

#if WANT_XFERLOG
FILE *xferlog = NULL;
#endif

int quit = 0;
static void handle_sigterm(int sig) {
  fprintf(stderr,"exiting due to signal\n");
  fprintf(stderr, "heap size: %d\n", mallinfo().usmblks);
  quit = 1;
}

struct conn *alloc_new_conn(int sock)
{
	struct conn *c = (struct conn *)(malloc(sizeof(struct conn)));

	if (c == NULL) return c;
	if (connected_clients >= FD_SETSIZE) {
		/* temp unavail */
		send(sock, "230 Server too busy, please try again later.\r\n", 46, 0);
		close(sock);
		return NULL;
	}

	FD_SET(sock, &master_fds);

	c->free_me = 0;

	if (last_conn != NULL) last_conn->next_conn = c;
	c->prev_conn = last_conn;
	c->next_conn = NULL;
	c->transfer = NULL;
	c->sock = sock;
	c->buf_len = c->auth = c->rest_pos = 0;
	strcpy(c->curr_dir, "/");
	strcpy(c->last_cmd, "");
	time(&(c->last_transfer));

	last_conn = c;
	return c;
}

struct ftran *alloc_new_ftran(int sock, struct conn *c)
{
	struct ftran *f = (struct ftran *)(malloc(sizeof(struct ftran)));

	if (f == NULL) return f;
	if (last_ftran != NULL) last_ftran->next_ftran = f;
#if HAVE_MMAP
	f->file_data = NULL;
#endif
	f->prev_ftran = last_ftran;
	f->next_ftran = NULL;
	f->owner = c;
	f->sock = sock;
	f->state = 0;

	last_ftran = f;
	return f;
}

void destroy_conn(struct conn *c)
{
	if (c == NULL) return;
	close(c->sock);

	/* remove from the set */
	FD_CLR(c->sock, &master_fds);

	/* update linked list */
	if (c->prev_conn != NULL) c->prev_conn->next_conn = c->next_conn;
	if (c->next_conn != NULL) c->next_conn->prev_conn = c->prev_conn;

	if (last_conn == c) last_conn = c->prev_conn;
	destroy_ftran(c->transfer);
	c->free_me = 1;
}

void destroy_ftran(struct ftran *f)
{
	if (f == NULL) return;
	close(f->sock);

#if HAVE_MMAP
	if (f->file_data) munmap(f->file_data, f->size);
#endif
	close(f->local_file);
	if (f->dir_listing) unlink(f->filename);

	f->owner->transfer = NULL;

	/* remove from the set(s) */
	FD_CLR(f->sock, &master_fds);
	FD_CLR(f->sock, &master_send_fds);

	/* update linked list */
	if (f->prev_ftran != NULL) f->prev_ftran->next_ftran = f->next_ftran;
	if (f->next_ftran != NULL) f->next_ftran->prev_ftran = f->prev_ftran;

	if (last_ftran == f) last_ftran = f->prev_ftran;
	free(f);
}

int process_all_clients(fd_set *active_clients, int num_ac)
{
	struct conn *c = NULL, *next = first_conn->next_conn;
	int bytes_avail;
	int checked_through = 0;

	/* run through the linked list */
	while (next != NULL && checked_through < num_ac) {
		c = next;
		next = c->next_conn;

		if (!FD_ISSET(c->sock, active_clients)) {
			continue;
		}

		checked_through++;

		/*
		 * see if there's an error on this socket
		 */
		if (recv(c->sock, NULL, 0, 0) < 0) {
			destroy_conn(c);
			free(c);
			continue;
		}

		/* see if there is data waiting on socket */
		ioctl(c->sock, FIONREAD, &bytes_avail);

		if (bytes_avail > 0) {
			/* overrun = disconnect */
			if (c->buf_len + bytes_avail > 255) {
				numeric(c, 503, "Buffer overrun; disconnecting.");
				destroy_conn(c);
				free(c);
				continue;
			}
			recv(c->sock, c->recv_buf + c->buf_len, bytes_avail, 0);
			c->buf_len += bytes_avail;

			/* let the handler process the data */
			parse_command(c);
		} else {
			/*
			 * select() has already told us there's something about
			 * this socket, so if there's no data (and no error), the socket
			 * must be closed
			 */
			destroy_conn(c);
			free(c);
		}
	}
	return checked_through;
}

int process_all_sendfiles(fd_set *active_clients, int num_ac)
{
	struct ftran *f = NULL, *next = first_ftran->next_ftran;
	int checked_through = 0;
 
	while (next != NULL && checked_through < num_ac) {
		struct sockaddr tempaddr;
		int tempaddr_len;

	        f = next;
	        next = f->next_ftran;

		/* state = 2: incoming PASV, state >3: send file */
		if (!FD_ISSET(f->sock, active_clients) || (f->state < 2)) {
			continue;
		}

		checked_through++;
		FD_CLR(f->sock, active_clients);

		if (f->state == 2) {		/* incoming PASV */
                        tempaddr_len = sizeof(struct sockaddr_in);
			int tempsock = accept(f->sock, (struct sockaddr *)&tempaddr, &tempaddr_len);
			close(f->sock);
			FD_CLR(f->sock, &master_fds);

			if (tempsock == -1) {
				destroy_ftran(f);
				continue;
			}

			f->sock = tempsock;
			init_file_transfer(f);
#if WANT_UPLOAD
                        if (f->upload) continue;
#endif
		}
		if (f->state < 5) {
			init_file_transfer(f);
#if WANT_UPLOAD
                        if (f->upload) continue;
#endif
		}

		/* for download, we send the first packets right away */
#if WANT_UPLOAD
		if (f->upload) {
#if HAVE_MMAP
#warning no mmap-enhanced uploading -- falling back to standard
#endif

			char buf[65536];
			int avail, size;

			ioctl(f->sock, FIONREAD, &avail);
			if (avail < 0) avail = 0;
			if (avail > 65536) avail = 65536; /* panic */

			errno = 0;
			size = recv(f->sock, buf, avail, 0);
                        f->pos += size;

			if (size > 0 && (write(f->local_file, buf, size) == size)) {
				continue;
			} else if (size == -1) {
				/* don't write xferlog... or? */
				numeric(f->owner, 426, "Error: %s", strerror(errno));
			}
		} else
#endif
		{
#if !HAVE_MMAP
			char buf[1460];
#endif
			int size;

               		size = f->size - f->pos;
                	if (size > 1460) size = 1460;
                	if (size < 0) size = 0;
	
#if HAVE_MMAP
                	send(f->sock, f->file_data + f->pos, size, 0);
                	f->pos += size;
			if (size == 1460) continue;
#else
                	size = read(f->local_file, buf, 1460);
                	send(f->sock, buf, size, 0);
                	if (size == 1460) continue;
#endif
		}

               	numeric(f->owner, 226, "Transfer complete.");
		time(&(f->owner->last_transfer));

#if WANT_XFERLOG
               	if (!f->dir_listing) {
                       	char temp[256];
                       	time_t now = time(NULL);
                       	struct tm *t = localtime(&now);

                       	strftime(temp, 256, "%a %b %d %H:%M:%S %Y", t);
#if WANT_UPLOAD
			fprintf(xferlog, "%s %u %s %lu %s b _ %c a %s ftp 0 * \n",
#else
                       	fprintf(xferlog, "%s %u %s %lu %s b _ o a %s ftp 0 *\n",
#endif
	                       temp, (int)(difftime(now, f->tran_start)),
                               inet_ntoa(f->sin.sin_addr), f->size,
                       	       f->filename,
#if WANT_UPLOAD
			       (f->upload) ? 'i' : 'o',
#endif
			       f->owner->username);
                       	fflush(xferlog);
               	}
#endif
               	destroy_ftran(f);
#if WANT_FULLSCREEN
	        update_display(first_conn);
#endif
	}

	return checked_through;
}

int main(int argc, char **argv)
{
	int ftp_port = FTP_PORT;
	int server_sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	struct sockaddr_in addr;
	int err, tempaddr_len;
	fd_set fds, fds_send;

	signal(SIGPIPE, SIG_IGN);
        signal(SIGTERM, handle_sigterm);
        signal(SIGINT, handle_sigterm);

	printf("BetaFTPD version %s, Copyright (C) 1999 Steinar H. Gunderson\n", VERSION);
	printf("BetaFTPD server comes with ABSOLUTELY NO WARRANTY; for details see the file\n");
	printf("COPYING. This is free software, and you are welcome to redistribute it\n");
	printf("under certain conditions; again see the file COPYING for details.\n");

	if (argc > 1)
		ftp_port = atoi(argv[1]);
	if (argc > 2) {
	  strcpy(anon_root_dir, argv[2]);
	}
	else {
	  strcpy(anon_root_dir, "./");
	}
	FD_ZERO(&master_fds);
	FD_ZERO(&master_send_fds);

	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(ftp_port);

	do {
		err = bind(server_sock, (struct sockaddr *)&addr, sizeof(struct
sockaddr));

		if (err == -1) {
			perror("bind()");

			if (errno == EADDRINUSE) {
				printf("Sleeping 5 seconds...\n");
				sleep(5);
			} else {
				exit(1);
			}
		}
	} while (err == -1);

	err = listen(server_sock, 20);
	FD_SET(server_sock, &master_fds);

	/* init dummy first connection */
	first_conn = alloc_new_conn(0);
	first_ftran = alloc_new_ftran(0, NULL);

#if WANT_XFERLOG
	/* open xferlog */
	xferlog = fopen("/usr/adm/xferlog", "a");
#endif

#if WANT_FORK
	switch (fork()) {
	case -1:
		perror("fork()");
		printf("fork() failed, exiting\n");
		exit(0);
	case 0:
		break;
	default:
		printf("BetaFTPD forked into the background\n");
		exit(0);
	}
#else
	printf("BetaFTPD active\n");
#endif

	for ( ;; ) {
		struct sockaddr_in tempaddr;
		int i;
		struct timeval timeout;

#if WANT_FULLSCREEN
                update_display(first_conn);
#endif

		/* reset fds (gets changed by select()) */
		fds = master_fds;
		fds_send = master_send_fds;

		/*
		 * wait up to 60 secs for any activity 
		 */
		timeout.tv_sec = 60;
		timeout.tv_usec = 0;

		fflush(stdout);
		i = select(FD_SETSIZE, &fds, &fds_send, NULL, &timeout);
		if (quit) return 0;

		/* why does this happen??? */
		if (FD_ISSET(0, &fds)) {
			i--;
			FD_CLR(i, &master_fds);
		}

		if (i == -1) {
			perror("select()");
			sleep(1);
		}

		/* remove any timed out sockets */
		time_out_sockets();

		if (i <= 0) continue;

		/* sends are given highest `priority' */
		i -= process_all_sendfiles(&fds_send, i);

		/* incoming PASV connections and uploads */
		i -= process_all_sendfiles(&fds, i);

		/*
		 * check the incoming PASV connections first, so
		 * process_all_clients() won't be confused.
		 */ 
		process_all_clients(&fds, i);

		if (FD_ISSET(server_sock, &fds)) {
			/* new client connecting */
                        tempaddr_len = sizeof(struct sockaddr_in);
			int tempsock = accept(server_sock, (struct sockaddr *)&tempaddr, &tempaddr_len);
			i--;

			if (tempsock != -1) {   /* paranoia */
				struct conn *c = alloc_new_conn(tempsock);
				if (c != NULL) {
					numeric(c, 220, "BetaFTPD " VERSION " ready.");
				}
			}
		}
	}

}

void time_out_sockets()
{
        struct conn *c = NULL, *next = first_conn->next_conn;
	time_t now = time(NULL);

        /* run through the linked list */
        while (next != NULL) {
                c = next;
                next = c->next_conn;

		if ((c->transfer == NULL || c->transfer->state != 5) &&
		    (difftime(c->last_transfer, now) > 900)) {
			/* RFC violation? */
			numeric(c, 421, "Timeout (15 minutes): Closing control connection.");
			destroy_conn(c);
			free(c);
		}
	}
}

void remove_bytes(struct conn *c, const int i)
{
	if (c->buf_len <= i) {
		c->buf_len = 0;
	} else {
		c->buf_len -= i;
		memmove(c->recv_buf, c->recv_buf + i, c->buf_len);
	}
}

void numeric(struct conn * const c, const int numeric, const char * const format, ...)
{
	char buf[256], fmt[256];
	va_list args;
	int i, err;

	snprintf(fmt, 256, "%03u %s\r\n", numeric, format);

	va_start(args, format);
	i = vsnprintf(buf, 256, fmt, args);
	va_end(args);

	/* printf(buf); */
	err = send(c->sock, buf, i, 0);
	if (err == -1 && errno == EPIPE) {
		destroy_conn(c);
		free(c);
	}
}

void init_file_transfer(struct ftran * const f)
{
	struct linger ling;
	int mode = IPTOS_THROUGHPUT, one = 1;
 
	/* we want max throughput */
	setsockopt(f->sock, SOL_IP, IP_TOS, (char *)&mode, sizeof(mode));
	setsockopt(f->sock, SOL_TCP, TCP_NODELAY, (char *)&one, sizeof(mode));

	f->state = 5;

#if WANT_UPLOAD
	if (f->upload) {
		FD_SET(f->sock, &master_fds);
	} else
#endif
		FD_SET(f->sock, &master_send_fds);

	ling.l_onoff = 1;
	ling.l_linger = 10;
	setsockopt(f->sock, SOL_SOCKET, SO_LINGER, &ling, sizeof(ling));

#if WANT_UPLOAD
	/*
	 * if we let an upload socket stay in master_send_fds, we would
	 * get data that would fool us into closing the socket... (sigh)
	 */
	if (f->upload) {
		FD_CLR(f->sock, &master_send_fds);
		FD_SET(f->sock, &master_fds);
	}
#endif
	
	/* notify user we can start */
	f->size = lseek(f->local_file, 0, SEEK_END);
	time(&(f->owner->last_transfer));

	if (f->dir_listing) {
		/* include size? */
		numeric(f->owner, 150, "Opening ASCII mode data connection for directory listing.");
	} else {
#if WANT_UPLOAD
		if (f->upload) {
			numeric(f->owner, 150, "Opening BINARY mode data connection for '%s'", f->filename);
		} else
#endif
		numeric(f->owner, 150, "Opening BINARY mode data connection for '%s' (%u bytes)", f->filename, f->size);
	}

#if HAVE_MMAP
#if WANT_UPLOAD
	if (f->upload == 0) {
		f->file_data = NULL;
	} else
#endif
        	f->file_data = mmap(NULL, f->size, PROT_READ, MAP_SHARED, f->local_file, 0);
        f->pos = f->owner->rest_pos;
#else
	lseek(f->local_file, f->owner->rest_pos, SEEK_SET);
#endif
}
