/* http_post - do a POST to an HTTP URL
**
** Copyright  1999 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#ifdef USE_SSL
#include <ssl.h>
#endif

/* Forwards. */
static void usage();
static void postURL( char* url, char* referer, char* user_agent, char* auth_token, char* cookie, char** args, int argc );
static void postURLbyParts( int protocol, char* host, int port, char* file, char* referer, char* user_agent, char* auth_token, char* cookie, char** args, int argc );
static void show_error( char* cause );
static void sigcatch( int sig );
static void strencode( char* to, int tosize, char* from );
static int b64_encode( unsigned char* ptr, int len, char* space, int size );


/* Globals. */
static char* argv0;
static int verbose;
static int timeout;
static char* url;

/* Protocol symbols. */
#define PROTO_HTTP 0
#ifdef USE_SSL
#define PROTO_HTTPS 1
#endif

/* Header FSM states. */
#define HDST_BOL 0
#define HDST_TEXT 1
#define HDST_LF 2
#define HDST_CR 3
#define HDST_CRLF 4
#define HDST_CRLFCR 5


int
main( int argc, char** argv )
    {
    int argn;
    char* referer;
    char* user_agent;
    char* auth_token;
    char* cookie;

    argv0 = argv[0];
    argn = 1;
    verbose = 0;
    timeout = 60;
    referer = (char*) 0;
    user_agent = "http_post";
    auth_token = (char*) 0;
    cookie = (char*) 0;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( strcmp( argv[argn], "-v" ) == 0 )
	    verbose = 1;
	else if ( strcmp( argv[argn], "-t" ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    timeout = atoi( argv[argn] );
	    }
	else if ( strcmp( argv[argn], "-r" ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    referer = argv[argn];
	    }
	else if ( strcmp( argv[argn], "-u" ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    user_agent = argv[argn];
	    }
	else if ( strcmp( argv[argn], "-a" ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    auth_token = argv[argn];
	    }
	else if ( strcmp( argv[argn], "-c" ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    cookie = argv[argn];
	    }
	else
	    usage();
	++argn;
	}
    if ( argn >= argc )
	usage;
    url = argv[argn];
    ++argn;

    (void) signal( SIGALRM, sigcatch );
    postURL( url, referer, user_agent, auth_token, cookie, &(argv[argn]), argc - argn );

    exit( 0 );
    }


static void
usage()
    {
    (void) fprintf( stderr, "usage:  %s [-t timeout] [-r referer] [-u user-agent] [-a username:password] url\n", argv0 );
    exit( 1 );
    }


/* url must be of the form http://host-name[:port]/file-name */
static void
postURL( char* url, char* referer, char* user_agent, char* auth_token, char* cookie, char** args, int argc )
    {
    char* s;
    int protocol;
    char host[2000];
    int host_len;
    int port;
    char* file = 0;
    char* http = "http://";
    int http_len = strlen( http );
    char* https = "https://";
    int https_len = strlen( https );
    int proto_len;

    if ( url == (char*) 0 )
        {
	(void) fprintf( stderr, "%s: null URL\n", argv0 );
        exit( 1 );
        }
    if ( strncmp( http, url, http_len ) == 0 )
	{
	proto_len = http_len;
	protocol = PROTO_HTTP;
	}
#ifdef USE_SSL
    else if ( strncmp( https, url, https_len ) == 0 )
	{
	proto_len = https_len;
	protocol = PROTO_HTTPS;
	}
#endif
    else
        {
	(void) fprintf( stderr, "%s: non-http URL\n", argv0 );
        exit( 1 );
        }

    /* Get the host name. */
    for ( s = url + proto_len; *s != '\0' && *s != ':' && *s != '/'; ++s )
	;
    host_len = s - url;
    host_len -= proto_len;
    strncpy( host, url + proto_len, host_len );
    host[host_len] = '\0';

    /* Get port number. */
    if ( *s == ':' )
	{
	port = atoi( ++s );
	while ( *s != '\0' && *s != '/' )
	    ++s;
	}
    else
#ifdef USE_SSL
	if ( protocol == PROTO_HTTPS )
	    port = 443;
	else
	    port = 80;
#else
	port = 80;
#endif

    /* Get the file name. */
    if ( *s == '\0' ) 
	file = "/";
    else
	file = s;

    postURLbyParts( protocol, host, port, file, referer, user_agent, auth_token, cookie, args, argc );
    }


static void
postURLbyParts( int protocol, char* host, int port, char* file, char* referer, char* user_agent, char* auth_token, char* cookie, char** args, int argc )
    {
    struct hostent *he;
    struct servent *se;
    struct protoent *pe;
    struct sockaddr_in sin;
    int sockfd;
#ifdef USE_SSL
    SSL_CTX* ssl_ctx;
    SSL* ssl;
#endif
    char head_buf[10000];
    char data_buf[50000];
    char enc_buf[5000];
    int head_bytes, data_bytes, i, header_state;
    char* eq;

    /* Connect to the server. */
    (void) alarm( timeout );
    he = gethostbyname( host );
    if ( he == (struct hostent*) 0 )
	{
	(void) fprintf( stderr, "%s: unknown host - %s\n", argv0, host );
	exit( 1 );
	}
    (void) alarm( timeout );
    se = getservbyname( "telnet", "tcp" );
    if ( se == (struct servent*) 0 )
	{
	(void) fprintf( stderr, "%s: unknown service\n", argv0 );
	exit( 1 );
	}
    (void) alarm( timeout );
    pe = getprotobyname( se->s_proto );
    if ( pe == (struct protoent*) 0 )
	{
	(void) fprintf( stderr, "%s: unknown protocol\n", argv0 );
	exit( 1 );
	}
    bzero( (caddr_t) &sin, sizeof(sin) );
    sin.sin_family = he->h_addrtype;
    (void) alarm( timeout );
    sockfd = socket( he->h_addrtype, SOCK_STREAM, pe->p_proto );
    if ( sockfd < 0 )
	show_error( "socket" );
    (void) alarm( timeout );
    if ( bind( sockfd, (struct sockaddr*) &sin, sizeof(sin) ) < 0 )
	show_error( "bind" );
    bcopy( he->h_addr, &sin.sin_addr, he->h_length );
    sin.sin_port = htons( port );
    (void) alarm( timeout );
    if ( connect( sockfd, (struct sockaddr*) &sin, sizeof(sin) ) < 0 )
	show_error( "connect" );
#ifdef USE_SSL
    if ( protocol == PROTO_HTTPS )
	{
	/* Make SSL connection. */
	int r;
	SSL_load_error_strings();
	SSLeay_add_ssl_algorithms();
	ssl_ctx = SSL_CTX_new( SSLv23_client_method() );
	ssl = SSL_new( ssl_ctx );
	SSL_set_fd( ssl, sockfd );
	r = SSL_connect( ssl );
	if ( r <= 0 )
	    {
	    (void) fprintf(
		stderr, "%s: %s - SSL connection failed - %d\n",
		argv0, url, r );
	    ERR_print_errors_fp( stderr );
	    exit( 1 );
	    }
	}
#endif

    /* Encode the POST data. */
    data_bytes = 0;
    for ( i = 0; i < argc ; ++i )
	{
	if ( data_bytes > 0 )
	    data_bytes += snprintf( &data_buf[data_bytes], sizeof(data_buf) - data_bytes, "&" );
	eq = strchr( args[i], '=' );
	if ( eq == (char*) 0 )
	    {
	    strencode( enc_buf, sizeof(enc_buf), args[i] );
	    data_bytes += snprintf( &data_buf[data_bytes], sizeof(data_buf) - data_bytes, "%s", enc_buf );
	    }
	else
	    {
	    *eq++ = '\0';
	    strencode( enc_buf, sizeof(enc_buf), args[i] );
	    data_bytes += snprintf( &data_buf[data_bytes], sizeof(data_buf) - data_bytes, "%s", enc_buf );
	    data_bytes += snprintf( &data_buf[data_bytes], sizeof(data_buf) - data_bytes, "=" );
	    strencode( enc_buf, sizeof(enc_buf), eq );
	    data_bytes += snprintf( &data_buf[data_bytes], sizeof(data_buf) - data_bytes, "%s", enc_buf );
	    }
	}

    /* Build request buffer, starting with the POST. */
    (void) alarm( timeout );
    head_bytes = snprintf( head_buf, sizeof(head_buf), "POST %s HTTP/1.0\r\n", file );
    /* HTTP/1.1 host header - some servers want it even in HTTP/1.0. */
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Host: %s\r\n", host );
    /* Content-length. */
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Content-Length: %d\r\n", data_bytes );
    if ( referer != (char*) 0 )
	/* Referer. */
	head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Referer: %s\r\n", referer );
    /* User-agent. */
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "User-Agent: %s\r\n", user_agent );
    /* Fixed headers. */
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Content-type: application/x-www-form-urlencoded\r\n" );
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Accept: */*\r\n" );
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Accept-Encoding: gzip, compress\r\n" );
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Accept-Language: en\r\n" );
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Accept-Charset: iso-8859-1,*,utf-8\r\n" );
    if ( auth_token != (char*) 0 )
	{
	/* Basic Auth info. */
	char token_buf[500];
	token_buf[b64_encode( auth_token, strlen( auth_token ), token_buf, sizeof(token_buf) )] = '\0';
	head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Authorization: Basic %s\r\n", token_buf );
	}
    /* Cookie. */
    if ( cookie != (char*) 0 )
	head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "Cookie: %s\r\n", cookie );
    /* Blank line. */
    head_bytes += snprintf( &head_buf[head_bytes], sizeof(head_buf) - head_bytes, "\r\n" );
    /* Now actually send it. */
#ifdef USE_SSL
    if ( protocol == PROTO_HTTPS )
	(void) SSL_write( ssl, head_buf, head_bytes );
    else
	(void) write( sockfd, head_buf, head_bytes );
#else
    (void) write( sockfd, head_buf, head_bytes );
#endif
    /* And send the POST data too. */
#ifdef USE_SSL
    if ( protocol == PROTO_HTTPS )
	(void) SSL_write( ssl, data_buf, data_bytes );
    else
	(void) write( sockfd, data_buf, data_bytes );
#else
    (void) write( sockfd, data_buf, data_bytes );
#endif

    /* Get lines until a blank one. */
    (void) alarm( timeout );
    header_state = HDST_BOL;
    for (;;)
	{
#ifdef USE_SSL
	if ( protocol == PROTO_HTTPS )
	    data_bytes = SSL_read( ssl, data_buf, sizeof(data_buf) );
	else
	    data_bytes = read( sockfd, data_buf, sizeof(data_buf) );
#else
	data_bytes = read( sockfd, data_buf, sizeof(data_buf) );
#endif
	if ( data_bytes <= 0 )
	    break;
	for ( i = 0; i < data_bytes; ++i )
	    {
	    if ( verbose )
		(void) write( 1, &data_buf[i], 1 );
	    switch ( header_state )
		{
		case HDST_BOL:
		switch ( data_buf[i] )
		    {
		    case '\n': header_state = HDST_LF; break;
		    case '\r': header_state = HDST_CR; break;
		    default: header_state = HDST_TEXT; break;
		    }
		break;
		case HDST_TEXT:
		switch ( data_buf[i] )
		    {
		    case '\n': header_state = HDST_LF; break;
		    case '\r': header_state = HDST_CR; break;
		    }
		break;

		case HDST_LF:
		switch ( data_buf[i] )
		    {
		    case '\n': goto end_of_headers;
		    case '\r': header_state = HDST_CR; break;
		    default: header_state = HDST_TEXT; break;
		    }
		break;

		case HDST_CR:
		switch ( data_buf[i] )
		    {
		    case '\n': header_state = HDST_CRLF; break;
		    case '\r': goto end_of_headers;
		    default: header_state = HDST_TEXT; break;
		    }
		break;

		case HDST_CRLF:
		switch ( data_buf[i] )
		    {
		    case '\n': goto end_of_headers;
		    case '\r': header_state = HDST_CRLFCR; break;
		    default: header_state = HDST_TEXT; break;
		    }
		break;

		case HDST_CRLFCR:
		switch ( data_buf[i] )
		    {
		    case '\n': case '\r': goto end_of_headers;
		    default: header_state = HDST_TEXT; break;
		    }
		break;
		}
	    }
	}
    end_of_headers:
    /* Dump out the rest of the headers buffer. */
    ++i;
    (void) write( 1, &data_buf[i], data_bytes - i );

    /* Copy the data. */
    for (;;)
        {
	(void) alarm( timeout );
#ifdef USE_SSL
	if ( protocol == PROTO_HTTPS )
	    data_bytes = SSL_read( ssl, data_buf, sizeof(data_buf) );
	else
	    data_bytes = read( sockfd, data_buf, sizeof(data_buf) );
#else
	data_bytes = read( sockfd, data_buf, sizeof(data_buf) );
#endif
	if ( data_bytes == 0 )
	    break;
	if ( data_bytes < 0 )
	    show_error( "read" );
	(void) write( 1, data_buf, data_bytes );
        }
#ifdef USE_SSL
    if ( protocol == PROTO_HTTPS )
	{
	SSL_free( ssl );
	SSL_CTX_free( ssl_ctx );
	}
#endif  
    (void) close( sockfd );
    }


static void
show_error( char* cause )
    {
    char buf[5000];
    (void) sprintf( buf, "%s: %s - %s", argv0, url, cause );
    perror( buf );
    exit( 1 );
    }


static void  
sigcatch( int sig )
    {       
    (void) fprintf( stderr, "%s: %s - timed out\n", argv0, url );
    exit( 1 );
    }


static void
strencode( char* to, int tosize, char* from )
    {
    int tolen;

    for ( tolen = 0; *from != '\0' && tolen + 4 < tosize; ++from )
	{
	if ( isalnum(*from) || strchr( "/_.", *from ) != (char*) 0 )
	    {
	    *to = *from;
	    ++to;
	    ++tolen;
	    }
	else
	    {
	    (void) sprintf( to, "%c%02x", '%', *from );
	    to += 3;
	    tolen += 3;
	    }
	}
    *to = '\0';
    }


/* Base-64 encoding.  This encodes binary data as printable ASCII characters.
** Three 8-bit binary bytes are turned into four 6-bit values, like so:
**
**   [11111111]  [22222222]  [33333333]
**
**   [111111] [112222] [222233] [333333]
**
** Then the 6-bit values are represented using the characters "A-Za-z0-9+/".
*/

static char b64_encode_table[64] = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',  /* 0-7 */
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',  /* 8-15 */
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',  /* 16-23 */
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',  /* 24-31 */
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',  /* 32-39 */
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',  /* 40-47 */
    'w', 'x', 'y', 'z', '0', '1', '2', '3',  /* 48-55 */
    '4', '5', '6', '7', '8', '9', '+', '/'   /* 56-63 */
    };

static int b64_decode_table[256] = {
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 00-0F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 10-1F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,  /* 20-2F */
    52,53,54,55,56,57,58,59,60,61,-1,-1,-1,-1,-1,-1,  /* 30-3F */
    -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 40-4F */
    15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,  /* 50-5F */
    -1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 60-6F */
    41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,  /* 70-7F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 80-8F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 90-9F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* A0-AF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* B0-BF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* C0-CF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* D0-DF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* E0-EF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1   /* F0-FF */
    };

/* Do base-64 encoding on a hunk of bytes.   Return the actual number of
** bytes generated.  Base-64 encoding takes up 4/3 the space of the original,
** plus a bit for end-padding.  3/2+5 gives a safe margin.
*/
static int
b64_encode( unsigned char* ptr, int len, char* space, int size )
    {
    int ptr_idx, space_idx, phase;
    char c;

    space_idx = 0;
    phase = 0;
    for ( ptr_idx = 0; ptr_idx < len; ++ptr_idx )
	{
	switch ( phase )
	    {
	    case 0:
	    c = b64_encode_table[ptr[ptr_idx] >> 2];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    c = b64_encode_table[( ptr[ptr_idx] & 0x3 ) << 4];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    ++phase;
	    break;
	    case 1:
	    space[space_idx - 1] =
	      b64_encode_table[
		b64_decode_table[space[space_idx - 1]] |
		( ptr[ptr_idx] >> 4 ) ];
	    c = b64_encode_table[( ptr[ptr_idx] & 0xf ) << 2];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    ++phase;
	    break;
	    case 2:
	    space[space_idx - 1] =
	      b64_encode_table[
		b64_decode_table[space[space_idx - 1]] |
		( ptr[ptr_idx] >> 6 ) ];
	    c = b64_encode_table[ptr[ptr_idx] & 0x3f];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    phase = 0;
	    break;
	    }
	}
    /* Pad with ='s. */
    while ( phase++ < 3 )
	if ( space_idx < size )
	    space[space_idx++] = '=';
    return space_idx;
    }
