// Solves the Wave Eq, Utt - c Uxx = 0, using P++
//
#include <A++.h>
#include <time.h>
#include <iostream>
#include <math.h>
#include "InterComm.h"
#include "IC_util.h"
#include "IC_support.h"

const char localconf[]="waveeq.conf";

int main(int argc, char **argv) {
  int iNumProcs;
  Optimization_Manager::Initialize_Virtual_Machine("",iNumProcs,argc,argv);

  std::cout << "Running on " << iNumProcs << " processors" << endl;
  
  // Length of physical dimensions and time
  double dLenX, dLenY;
  // Steps sizes
  double dDX, dDY, dDT,dC;
  // Timing Variables

  ConfFile conf;
  int ret=conf.readConfFile(localconf);
  if (ret!=CF_OK) {
    std::cerr << "Configuration file '" << localconf << "' missing!\n";
    return -1;
  }

  int iNumX, iNumY, iNSteps;
  const char *ptr=conf.getValue4("SimulationParameters","XGridSize");
  if (ptr)
    iNumX=atoi(ptr);
  else
    std::cerr << "Missing value 'XGridSize'...\n";
  ptr=conf.getValue4("SimulationParameters","YGridSize");
  if (ptr)
    iNumY=atoi(ptr);
  else
    std::cerr << "Missing value 'YGridSize'...\n";
  ptr=conf.getValue4("SimulationParameters","NTimeSteps");
  if (ptr)
    iNSteps=atoi(ptr);
  else
    std::cerr << "Missing value 'NTimeSteps'...\n";
  ptr=conf.getValue4("SimulationParameters","TimeStep");
  if (ptr)
    dDT = atof(ptr);
  else
    dDT = 1.0e-2;

  int i,j;

  // Set Parameters
  dLenX = 1.0; dLenY = 1.0;
  dDX = dLenX/(iNumX/2);
//  dDX = dLenX/iNumX;
  dDY = dLenY/iNumY;
  dC = 1.0;

  // Arrays to hold solution at steps n-1,n,and n+1
  // Size is NumX+2 to allow for BC on either side which we are peroidic
  doubleArray daUnm1(iNumX+2,iNumY+2),daUn(iNumX+2,iNumY+2);
  doubleArray daUnp1(iNumX+2,iNumY+2);

  doubleArray daX(iNumX+2), daY(iNumY+2);
  for(i=1;i<iNumX+1;i++) {
    daX(i) = (i-1)*dDX;
//    printf("i - x: %d %.3g\n",i,daX(i));
  }
#ifdef DISPLAY
  daX.display("X");
#endif
  for(j=1;j<iNumY+1;j++)
    daY(j) = (j-1)*dDY;

  Index I(1,iNumX), J(1,iNumY); // Indices for computational domain
  Index If,Jf; // Indices for entire domain

  // Impose an initial wave in the form sin(wt+x/L) with constraint that 
  // w = c/L
  double dW;
  dW = dC/dLenX;
  double dTime;
  dTime = -1.0*dDT;
  double dPi;
  dPi = M_PI;
  for(j=1;j<iNumY+1;j++) {
    daUnm1(I,j) = sin(dW*dTime + (daX(I)*2*dPi)/dLenX); 
#ifdef DISPLAY
    daUnm1.display("Unm1");
#endif
    daUn(If,j)  = sin(dW*0 + (daX(If)*2*dPi)/dLenX); 
#ifdef DISPLAY
    daUn.display("Un");
#endif
  }

  // Apply BC
  daUnm1(0,J) = daUnm1(iNumX,J);
  daUnm1(iNumX+1,J) = daUnm1(1,J);
  daUnm1(I,0) = daUnm1(I,iNumY);
  daUnm1(I,iNumY+1) = daUnm1(I,1);
  daUn(0,J) = daUn(iNumX,J);
  daUn(iNumX+1,J) = daUn(1,J);
  daUn(I,0) = daUn(I,iNumY);
  daUn(I,iNumY+1) = daUn(I,1);

#ifdef DISPLAY
  daUnm1.display("Unm1");
  daUn.display("Un");
#endif

  std::cout << "Plotting the output for time step 0...\n";
  gnuplot("all",daUn,daX(0),dDX,daY(0),dDY,0);

  // Evolve a step forward in time
  dTime = dDT;
  for(i=1;i<iNSteps;i++) {
    daUnp1(I,J) = ((dC*dC*dDT*dDT)/(dDX*dDX))*
                  (daUn(I-1,J)-2*daUn(I,J)+daUn(I+1,J)) +
                  2*daUn(I,J) - daUnm1(I,J);
    // Apply BC
    daUnp1(0,J) = daUnp1(iNumX,J);
    daUnp1(iNumX+1,J) = daUnp1(1,J);
    daUnp1(I,0) = daUnp1(I,iNumY);
    daUnp1(I,iNumY+1) = daUnp1(I,1);
#ifdef DISPLAY
    char caTmp[80];
    sprintf(caTmp,"U at %f sec",dTime);
    daUnp1.display(caTmp);
#endif
    daUnm1 = daUn;
    daUn = daUnp1;
    std::cout << "Plotting the output for time step " << i << "...\n";
    gnuplot("all",daUn,daX(0),dDX,daY(0),dDY,i);
    dTime += dDT;
  }
//  printf("dDX: %.3g dDY: %.3g\n",dDX,dDY);
  Optimization_Manager::Exit_Virtual_Machine();

  return 0;
}
