/*
 * File: compute_schedule.h
 * Auth: Jae-Yong Lee (jylee@cs.umd.edu)
 *       Chritian Hansen (chansen@cs.umd.edu)
 *       University of Maryland, College Park
 * Desc: Functions for computing schedules 
 * Date: Feb. 2004
 */

#include "compute_schedule.h"
#include "fort_wrap.h"
#include "communication.h"
#include "gen_schedules.h"
#include "intercomm.h"
#include "error.h"
#include <stdlib.h>
#include <string.h>

/* Compute a schedule for the given regions */
IC_Sched* IC_Compute_schedule(IC_Program* this, IC_Program* other, IC_Desc* desc, IC_Region** region_set, int set_size){
 
  int i, other_type;
  HowToSendDAD howsend;
  setOfRegion_s *ar_Set;
  IC_Sched* sched = 0;
  
  int my_num_data = 0;
  int other_num_data = 0;
  int pgme_order;

  setcall("IC_Compute_schedule");
  if (this == 0 || other == 0 || desc == 0 || region_set == 0) { 
    errormsg("invalid parameters");
    return 0;
  }
  
  ar_Set = Alloc_setOfRegion();
  for (i = 0; i < set_size; ++i) {
    Add_Region_setOfRegion(region_set[i], ar_Set);
    my_num_data += region_set[i]->size;
  }
  
  exch_transfer_info((pgme_s*)other, desc->type,my_num_data, &other_type, &other_num_data);

  /* Checking whether numbers of data elements are equal. */
  if (my_num_data != other_num_data) {
    errormsg("send and receive regions sizes are not equal");
    return 0;
  }
  
  /* Default method for how to send a compact data descriptor */
  howsend = SendOneToOne;
  
  /* Compact to Compact */
  if (desc->type == IC_BDECOMP && other_type == IC_BDECOMP) {
#ifdef IC_DEBUG
    printf("BDECOMP<->BDECOMP\n");
#endif
    sched = ComputeSchedule_Two_Compacts(howsend, desc->spec, ar_Set, (pgme_s*)this, (pgme_s*)other);
  /* Compact to Non-Compact */
  } else if (desc->type == IC_BDECOMP && other_type == IC_TTABLE) {
#ifdef IC_DEBUG
    printf("BDECOMP<->TTABLE\n");
#endif
    sched = ComputeScheduleOnCompact_Mixed_Both(howsend, desc->spec, ar_Set, (pgme_s*)this, (pgme_s*)other);
  /* Non-Compact to Compact */
  } else if (desc->type == IC_TTABLE && other_type == IC_BDECOMP) {
#ifdef IC_DEBUG
    printf("TTABLE<->BDECOMP\n");
#endif
    sched = ComputeScheduleOnNonCompact_Mixed_Both(howsend, desc->spec, ar_Set, (pgme_s*)this, (pgme_s*)other);
  /* Non-Compact to Non-Compact */
  } else if (desc->type == IC_TTABLE && other_type == IC_TTABLE) {
#ifdef IC_DEBUG
    printf("TTABLE<->TTABLE\n");
#endif
    
    /* Determine the order of programs (for the case with two compact data descriptors) */
    pgme_order = strcmp(this->name, other->name);
    
    if (pgme_order > 0) {
      sched = ComputeScheduleFor_Two_NonCompacts(First, desc->spec, ar_Set, (pgme_s*)this, (pgme_s*)other);
    } 
    else if (pgme_order < 0) {
      sched = ComputeScheduleFor_Two_NonCompacts(Second, desc->spec, ar_Set, (pgme_s*)this, (pgme_s*)other);
    }
    else {
      errormsg("communication within a single program is not supported");
      return 0;
    }
  }
  else { /* Invaild data descriptor types */
    errormsg("internal error");
    return 0;
  }
  
  Free_setOfRegion(ar_Set);
  
  return sched;
}

/*
 * Internal Compute Schedule Functions 
 */

/*
 * Compute schedules for two compact programs
 * Both programs independantly compute schedules
 */
sched_s* ComputeSchedule_Two_Compacts(HowToSendDAD howsend, 
				      void *datadescriptor,
				      setOfRegion_s *ar_Set,
				      pgme_s *my_pgme,
				      pgme_s *other_pgme)
{
  int me;
  int num_my_blocks;
  block** my_blocks;
  info_dd_arSet* info;
  sched_s* scheds;
  int nDims;
  int num_blocks;
  block** block_array;
  
  /*  
  int col_me, col_other;
  
  if(strcmp(my_pgme->name,"prog1") ==0) 
    {
      col_me = 1;
      col_other = 0;
    }
  else{
    col_me = 0;
    col_other = 1;
  }
  
  col_me = 1;
  col_other = 1;
  
  errormsg("Called with Fortan style");
  */
  me = MyPosMyPgme();

  /* 
   * Flatten my compact descriptor and resgion set before sending to 
   * other program. 
   * This should be modified to search the tree structure later    
  */
  num_blocks = get_num_of_blocks((decomp_Birreg*)datadescriptor);
  nDims = ((decomp_Birreg*)datadescriptor)->nDims;  
  block_array = (block**)malloc(num_blocks*sizeof(block*));
  store_block_array_ptr(((decomp_Birreg*)datadescriptor), block_array);
  my_blocks = get_my_blocks(me, block_array, num_blocks, &num_my_blocks, nDims);
  /* send and receiver the descriptor and region set  */
  send_compact_DAD_SetOfRegions(howsend, me, ar_Set, datadescriptor, my_pgme, other_pgme);
  
  info = receive_compact_DAD_SetOfRegions(howsend, me, my_pgme, other_pgme);
  if (!info)
    return 0;

  /* Generate communcation schedules only for this process */

  scheds = GenerateSchedules_Two_Compacts(NULL,
					  num_my_blocks,
					  NumNodePgme(other_pgme),
					  my_blocks,
					  info->block_array, 
					  ar_Set, 
					  info->ar_Set,
					  info->num_of_blocks_others,
					  ((decomp_Birreg*)datadescriptor)->col_majeur,
					  info->mem_layout
					  );
  /* Free memory */
  free(block_array);
  free(my_blocks);  
  Free_info_dd_arSet(info);
  
  return scheds;
} 

/*
 * Compute schedules for two compact programs
 * One program computes schedules. 
 * The responsible program calls this function.
 * The other programs calls 'ComputeScheduleOnOther'.
 */
sched_s* ComputeScheduleOnMe_Two_Compacts(HowToSendDAD howsend,
					  void *datadescriptor,
					  setOfRegion_s *ar_Set,
					  pgme_s *my_pgme,
					  pgme_s *other_pgme)
{
  int me;
  int nDims;
  int num_blocks;
  block** block_array;
  int num_my_blocks;
  block** my_blocks;
  info_dd_arSet* info;
  sched_s* scheds = NULL;
  sched_s** tab_sched_other = NULL;
  
  me = MyPosMyPgme();  

  /* 
   * Flatten my compact descriptor and resgion set.
   * This should be modified to search the tree structure later
   */

  num_blocks = get_num_of_blocks(((decomp_Birreg*)datadescriptor));
  nDims = ((decomp_Birreg*)datadescriptor)->nDims;
  block_array = (block**)malloc(num_blocks * sizeof(block*));
  store_block_array_ptr(((decomp_Birreg*)datadescriptor),block_array);
  my_blocks = get_my_blocks(me, block_array, num_blocks, &num_my_blocks,nDims);

  /* 
   * Receive the information but not send because only this program 
   * compute schedules for both programs.
   */
     
  info = receive_compact_DAD_SetOfRegions(howsend,me,my_pgme,other_pgme);  
  if (!info)
    return 0;
  
  tab_sched_other = CreateTabScheds(NumNodePgme(other_pgme), NumNodePgme(my_pgme));
 
  /* Generate communcation schedules for both programs */
  scheds = GenerateSchedules_Two_Compacts(tab_sched_other,
					  num_my_blocks,
					  NumNodePgme(other_pgme),
					  my_blocks,
					  info->block_array, 
					  ar_Set, 
					  info->ar_Set,
					  info->num_of_blocks_others,
					  ((decomp_Birreg*)datadescriptor)->col_majeur,
					  info->mem_layout
					  );
  
  /* send the computed schedule to other program */
  sendScheduleToPgme(tab_sched_other,other_pgme);
  
  /* Free memory */  
  FreeTabScheds(tab_sched_other,NumNodePgme(other_pgme));  
  free(block_array);
  free(my_blocks);  
  Free_info_dd_arSet(info);
  
  return scheds;
}

/*
 * Compute schedules for two compact programs.
 * One program computes schedules. 
 * The responsible program calls 'ComputeScheduleOnMe'.
 * The other programs calls this function.
 */
sched_s* ComputeScheduleOnOther_Two_Compacts(HowToSendDAD howsend,
					     void *datadescriptor,
					     setOfRegion_s *ar_Set,
					     pgme_s *my_pgme,
					     pgme_s *other_pgme)
{
  int me;
  sched_s* scheds = NULL;
  
  me = MyPosMyPgme();
  
  /* Send my compact descriptor and region set to other program */
  send_compact_DAD_SetOfRegions(howsend,me,ar_Set,datadescriptor,my_pgme,other_pgme);
  
  /* receive the computed schedules */
  scheds = ReceiveScheduleReverseOrder(other_pgme);
  
  return scheds;
}

/*
 * Compute schedules for one compact program and one non-compact program.
 * Non_compact program should call this function.
 */
sched_s* ComputeScheduleOnNonCompact_Mixed(HowToSendDAD howsend,
					   SendBackMethod sendback,
					   void *datadescriptor,
					   setOfRegion_s *ar_Set,
					   pgme_s *my_pgme,
					   pgme_s *other_pgme)    
{
  int me;
  info_dd_arSet* info;
  sched_s* scheds;
  sched_s** tab_sched_me;
  sched_s** tab_sched_other;
  simple_ttable* table;
  int OverallSize;
  
  me = MyPosMyPgme();

  OverallSize = SizeSetOfRegion(ar_Set);

  /* Receive a compact descriptor and region set from other program */
  info = receive_compact_DAD_SetOfRegions(howsend, me, my_pgme, other_pgme);
  if (!info)
    return 0;

  /* collecting info for responsible part of resgions */
  send_ttable_info(my_pgme, datadescriptor, ar_Set, OverallSize);
  table = receive_ttable_info(my_pgme, OverallSize);
  
  /* Allocation tables for schedules. */
  tab_sched_me = CreateTabScheds(NumNodePgme(my_pgme), NumNodePgme(other_pgme));
  tab_sched_other = CreateTabScheds(NumNodePgme(other_pgme), NumNodePgme(my_pgme));
  
  /* Generate schedules for both programs */
  GenerateSchedules_Mixed_On_Non_Compact(tab_sched_me,
					 tab_sched_other,
					 table, 
					 NumNodePgme(other_pgme),
					 info->block_array, 
					 info->ar_Set,
					 info->num_of_blocks_others,
					 info->mem_layout,
					 my_pgme,
					 OverallSize);
  
  /* send the computed schedules to other side(compact) */
  if (sendback == SendBackAll) {
    /* Send schedules to all processes of other program */
    sendScheduleToPgme(tab_sched_other, other_pgme); 
  } else {   
    /* Send them to just part of compact nodes and the nodes deal with them*/
    sendScheduleTableToOneNode(tab_sched_other, other_pgme, cal_source_node(my_pgme,other_pgme)); 
  } 
  
  /* Send schedules to nodes in my side(non-compact) */
  sendScheduleToPgme(tab_sched_me, my_pgme);
  
  /* Receive the schedules */
  scheds = ReceiveSchedule(my_pgme); 
  
  /* Free memory */
  FreeTabScheds(tab_sched_me, NumNodePgme(my_pgme));
  FreeTabScheds(tab_sched_other, NumNodePgme(other_pgme));
  Free_info_dd_arSet(info);
  FreeSimpleTable(table);
  
  return scheds;
}

/*
 * Compute schedules for one compact program and one non-compact program.
 * The compact program(not compute schedules) should call this function.
 */
sched_s* ComputeScheduleOnCompact_Mixed(HowToSendDAD howsend,
					SendBackMethod sendback,
					void *datadescriptor,
					setOfRegion_s *ar_Set,
					pgme_s *my_pgme,
					pgme_s *other_pgme)
{
  int me;
  sched_s* scheds;

  me = MyPosMyPgme();

  /* Send my compact descriptor and region set to other non-compact program */
  send_compact_DAD_SetOfRegions(howsend,me,ar_Set,datadescriptor,my_pgme,other_pgme);
  
  /* receive the computed schedules */
  scheds = NULL;
  
  if (sendback == SendBackAll) {
    /* receive schedules from all non-compact nodes */
    scheds = ReceiveSchedule(other_pgme);
  } else {
    scheds = Receive_Distribute_Schedules(my_pgme, other_pgme, me);   
  }

  return scheds;
}

/*
 * Compute schedules for one compact program and one non-compact program
 * Both compact and non-compact sides compute schedules.
 * Non-compact side calls this function.
*/
sched_s* ComputeScheduleOnNonCompact_Mixed_Both(HowToSendDAD howsend,
						void *datadescriptor,
						setOfRegion_s *ar_Set,
						pgme_s *my_pgme,
						pgme_s *other_pgme)
{
  int me;
  info_dd_arSet* info;
  sched_s* scheds = NULL;
  sched_s** tab_sched_me;
  sched_s** tab_sched_other;
  simple_ttable* table;
  int OverallSize;
  
  me = MyPosMyPgme();
  
  OverallSize = SizeSetOfRegion(ar_Set);  
  info = receive_compact_DAD_SetOfRegions(howsend, me, my_pgme, other_pgme);
  if (!info)
    return 0;
  
  /* collecting info for responsible part of resgions */  
  send_ttable_info_two_pgmes(my_pgme, other_pgme, datadescriptor, ar_Set, OverallSize, my_pgme);
  table = receive_ttable_info_two_pgmes(0, my_pgme, other_pgme, OverallSize, 0);  
  if (!table)
    return 0;

  tab_sched_me = CreateTabScheds(NumNodePgme(my_pgme), NumNodePgme(other_pgme));
  tab_sched_other = CreateTabScheds(NumNodePgme(other_pgme), NumNodePgme(my_pgme));
  
  /* Generate Schedules */  
  GenerateSchedules_Mixed_Both(tab_sched_me,
			       tab_sched_other,
			       table, 
			       NumNodePgme(other_pgme),
			       info->block_array, 
			       info->ar_Set,
			       info->num_of_blocks_others,
			       info->mem_layout,
			       my_pgme,
			       other_pgme,
			       OverallSize, 
			       0);  

  sendSchedule(tab_sched_other, tab_sched_me, other_pgme, my_pgme);
  scheds = ReceiveSchedules(my_pgme,other_pgme);
  
  /* Free memory */
  FreeTabScheds(tab_sched_me,NumNodePgme(my_pgme));
  FreeTabScheds(tab_sched_other,NumNodePgme(other_pgme));  
  FreeSimpleTable(table);
  Free_info_dd_arSet(info);
  
  return scheds;
}

/*
 * Compute schedules for one compact program and one non-compact program
 * Both sides compute schedules.
 * Compact side calls this function.
 */
sched_s* ComputeScheduleOnCompact_Mixed_Both(HowToSendDAD howsend,
					     void *datadescriptor,
					     setOfRegion_s *ar_Set,
					     pgme_s *my_pgme,
					     pgme_s *other_pgme)
{
  int me;
  int nDims;
  setOfRegion_s* ar_Set_other = 0;
  int num_blocks;
  block** block_array;  
  info_dd_arSet* info;
  int* temp_buff = NULL;
  int temp_count;
  sched_s** tab_sched_me;
  sched_s** tab_sched_other;  
  simple_ttable* table;  
  sched_s* scheds = NULL;
  int OverallSize;
  
  me = MyPosMyPgme();

  OverallSize = SizeSetOfRegion(ar_Set);  

  num_blocks = get_num_of_blocks(((decomp_Birreg*)datadescriptor));
  nDims = ((decomp_Birreg*)datadescriptor)->nDims;
  block_array = (block**)malloc(num_blocks*sizeof(block*));
  store_block_array_ptr(((decomp_Birreg*)datadescriptor), block_array);  

  temp_buff = build_info(nDims, ar_Set, num_blocks, block_array, ((decomp_Birreg*)( datadescriptor))->col_majeur,&temp_count);
  info = interpret_info(temp_buff, my_pgme, my_pgme);
  free(temp_buff);
  if (!info)
    return 0;
  
  send_compact_DAD_SetOfRegions(howsend, me, ar_Set, datadescriptor, my_pgme, other_pgme);
  
  /* receive the part of non-compact descriptor from non-compact processes */
  table  = receive_ttable_info_two_pgmes(NumNodePgme(other_pgme), other_pgme, my_pgme, OverallSize, 1);
  if (!table)
    return 0;

  tab_sched_me = CreateTabScheds(NumNodePgme(my_pgme), NumNodePgme(other_pgme));
  tab_sched_other = CreateTabScheds(NumNodePgme(other_pgme), NumNodePgme(my_pgme));
  
  /* Generate schedules */
  GenerateSchedules_Mixed_Both(tab_sched_other,
			       tab_sched_me,
			       table,
			       NumNodePgme(my_pgme),
			       info->block_array,
			       info->ar_Set,
			       info->num_of_blocks_others,
			       info->mem_layout,
			       my_pgme,
			       other_pgme,
			       OverallSize,
			       1);
  
  sendSchedule(tab_sched_me, tab_sched_other, my_pgme, other_pgme);
  scheds = ReceiveSchedules(other_pgme, my_pgme);
  
  /* Free memory */
  FreeTabScheds(tab_sched_me, NumNodePgme(my_pgme));
  FreeTabScheds(tab_sched_other, NumNodePgme(other_pgme));  
  FreeSimpleTable(table);
  Free_info_dd_arSet(info);
  free(block_array);

  return scheds;  
}

/*
 * Compute schedules for two non-compact programs
 * Both programs compute schedules and call this function.
 */
sched_s* ComputeScheduleFor_Two_NonCompacts(WhatAmI resp,
					    void *datadescriptor,
					    setOfRegion_s *ar_Set,
					    pgme_s *my_pgme,
					    pgme_s *other_pgme)
{
  int me;
  info_dd_arSet* info;
  sched_s* scheds = NULL;
  sched_s** tab_sched_me;
  sched_s** tab_sched_other;
  simple_ttable* table_me;
  simple_ttable* table_other;
  int OverallSize;
  
  me = MyPosMyPgme();

  OverallSize = SizeSetOfRegion(ar_Set);
  
  /* collecting info for responsible part of resgions */
  if (resp == First)
    { 
      send_ttable_info_two_pgmes(my_pgme, other_pgme, datadescriptor, ar_Set, OverallSize, my_pgme);
    }
  else{
    send_ttable_info_two_pgmes(other_pgme, my_pgme, datadescriptor, ar_Set, OverallSize, my_pgme);
  }


  if (resp == First){
    table_me = receive_ttable_info_two_pgmes(0, my_pgme, other_pgme, OverallSize, 0);
  }
  else{
    table_me = receive_ttable_info_two_pgmes(NumNodePgme(other_pgme), my_pgme, other_pgme, OverallSize, 0);
  }
  if (!table_me)
    return 0;

  if (resp == First){
    table_other = receive_ttable_info_two_pgmes(0, other_pgme, my_pgme, OverallSize, 1);
  }
  else{
    table_other = receive_ttable_info_two_pgmes(NumNodePgme(other_pgme), other_pgme, my_pgme, OverallSize, 1);
  }
  if (!table_other)
    return 0;
  
  tab_sched_me = CreateTabScheds(NumNodePgme(my_pgme), NumNodePgme(other_pgme));
  tab_sched_other = CreateTabScheds(NumNodePgme(other_pgme), NumNodePgme(my_pgme));
  
  /* Generate scheudles */
  GenerateSchedules_Two_Non_Compacts(tab_sched_me, 
				     tab_sched_other, 
				     table_me, 
				     table_other, 
				     my_pgme, 
				     other_pgme, 
				     OverallSize);
  
  /* Send schedules */
  if (resp == Sender)
    sendSchedule(tab_sched_other, tab_sched_me, other_pgme, my_pgme);
  else
    sendSchedule(tab_sched_me, tab_sched_other, my_pgme, other_pgme);
  
  /* Receive schedules */
  if (resp == Sender)
    scheds = ReceiveSchedules(my_pgme, other_pgme);
  else 
    scheds = ReceiveSchedules(other_pgme, my_pgme);
  
  /* Free memory */
  FreeTabScheds(tab_sched_me,NumNodePgme(my_pgme));
  FreeTabScheds(tab_sched_other,NumNodePgme(other_pgme));
  FreeSimpleTable(table_me);
  FreeSimpleTable(table_other);
 
  return scheds;
}






