#include <iostream>
#include <cstdlib>
#include <cstring>
#include <assert.h>
#include "IC_EndPoint.h"
#include "c-wrapper.h"
#include "carray_descriptor.h"

using namespace std;

// #define IC_DEBUG

#define IC_ENDPOINT_CONSTRUCTOR_F77 F77_FUNC_(ic_endpoint_constructor, IC_ENDPOINT_CONSTRUCTOR)
#define IC_ENDPOINT_DESTRUCTOR_F77  F77_FUNC_(ic_endpoint_destructor, IC_ENDPOINT_DESTRUCTOR)
#define IC_PRINT_ERROR_MESSAGE_F77  F77_FUNC_(ic_print_error_message, IC_PRINT_ERROR_MESSAGE)
#define IC_PRINT_ERROR_F77 F77_FUNC_(ic_print_error, IC_PRINT_ERROR)

#define IC_INIT_F77 F77_FUNC_(ic_init, IC_INIT)
#define IC_WAIT_F77 F77_FUNC_(ic_wait, IC_WAIT)
#define IC_SYNC_F77 F77_FUNC_(ic_sync, IC_SYNC)

#define IC_CREATE_BDECOMP_DESC_HELPER_F77 F77_FUNC_(ic_create_bdecomp_desc_helper, IC_CREATE_BDECOMP_DESC_HELPER)
#define IC_CREATE_TTABLE_DESC_HELPER_F77  F77_FUNC_(ic_create_ttable_desc_helper, IC_CREATE_TTABLE_HELPER_DESC)
#define IC_CREATE_ENUM_REGION_HELPER_F77  F77_FUNC_(ic_create_enum_region_helper, IC_CREATE_ENUM_REGION_HELPER)
#define IC_CREATE_BLOCK_REGION_HELPER_F77 F77_FUNC_(ic_create_block_region_helper, IC_CREATE_BLOCK_REGION_HELPER)
#define IC_COMPUTE_SCHEDULE_HELPER_F77    F77_FUNC_(ic_compute_schedule_helper, IC_COMPUTE_SCHEDULE_HELPER)

#define IC_SEND_CHAR_F77   F77_FUNC_(ic_send_char, IC_SEND_CHAR)
#define IC_SEND_SHORT_F77  F77_FUNC_(ic_send_short, IC_SEND_SHORT)
#define IC_SEND_INT_F77    F77_FUNC_(ic_send_int, IC_SEND_INT)
#define IC_SEND_LONG_F77   F77_FUNC_(ic_send_long, IC_SEND_LONG)
#define IC_SEND_FLOAT_F77  F77_FUNC_(ic_send_float, IC_SEND_FLOAT)
#define IC_SEND_DOUBLE_F77 F77_FUNC_(ic_send_double, IC_SEND_DOUBLE)

#define IC_RECV_CHAR_F77   F77_FUNC_(ic_recv_char, IC_RECV_CHAR)
#define IC_RECV_SHORT_F77  F77_FUNC_(ic_recv_short, IC_RECV_SHORT)
#define IC_RECV_INT_F77    F77_FUNC_(ic_recv_int, IC_RECV_INT)
#define IC_RECV_LONG_F77   F77_FUNC_(ic_recv_long, IC_RECV_LONG)
#define IC_RECV_FLOAT_F77  F77_FUNC_(ic_recv_float, IC_RECV_FLOAT)
#define IC_RECV_DOUBLE_F77 F77_FUNC_(ic_recv_double, IC_RECV_DOUBLE)

#define IC_FREE_SCHED_F77   F77_FUNC_(ic_free_sched, IC_FREE_SCHED)
#define IC_FREE_REGION_F77  F77_FUNC_(ic_free_region, IC_FREE_REGION)
#define IC_FREE_DESC_F77    F77_FUNC_(ic_free_desc, IC_FREE_DESC)
#define IC_FREE_PROGRAM_F77 F77_FUNC_(ic_free_program, IC_FREE_PROGRAM)

#define IC_QUIT_F77 F77_FUNC_(ic_quit, IC_QUIT)

extern "C" {

void* parray[IC_MAX_POINTERS];
int pnext = 0;
int nentries=0;

//---------------------------------- IC internal functions --------------------
//

int pinsert(void* p) {
  if (!p)
    return -1;

  if (nentries==IC_MAX_POINTERS)
    return IC_EndPoint::IC_POINTER_TABLE_ERROR;
  
  while (parray[pnext]) {
    pnext = (pnext+1) % IC_MAX_POINTERS;
  }
  parray[pnext] = p;
  ++nentries;
  assert(nentries<=IC_MAX_POINTERS);
  return pnext;
}

void* plookup(int i) {
  if(i<0 || i>=IC_MAX_POINTERS)
    return NULL;

  return parray[i];
}

void* premove(int i) {
  if(i<0 || i>=IC_MAX_POINTERS)
    return NULL;
  void* p;
  p = parray[i];
  parray[i] = NULL;
  --nentries;
  assert(nentries>=0);
  return p;
}

char* ftoc(char* s, int l) {
  char* t;
  int i = 0;

  t = (char*)calloc(l+1, sizeof(char));
  while (i < l) {
    t[i] = s[i];
    ++i;
  }
  t[i] = 0;

  return t;
}

//---------------------------------- IC functions ----------------------------
//

void IC_ENDPOINT_CONSTRUCTOR_F77(IC_EndPoint** o, 
    const char* endpointName, const unsigned& mynproc, const unsigned& onproc,
    const unsigned& myao, const unsigned& oao, int& status) {
  cout << "endpoint is: [" << endpointName << "]" << endl;
  (*o)=new IC_EndPoint(endpointName,mynproc,onproc,myao,oao,status);
#ifdef DEBUG
  cout << "object is " << (void*)o << "..\n";
#endif
}

void IC_ENDPOINT_DESTRUCTOR_F77(IC_EndPoint** o) {
#ifdef DEBUG
  cout << "deleting endpoint " << (void*)(*o) << "..\n";
#endif
//  sleep(10);
  delete *o;
}

void IC_PRINT_ERROR_MESSAGE_F77(char* msg, unsigned& status) {
  IC_EndPoint::printErrorMessage(msg,status);
}

void IC_PRINT_ERROR_F77(char* m, int l) {
  char* msg;
  
  msg = ftoc(m, l);
  IC_Print_error(msg);
  free(msg);
}

void IC_INIT_F77(char* g, int* tasks, int* rank, int* p, int l) {
  char* group;
  IC_Program* thisp;

  // let's make sure the pointer table is properly initialized
  unsigned i;
  for(i=0;i<IC_MAX_POINTERS;++i)
    parray[i]=NULL;

  group = ftoc(g, l);

  thisp = IC_Init(group, *tasks, *rank);

  *p = pinsert((void*)thisp);
#ifdef IC_DEBUG
  printf("IC_Init(%s, %d, %d)=%d\n", group, *tasks, *rank, *p);
#endif
  free(group);
}

void IC_WAIT_F77(char* g, int* tasks, int* p, int l) {
  char* group;
  IC_Program* otherp;

  group = ftoc(g, l);

#ifdef IC_DEBUG
  printf("waiting for %s\n",group);
#endif

  otherp = IC_Wait(group, *tasks);

  *p = pinsert((void*)otherp);
#ifdef IC_DEBUG
  printf("IC_Wait(%s, %d)=%d\n", group, *tasks, *p);
#endif
  free(group);
}

void IC_SYNC_F77(int* p1, int* p2, int* sts) {
  IC_Program* thisp;
  IC_Program* otherp;

  thisp = (IC_Program*)plookup(*p1);
  otherp = (IC_Program*)plookup(*p2);

  *sts = IC_Sync(thisp, otherp);

#ifdef IC_DEBUG
  printf("IC_Sync(%d, %d)=%d\n", *p1, *p2, *sts);
#endif
}

#include "general_bsparti.h"

void IC_CREATE_BDECOMP_DESC_HELPER_F77(int* rank, const F90_intArray** b, 
  const F90_intArray** t, int* count, int* d) {
  int i, j, *blocks, *tasks;
  IC_Desc* desc;
  decomp_Birreg* decomp;

  tasks = (int*)calloc(*count, sizeof(int));
  assert(tasks);
  blocks = (int*)calloc((*count)*2*(*rank), sizeof(int));
  assert(blocks);
  // fortran arrays start at 1 
  for (i=0; i<*count; ++i) {
    for (j=0; j<*rank; ++j) {
      blocks[i*2*(*rank)+j] = (*b)->base[j*2*(*count)+i] - 1;
      blocks[i*2*(*rank)+(*rank)+j] = (*b)->base[j*2*(*count)+(*count)+i] - 1;
    }
    tasks[i]=(*t)->base[i];
  }
#ifdef IC_DEBUG
  for (i=0; i<*count; ++i)
    for (j=0; j<*rank; ++j)
      printf("%d %d\n",blocks[i*2*(*rank)+j],blocks[i*2*(*rank)+(*rank)+j]);
#endif
  
  desc = IC_Create_bdecomp_desc(*rank, blocks, tasks, *count);
  decomp = (decomp_Birreg*)desc->spec;
  decomp->col_majeur = 1;
#ifdef IC_DEBUG
  print_irr(decomp->root);
#endif

  *d = pinsert((void*)desc);
  free(blocks);
  free(tasks);
  
#ifdef IC_DEBUG
  printf("IC_Create_bdecomp(%d, x, x, %d)=%d\n", *rank, *count, *d);
#endif
}

void IC_CREATE_TTABLE_DESC_HELPER_F77(const F90_intArray** g, 
  const F90_intArray** l, const F90_intArray** t, int* count, int* d) {
  int i, *globals, *locals, *tasks;
  IC_Desc* desc;

  globals = (int*)calloc(*count, sizeof(int));
  locals = (int*)calloc(*count, sizeof(int));
  tasks = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    globals[i] = (*g)->base[i] - 1;
    locals[i] = (*l)->base[i] - 1;
    tasks[i] = (*t)->base[i];
  }
  
  desc = IC_Create_ttable_desc(globals, locals, tasks, *count);

  *d = pinsert((void*)desc);
  free(tasks);
  free(locals);
  free(globals);
  
#ifdef IC_DEBUG
  printf("IC_Create_ttable(x, x, x, %d)=%d\n", *count, *d);
#endif
}

void IC_CREATE_BLOCK_REGION_HELPER_F77(int* rank, const F90_intArray** l, 
  const F90_intArray** u, const F90_intArray** s, int* r) {
  int i, *lower, *upper, *stride;
  IC_Region* region;

  lower = (int*)calloc(*rank, sizeof(int));
  upper = (int*)calloc(*rank, sizeof(int));
  stride = (int*)calloc(*rank, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *rank; ++i) {
    lower[i] = (*l)->base[i] - 1;
    upper[i] = (*u)->base[i] - 1;
    stride[i] = (*s)->base[i];
  }

  region = IC_Create_block_region(*rank, lower, upper, stride);
#ifdef IC_DEBUG
//  Print_Region(region);
#endif

  *r = pinsert((void*)region);
  free(upper);
  free(lower);
  free(stride);
}

void IC_CREATE_ENUM_REGION_HELPER_F77(const F90_intArray** x, int* count, 
  int* r) {
  int i, *indices;
  IC_Region* region;

  indices = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    indices[i] = (*x)->base[i] - 1;
  }

  region = IC_Create_enum_region(indices, *count);

  *r = pinsert((void*)region);
  free(indices);
}

void IC_COMPUTE_SCHEDULE_HELPER_F77(int* p1, int* p2, int* d, 
  const F90_intArray** r, int* num_of_regions, int* s) {
  IC_Program* thisp;
  IC_Program* otherp;
  IC_Desc* desc;
  IC_Region** regions;
  IC_Sched* sched;
  int i;

  thisp = (IC_Program*)plookup(*p1);
  otherp = (IC_Program*)plookup(*p2);
  desc = (IC_Desc*)plookup(*d);
  regions = (IC_Region**)calloc(*num_of_regions, sizeof(IC_Region*));
  for (i = 0; i < *num_of_regions; ++i) {
    regions[i] = (IC_Region*)plookup((*r)->base[i]);
  }

  sched = IC_Compute_schedule(thisp, otherp, desc, regions, *num_of_regions);

  *s = pinsert((void*)sched);
  free(regions);
}

void IC_FREE_SCHED_F77(int* s) {
  IC_Sched* sched;

  sched = (IC_Sched*)premove(*s);

  IC_Free_sched(sched);
}

void IC_FREE_REGION_F77(int* r) {
  IC_Region* region;

  region = (IC_Region*)premove(*r);

  IC_Free_region(region);
}

void IC_FREE_DESC_F77(int* d) {
  IC_Desc* desc;

  desc = (IC_Desc*)premove(*d);

  IC_Free_desc(desc);
}

void IC_FREE_PROGRAM_F77(int* p) {
  IC_Program* otherp;

  otherp = (IC_Program*)premove(*p);

  IC_Free_program(otherp);
}

void IC_QUIT_F77(int* p) {
  IC_Program* thisp;

  thisp = (IC_Program*)premove(*p);

  IC_Quit(thisp);
}

}
