#include <assert.h>
#include <dlfcn.h>
#include <errno.h>
#include <iostream>
#include <malloc.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <typeinfo>
#include <unistd.h>
#include <sys/types.h>
#include "IC_support.h"
#include "IC_util.h"
#include "InterComm.h"

#if defined(DBDBG)
//#define DCDBG_MUTEX_LOCKUNLOCK
#endif

#include <stdio.h>
#ifdef PPLUSPLUS
#include <A++.h>
#endif


#define GRID

#ifdef MY_MALLOC_STUFF

long usedBytes=0;

void *getLibraryFunction(const char *funcName) {
// returns a pointer to the implementation of 'funcName' in
// the libc library.  If not found, terminates the program.
  void *res;

  if ((res = dlsym(RTLD_NEXT, funcName)) == NULL) {
    fprintf(stderr, "dlsym %s error:%s\n", funcName, dlerror());
    _exit(1);
  }
  return res;
}

void exit(int status) {
  typedef void (*exit_t)(int) __attribute__ ((noreturn));

  static exit_t real_exit=NULL;
  if (!real_exit)
    real_exit=(exit_t)getLibraryFunction("exit");

  printf("current memory usage is approximately %d\n",usedBytes);
  real_exit(status);
}

void free(void *ptr) {
  typedef void (*free_t)(void*);

  static free_t real_free=NULL;
  if (!real_free)
    real_free=(free_t)getLibraryFunction("free");

  real_free(ptr);
}


void *valloc(size_t size) {
  printf("vallocating %d bytes\n",size);
  typedef void* (*valloc_t)(size_t);
  static valloc_t real_valloc=NULL;
  if (!real_valloc)
    real_valloc=(valloc_t)getLibraryFunction("valloc");

  void* m=real_valloc(size);
  if (m==NULL) {
    printf("error vallocating memory\n");
    exit(-1);
  }
  usedBytes+=size;
  return m;
}


void *realloc(void* ptr, size_t size) {
//  printf("reallocating %d bytes\n",size);
  typedef void* (*realloc_t)(void*,size_t);
  static realloc_t real_realloc=NULL;
  if (!real_realloc)
    real_realloc=(realloc_t)getLibraryFunction("realloc");

  void* m=real_realloc(ptr,size);
  if (m==NULL) {
    printf("error reallocating memory\n");
    exit(-1);
  }
  // not really right, but that's all I want now!
  usedBytes+=size;
  return m;
}

void *calloc(size_t nelem, size_t elsize) {
//  printf("callocating %d bytes\n",nelem*elsize);
  typedef void* (*calloc_t)(size_t,size_t);
  static calloc_t real_calloc=NULL;
  if (!real_calloc)
    real_calloc=(calloc_t)getLibraryFunction("calloc");

  void* m=real_calloc(nelem,elsize);
  if (m==NULL) {
    printf("error callocating memory\n");
    exit(-1);
  }
  usedBytes+=nelem*elsize;
  return m;
}

void *malloc(size_t size) {
//  printf("allocating %d bytes\n",size);
  typedef void* (*malloc_t)(size_t);
  static malloc_t real_malloc=NULL;
  if (!real_malloc)
    real_malloc=(malloc_t)getLibraryFunction("malloc");

  void* m=real_malloc(size);
  if (m==NULL) {
    printf("error allocating memory\n");
    exit(-1);
  }
  usedBytes+=size;
  return m;
}
#endif

#ifdef PPLUSPLUS
void gnuplot(const char* prefix, const doubleArray& a, const double baseX, 
  const double dX, const double baseY, const double dY, const unsigned ts) {
  int i, j;
  printf("generating output -> \"~/%s-%05d-%03d.pbm\"\n",prefix,getuid(),ts);
  FILE* gnuplot=popen("gnuplot","w");
  assert(gnuplot);
  fprintf(gnuplot,"set output \"~/%s-%05d-%03d.pbm\"\n",prefix,getuid(),ts);
  fprintf(gnuplot,"set terminal pbm\n");
#ifdef GRID
  fprintf(gnuplot,"set dgrid3d %d,%d,4\n",a.getLength(0),a.getLength(1));
//  printf("set dgrid3d %d,%d,4\n",a.getLength(0),a.getLength(1));
  fprintf(gnuplot,"set hidden3d\n");
  fprintf(gnuplot,"set data style line\n");
#else
  fprintf(gnuplot,"set data style points\n");
#endif
  fprintf(gnuplot,"set nokey\n");
  fprintf(gnuplot,"set xrange [0:2]\n");
  fprintf(gnuplot,"set yrange [0:2]\n");
  fprintf(gnuplot,"set zrange [-1.2:1.2]\n");
  fprintf(gnuplot,"set title \"Wave Equation\"\n");
  fprintf(gnuplot,"splot \"-\"\n");
  for(j=a.getBase(1);j<=a.getBound(1);j+=a.getStride(1)) {
    for(i=a.getBase(0);i<=a.getBound(0);i+=a.getStride(0)) {
      fprintf(gnuplot,"%.3g %.3g %.3g\n",baseX+i*dX,baseY+j*dY,a(i,j));
//      printf("%.3g %.3g %.3g\n",i*dX,j*dY,a(i,j));
    }
  }
  fprintf(gnuplot,"end\n");
  pclose(gnuplot);
}
#endif

//
//
//  class Mutex
//


// ctor/dtor
Mutex::Mutex(char *sbName_in, pthread_mutexattr_t *pattr) {
#if defined(DC_MUTEX_DEBUG)
  char sbTmp[40];
  snprintf(sbTmp, 80, "%s(%lx)", (sbName_in ? sbName_in : "Mutex"), this);
  sbName = StrDup(sbTmp);
#endif /* DC_MUTEX_DEBUG */
#if defined(CONFIG_HAVE_PTHREAD_D4)
  // must use constant instead of NULL
  int ret = pthread_mutex_init(&mutex,
			       pattr ? pattr : pthread_mutexattr_default);
#else /* CONFIG_HAVE_PTHREAD_D4 */
#ifdef DEBUG
  std::cout << "Mutex::Mutex: Using error-checking mutexes...\n";
  pthread_mutexattr_t mattr;
  pthread_mutexattr_init(&mattr);
  int ret=pthread_mutexattr_settype(&mattr,PTHREAD_MUTEX_ERRORCHECK_NP);
  assert(ret==0);
  ret=pthread_mutex_init(&mutex,&mattr);
#else
  int ret=pthread_mutex_init(&mutex, pattr);
#endif
#endif /* CONFIG_HAVE_PTHREAD_D4 */
  if (ret != 0) {
    std::cerr << "Mutex(): failed pthread_mutex_init - "
	 << strerror(errno) << endl;
  }
#if defined(DC_MUTEX_DEBUG)
  fLocked = false;
#endif /* DC_MUTEX_DEBUG */
}
Mutex::~Mutex(void) {
#if defined(DC_MUTEX_DEBUG)
  if (fLocked) {
    std::cerr << ">>> " << sbName
	 << ".~Mutex(): should not destroy a locked mutex." << endl;
  }
  delete[] sbName;
#endif /* DC_MUTEX_DEBUG */
  int ret = pthread_mutex_destroy(&mutex);
  if (ret != 0) {
    std::cerr << "~Mutex(): failed pthread_mutex_destroy - "
	 << strerror(ret) << endl;
  }
}

void Mutex::lock(void) {
#if defined(DC_MUTEX_DEBUG)
  if (fLocked && pthread_equal(tidOwner, pthread_self())) {
    std::cerr << ">>> " << sbName
	 << ".lock(): current thread already lock owner." << endl;    
  }
#endif /* DC_MUTEX_DEBUG */
  int ret = pthread_mutex_lock(&mutex);
  if (ret != 0) {
    std::cerr << "Mutex::lock() failed pthread_mutex_lock - "
	 << strerror(errno) << endl;
  }
#if defined(DC_MUTEX_DEBUG)
  else {
    fLocked = true;
    tidOwner = pthread_self();
#if defined(DCDBG_MUTEX_LOCKUNLOCK)
    std::cerr << ">>> " << sbName << ".lock(): ok." << endl;
#endif /* DCDBG_MUTEX_LOCKUNLOCK */
  }
#endif /* DC_MUTEX_DEBUG */
}
void Mutex::unlock(void) {
#if defined(DC_MUTEX_DEBUG)
  if (!fLocked) {
    std::cerr << ">>> " << sbName << ".unlock(): mutex is not locked." << endl;
  } else if (!pthread_equal(tidOwner, pthread_self())) {
    std::cerr << ">>> " << sbName
	 << ".unlock(): current thread not lock owner." << endl;
  }
#if defined(DCDBG_MUTEX_LOCKUNLOCK)
  else {
    std::cerr << ">>> " << sbName << ".unlock(): ok." << endl;
  }
#endif /* DCDBG_MUTEX_LOCKUNLOCK */
  fLocked = false;
#endif /* DC_MUTEX_DEBUG */
  int ret = pthread_mutex_unlock(&mutex);
  if (ret != 0) {
    std::cerr << "Mutex::unlock() failed pthread_mutex_unlock - "
	 << strerror(errno) << endl;
  }  
}

// return T if it was locked
bool Mutex::trylock(void) {
  unsigned nt=0;
  int ret;
  do {
    ret = pthread_mutex_trylock(&mutex);
    if (ret == EBUSY) return false;
    if (ret != 0) {
      std::cerr << "Mutex::trylock() failed pthread_mutex_trylock - "
           << strerror(errno) << endl;
    }
    nt++;
  } while(ret==EINTR && nt<3);
  assert(!(ret==EINTR && nt>=3));
  return true;
}


// assume: locked
// called for cases where mutex ownership changes w/o calling lock/unlock
void Mutex::setOwner(void) {
#if defined(DC_MUTEX_DEBUG)
  fLocked = true;
  tidOwner = pthread_self();
#if defined(DCDBG_MUTEX_LOCKUNLOCK)
  std::cerr << ">>> " << sbName << ".setOwner(): ok." << endl;
#endif /* DCDBG_MUTEX_LOCKUNLOCK */
#endif /* DC_MUTEX_DEBUG */  
}

// ****************************************************************************

//
//  class ConfFile
//

//----- CF_AttrType 

CF_AttrType::CF_AttrType(const char* myname, const unsigned maxvalues) : 
  name(myname), values(maxvalues), inspected(false) {
}

CF_AttrType::CF_AttrType(const CF_AttrType& a) : name(a.name), 
  values(a.values.getMaxElements()) {
}

CF_AttrType::~CF_AttrType() { 
}

//----- CF_SectionType 

CF_SectionType::CF_SectionType(const char* myname, const unsigned maxattrs) : 
  name(myname), attrs(maxattrs) {
}

CF_SectionType::CF_SectionType(const CF_SectionType& s) : name(s.name), 
  attrs(s.attrs.getMaxElements()) {
}

CF_SectionType::~CF_SectionType() { 
}

//----- ConfFile

ConfFile::ConfFile() : sects(1), lastErrno(CF_OK) {
}

ConfFile::ConfFile(const ConfFile& myconf) {
  assert(0);
}

void ConfFile::removeAllSpaces(char *line) const {
  int i, j, k, linelen;

  linelen=strlen(line);
  for(i=0;i<linelen;++i) {
    if (line[i]==' ') {
      for(j=i+1;j<linelen;++j)
        if (line[j]!=' ')  
          break;
      // possibly overlapping area, let's use memmove
      memmove(line+i,line+j,linelen-j);
      for(k=0;k<j-i;++k) 
        line[linelen-k-1]='\0';
      linelen=strlen(line);
    }
  }
}

char* ConfFile::my_strtok(const char *str, const char *delim) const {
   static char buf[CF_STRLEN];
   static char *ptr;
   static int prefix;

   if (str!=NULL) { 
     StrNCpy(buf,str,CF_STRLEN);
     ptr=buf;
   }
   else {
     ptr=ptr+prefix+1;
     if (*ptr=='\0')
       return NULL;
   }
   prefix=strcspn(ptr,delim);
   ptr[prefix]='\0';

   return ptr;
}

int ConfFile::readConfFile(const char *filename) {
  unsigned nattrib, nsections, nvalues, state;
  int ret;
  char line[CF_STRLEN], values[CF_STRLEN], *ptr;
  FILE *fp;

  fp=fopen(filename,"r");
  if (fp==NULL) {
//    perror("fopen");
    return lastErrno=CF_CANTOPEN;
  }
  nsections=0;
  while(!feof(fp)) {
    ret=fscanf(fp,"%[] #0-9<=>,A-Z_a-z[/_.:-]",line);
    if (ret==-1)
      break;
    if (ret==0) {
      fgets(line,CF_STRLEN,fp);
      continue;
    }
    removeAllSpaces(line);
#ifdef DEBUG
    printf("line: %s\n",line);
#endif
    
    if (line[0]=='#') // line is a comment 
      continue;

    if (line[0]=='[') {// line is a new section 
      char sectionName[CF_STRLEN];
      sscanf(line,"[%[_/.0-9A-Za-z]]",sectionName);
      CF_SectionType sec(StrDup(sectionName));
      sects.add(sec);
      nattrib=0;
      nsections++;
      state=CF_READING_ATTRIBUTES;
      continue;
    }

    if (state==CF_READING_ATTRIBUTES) { // line is an attribute 
      char attrName[CF_STRLEN];
      sscanf(line,"%[_0-9A-Za-z]=%[_0-9<>,A-Za-z/_.:-]",attrName,values);
      CF_AttrType attr(StrDup(attrName));
      sects.getElement(nsections-1)->attrs.add(attr);
      state=CF_READING_VALUES;
    }

    if (state==CF_READING_VALUES) { // reading values
      if (values[0]=='<') { // it's a list of values 
        nvalues=0;
        ptr=my_strtok(&values[1],",>");
        do {
          if(ptr==NULL)
            break;

          if(*ptr=='\0')
            sects.getElement(nsections-1)->attrs.getElement(nattrib)->values.add(StrDup(""));
          else
            sects.getElement(nsections-1)->attrs.getElement(nattrib)->values.add(StrDup(ptr));
          ptr=my_strtok(NULL,",>");
          nvalues++;
        } while (1);
      }
      else
        sects.getElement(nsections-1)->attrs.getElement(nattrib)->values.add(StrDup(values));
      nattrib++;
      state=CF_READING_ATTRIBUTES;
    }
  }
  fclose(fp);
#ifdef DEBUG
  std::cout << *this << endl;
#endif

  return lastErrno=CF_OK;
}

int ConfFile::getNSections() const {
  return sects.getNElements();
}

int ConfFile::getNSectionEntries(const char *section) const {
  unsigned i;

  for(i=0;i<sects.getNElements();++i) {
    if (strcmp(sects.getElement(i)->name,section)==0)
      return sects.getElement(i)->attrs.getNElements();
  }
  return -1;
}

int ConfFile::getNAttrEntries(const char *section, const char *attr) const {
  unsigned i, j;

  for(i=0;i<sects.getNElements();++i) 
    if (strcmp(sects.getElement(i)->name,section)==0)
      for(j=0;j<sects.getElement(i)->attrs.getNElements();++j)
        if (strcmp(sects.getElement(i)->attrs.getElement(j)->name,attr)==0)
          return sects.getElement(i)->attrs.getElement(j)->values.getNElements();
  return -1;
}

const char* ConfFile::getValue4(const char *section, const char *attr) const {
  unsigned i, j;

  for(i=0;i<sects.getNElements();++i) 
    if (strcmp(sects.getElement(i)->name,section)==0)
      for(j=0;j<sects.getElement(i)->attrs.getNElements();++j)
        if (strcmp(sects.getElement(i)->attrs.getElement(j)->name,attr)==0) {
          sects.getElement(i)->attrs.getElement(j)->inspected=true;
          return *sects.getElement(i)->attrs.getElement(j)->values.getElement(0);
        }
 
  return NULL;
}

const char* ConfFile::getValue4N(const char *section, const char *attr, 
    const unsigned n) const {
  unsigned i, j;

  for(i=0;i<sects.getNElements();++i) 
    if (strcmp(sects.getElement(i)->name,section)==0)
      for(j=0;j<sects.getElement(i)->attrs.getNElements();++j)
        if (strcmp(sects.getElement(i)->attrs.getElement(j)->name,attr)==0) {
          if (n>=sects.getElement(i)->attrs.getElement(j)->values.getNElements())
            return NULL;
          sects.getElement(i)->attrs.getElement(j)->inspected=true;
          return *sects.getElement(i)->attrs.getElement(j)->values.getElement(n);
        }
 
  return NULL;
}

int ConfFile::getPosition(const char *section, const char *attr, 
  const char *val) const {
  unsigned i, j, k;

  for(i=0;i<sects.getNElements();++i)
    if (strcmp(sects.getElement(i)->name,section)==0)
      for(j=0;j<sects.getElement(i)->attrs.getNElements();++j)
        if (strcmp(sects.getElement(i)->attrs.getElement(j)->name,attr)==0) {
          unsigned n = sects.getElement(i)->attrs.getElement(j)->values.getNElements();
          for (k=0;k<n;++k)
            if (strcmp(*sects.getElement(i)->attrs.getElement(j)->values.getElement(k), val) == 0)
              return k;
        }
  return -1;
}

bool ConfFile::areThereAnyNonInspectedEntries() const {
  unsigned i, j;

  bool toret=false;
  for(i=0;i<sects.getNElements();++i)
    for(j=0;j<sects.getElement(i)->attrs.getNElements();++j)
      if (!sects.getElement(i)->attrs.getElement(j)->inspected) {
        std::cerr << "WARNING: '" << sects.getElement(i)->name << "->" << sects.getElement(i)->attrs.getElement(j)->name << "' in the configuration file has not been used! May be incorrectly set or be a duplicate...\n";
        toret=true;
      }

  return toret;
}

ConfFile::~ConfFile() {
  unsigned i, j, k;
  for(i=0;i<sects.getNElements();++i) {
    for(j=0;j<sects.getElement(i)->attrs.getNElements();++j) {
      for(k=0;k<sects.getElement(i)->attrs.getElement(j)->values.getNElements();++k)
        delete[] *sects.getElement(i)->attrs.getElement(j)->values.getElement(k);
      delete[] sects.getElement(i)->attrs.getElement(j)->name;
    }
    delete[] sects.getElement(i)->name;
  }
}

#ifndef INSURE
std::ostream& operator<< (std::ostream& o, const CF_AttrType& a) {
  o << "attrname: " << a.name << " nvalues: " << a.values.getNElements() << " values:";
  unsigned i;
  for(i=0;i<a.values.getNElements();++i)
    o << " " << *a.values.getElement(i);

  return o;
}

std::ostream& operator<< (std::ostream& o, const CF_SectionType& s) {
  o << "sectionname: " << s.name << " nattrs: " << s.attrs.getNElements() << " attrs: ";
  unsigned i;
  for(i=0;i<s.attrs.getNElements();++i)
    o << *s.attrs.getElement(i) << std::endl;

  return o;
}

std::ostream& operator<< (std::ostream& o, const ConfFile& cf) {
  o << "nsections: " << cf.sects.getNElements() << " sections: ";
  unsigned i;
  for(i=0;i<cf.sects.getNElements();++i)
    o << *cf.sects.getElement(i) << std::endl;

  return o;
}
#endif

// ****************************************************************************

//
//  class RArray
//


template<class entryT>
RArray<entryT>::RArray(const unsigned mymaxelements, const bool mycompactIt) : 
   nelements(0), maxelements(mymaxelements), sorted(false), 
   compactIt(mycompactIt) { 
  assert(maxelements>0);
  container=new entryT*[maxelements];
  unsigned i;
  for(i=0;i<maxelements;++i)
    container[i]=NULL;
}

template<class entryT>
RArray<entryT>::RArray(const RArray<entryT>& la) : container(NULL), 
  nelements(la.nelements), maxelements(la.maxelements), sorted(la.sorted),
  compactIt(la.compactIt) {
  assert(maxelements>0);
  container=new entryT*[maxelements];
  unsigned i;
  for(i=0;i<maxelements;++i) {
    if (la.container[i])
      container[i]=new entryT(*(la.container[i]));
    else
      container[i]=NULL;
  }
}

template<class entryT>
RArray<entryT>& RArray<entryT>::operator=(const RArray<entryT>& la) {
  return assign(la);
}

template<class entryT>
RArray<entryT>& RArray<entryT>::assign(const RArray<entryT>& la) {
  if (this!=&la) {
    unsigned j;
    for(j=0;j<maxelements;++j) {
      if (container[j]) {
        delete container[j];
        container[j]=NULL;
      }
    }
    if (maxelements!=la.maxelements) {
      delete[] container;
      container=new entryT*[la.maxelements];
      maxelements=la.maxelements;
    }
    nelements=la.nelements;
    sorted=la.sorted;
    unsigned i;
    for(i=0;i<maxelements;++i) {
      if (la.container[i])
        container[i]=new entryT(*(la.container[i]));
      else
        container[i]=NULL;
    }
  }
  return *this;
}

template<class entryT>
RArray<entryT>::RArray(const entryT* arr, const unsigned size, 
    const int mymaxelements) {
  assert(size>0);
  if (mymaxelements>-1) {
    assert(static_cast<unsigned>(mymaxelements)>size);
    maxelements=mymaxelements;
    container=new entryT*[maxelements];
  }
  else {
    maxelements=size;
    container=new entryT*[size];
  }
  unsigned i;
  for(i=0;i<size;++i) {
    entryT* el=new entryT(arr[i]);
    container[i]=el;
  }
  for(;i<maxelements;++i)
    container[i]=NULL;
  nelements=size;
  sorted=false;
}

template<class entryT>
void RArray<entryT>::resize() {
  unsigned oldme=maxelements;
  maxelements= maxelements==0 ? 1:maxelements*2;
  entryT** newcontainer=new entryT*[maxelements];
  unsigned i;
  for(i=0;i<oldme;++i) {
    newcontainer[i]=container[i];
  }
  for(i=oldme;i<maxelements;++i)
    newcontainer[i]=NULL;
  delete[] container;
  container=newcontainer;
}

template<class entryT>
unsigned RArray<entryT>::compact() {
  unsigned i, j=0;
  unsigned trashed=0;
  for(i=0;i<maxelements;++i) {
    if (container[i]) {
      if (i!=j) {
        container[j]=container[i];
        container[i]=NULL;
        trashed++;
      }
      j++;
    }
  }
  return trashed;
}

/*
template<class entryT>
void RArray<entryT>::addABunch(const unsigned nel,...) {
  va_list marker;
  unsigned j;

  va_start(marker,nel); // Initialize variable arguments
  for(j=0;j<nel;++j) {
    entryT el=va_arg(marker,entryT);
    entryT* newel=new entryT(el);
    assert(container[nelements+j]==NULL);
    if (nelements+j>=maxelements) {
      unsigned freed=compact();
      if (freed<0.1*maxelements)
        makeItDouble();
    }
    container[nelements+j]=newel;
  }
  va_end(marker); // Reset variable arguments
  nelements+=nel;
}
*/

template<class entryT>
void RArray<entryT>::add(const entryT& el) {
  if (nelements==maxelements) {
    if (maxelements==0)
      resize();
    else {
      if (compactIt) {
        unsigned freed=compact();
        nelements-=freed;
        if (freed<static_cast<unsigned>(ceil(0.1*maxelements)))
          resize();
      }
      else
        resize();
    }
  }
  entryT* newel=new entryT(el);
  assert(container[nelements]==NULL);
  container[nelements]=newel;
  nelements++;
}

template<class entryT>
void RArray<entryT>::addElementAt(const unsigned i, const entryT& el) {
  if (i>=maxelements) {
    do {
      resize();
    } while(i>=maxelements);
  }
  // since we are adding a new element, the position in the container must
  // be empty if we are to increment the number of elements in the array
  if (container[i]==NULL)
    nelements++;
  delete container[i];
  entryT* newel=new entryT(el);
  container[i]=newel;
}

template<class entryT>
void RArray<entryT>::sort(int (*compar)(const void *a, const void *b)) {
  if (compar==NULL) {
    std::cout << "There is no cmp function defined for " << typeid(entryT).name() << " type\n";
  }
  assert(compar!=NULL);
// hcma  qsort(static_cast<void*>(container),maxelements,sizeof(entryT*),compar);
  sorted=true;
}

template<class entryT>
void RArray<entryT>::intersection(const RArray<entryT>& ra, 
    const RArray<entryT>& rb,int (*compar)(const void *a, const void *b)) {
// finds the intersection of two lists. 
  assert(nelements==0 || maxelements==0);
  assert(ra.sorted && rb.sorted);
  unsigned ai=0;
  unsigned bi=0;
  unsigned ci=0;

  unsigned newmaxelements=IC_MIN(ra.maxelements,rb.maxelements);
  unsigned j;
  if (maxelements!=newmaxelements) {
    delete[] container;
    container=new entryT*[newmaxelements];
  }
  for(j=0;j<newmaxelements;++j) {
    if (maxelements==newmaxelements)
      delete container[j];
    container[j]=NULL;
  }
  maxelements=newmaxelements;
  while(1) {
    entryT *ael, *bel;
    ael=ra.container[ai];
    bel=rb.container[bi];
    int cmp=compar(&ael,&bel);
    if (cmp==0) {
      setElement(ci,ael);
      ci++;
      ai++;
      bi++;
    }
    else if (cmp<0)
      ai++;
    else
      bi++;
    if (ai==ra.maxelements || bi==rb.maxelements)
      break;
  }
  nelements=ci;
  sorted=true;
}

template<class entryT>
void RArray<entryT>::difference(const RArray<entryT>& ra, 
    const RArray<entryT>& rb,int (*compar)(const void *a, const void *b)) {
// finds the set difference of two lists. 
  assert(nelements==0 || maxelements==0);
  assert(ra.sorted && rb.sorted);
  unsigned ai=0;
  unsigned bi=0;
  unsigned ci=0;

  unsigned newmaxelements=ra.maxelements;
  unsigned j;
  if (maxelements!=newmaxelements) {
    delete[] container;
    container=new entryT*[newmaxelements];
  }
  for(j=0;j<newmaxelements;++j) {
    if (maxelements==newmaxelements)
      delete container[j];
    container[j]=NULL;
  }
  maxelements=newmaxelements;
  while(1) {
    entryT *ael, *bel;
    ael=ra.container[ai];
    bel=rb.container[bi];
    int cmp=compar(&ael,&bel);
    if (cmp==0) {
      ai++;
      bi++;
    }
    else if (cmp<0) {
      setElement(ci,ael);
      ci++;
      ai++;
    }
    else
      bi++;
    if (ai==ra.maxelements || bi==rb.maxelements)
      break;
  }
  if (bi==rb.maxelements) {
    unsigned i;
    for(i=ai;i<ra.maxelements;++i) {
      if (ra.container[i]) {
        setElement(ci,ra.container[i]);
        ci++;
      }
    }
  }
  nelements=ci;
  sorted=true;
}

template<class entryT>
bool RArray<entryT>::contains(const RArray<entryT>& ra, 
    int (*compar)(const void *a, const void *b)) const {
// tells whether "this" set contains set "a"
  if (ra.nelements>nelements)
    return false;
  assert(sorted && ra.sorted);
  unsigned i, j=0;
  for(i=0;i<ra.nelements;++i) {
    entryT *ael, *el;
    ael=ra.container[i];
    do {
      el=container[j];
      int cmp=compar(&ael,&el);
      if (cmp==0) {
        j++;
        break;
      }
      else if (cmp<0)
        return false;
      ++j;
      if (j==nelements)
        return false;
    } while(1);
  }
  return true;
}

template<class entryT>
bool RArray<entryT>::isEqual(const RArray<entryT>& ra, 
    int (*compar)(const void *a, const void *b)) const {
  if(nelements!=ra.nelements)
    return false;
  assert(sorted && ra.sorted);
  unsigned i;
  for(i=0;i<nelements;++i) {
    entryT *ael, *el;
    el=container[i];
    ael=ra.container[i];
    if (compar(&el,&ael)!=0)
      return false;
  }
  return true;
}

template<class entryT>
void RArray<entryT>::invert() {
  unsigned i=0;
  unsigned j=maxelements-1;
  while(i<j) {
    if (container[i] && container[j]) {
      entryT* temp=container[i];
      container[i]=container[j];
      container[j]=temp;
      i++; j--;
    }
    if (!container[i])
      i++;
    if (!container[j])
      j--;
  }
}

template<class entryT>
entryT* RArray<entryT>::getElement(const unsigned i) const {
  if (i>=maxelements)
    return NULL;
  return container[i];
}

template<class entryT>
void RArray<entryT>::setElement(const unsigned i, const entryT* el, 
  const bool deleteOld) {
  assert(i<maxelements);
  if (deleteOld) {
    delete container[i];
    container[i]=NULL;
  }
  if (el) {
    entryT* nel=new entryT(*el);
    container[i]=nel;
  }
}

template<class entryT>
entryT* RArray<entryT>::find(const entryT& myentry, 
    int (*compar)(const void *, const void *)) const {
  void* ptr1=(void *)&myentry;
  if(sorted) {
// hcma    return (entryT*)(bsearch(&ptr1,container,maxelements,sizeof(entryT*),compar));
    return NULL;
  }
  else {
    unsigned i;
    for(i=0;i<nelements;++i) {
      void* ptr2=(void *)&container[i];
      if(compar(&ptr1,ptr2)==0)
        return container[i];
    }
  }
  return NULL;
}

template<class entryT>
int RArray<entryT>::ord(const entryT& myentry, 
    int (*compar)(const void *, const void *)) const {
  void* ptr1=(void *)&myentry;
  if(sorted) {
// hcma    void* ptr=(entryT*)bsearch(&ptr1,container,maxelements,sizeof(entryT*),compar);
    void* ptr=NULL;
    if (ptr)
      return (((char*)ptr-(char*)container)/sizeof(entryT*));
  }
  else {
    unsigned i;
    for(i=0;i<nelements;++i) {
      void* ptr2=(void *)&container[i];
      if(compar(&ptr1,ptr2)==0)
        return i;
    }
  }
  return -1;
}

template<class entryT>
void RArray<entryT>::removeEntryAt(const unsigned i) {
  assert(nelements>0 && container[i]!=NULL);
  delete container[i];
  container[i]=NULL;
  nelements--;
}

template<class entryT>
void RArray<entryT>::destroy() {
  if (!container)
    return;
  unsigned i;
  for(i=0;i<maxelements;++i) {
//    std::cout << "deleting " << i << endl;
    if (container[i]) {
      delete container[i];
      container[i]=NULL;
    }
  }
  nelements=0;
}

#ifndef INSURE
template<class entryT>
void RArray<entryT>::printOn(std::ostream& o) const {
  unsigned i;
  o << nelements << " entries in the resizable array -- list of entries follow\n";
  for(i=0;i<maxelements;++i) {
    if (container[i])
      o << " [" << i << "]: " << *container[i];
    /**
    else
      o << " [" << i << "]: NULL";
    else
    ***/
    /*
    if (container[i]!=NULL)
      o << *container[i] << " ";
    */
  }
}
#endif

template<class entryT>
RArray<entryT>::~RArray() {
  destroy();
  delete[] container;
  container=NULL;
}
