// This code shows data being exchanged between a Fortran90 program and a C++
// program using P++

// Note that P++ uses column-major for representing the data, but uses the
// C-style indexing schema for the array elements. That is a(i,j) in Fortran
// is actually a(j,i) in C++/P++
//
// Auth: Henrique Andrade (hcma@cs.umd.edu)
//       University of Maryland, College Park
// Date: 01/14/2004

#include <iostream> 
#include <iomanip> 
#include <stdlib.h>
// using P++
#include <A++.h>
// using InterComm high-level library
#define PPLUSPLUS
#include "../../src/p++/IC_EndPoint.h"

#define PRINT_1D_ARRAY(ARRAY) \
{ \
  int i; \
  assert(ARRAY.numberOfDimensions()==1); \
  for(i=ARRAY.getBase(0);i<=ARRAY.getBound(0);i++) { \
    std::cout << setiosflags(std::ios::fixed)<< std::setw(6) << std::setprecision(1) << ARRAY(i) << " "; \
  } \
  std::cout << endl; \
}

#define PRINT_2D_ARRAY(ARRAY) \
{ \
  int i, j; \
  assert(ARRAY.numberOfDimensions()==2); \
  for(i=ARRAY.getBase(0);i<=ARRAY.getBound(0);i++) { \
    for(j=ARRAY.getBase(1);j<=ARRAY.getBound(1);j++) { \
      std::cout << setiosflags(std::ios::fixed)<< setw(6) << setprecision(1) << ARRAY(i,j) << " "; \
    } \
    std::cout << endl; \
  } \
}

#define PRINT_3D_ARRAY(ARRAY) \
{ \
  int i, j, k; \
  assert(ARRAY.numberOfDimensions()==3); \
  for(i=ARRAY.getBase(0);i<=ARRAY.getBound(0);i++) { \
    for(j=ARRAY.getBase(1);j<=ARRAY.getBound(1);j++) { \
      for(k=ARRAY.getBase(2);k<=ARRAY.getBound(2);k++) { \
        std::cout << setiosflags(std::ios::fixed)<< setw(6) << setprecision(1) << ARRAY(i,j,k) << " "; \
      } \
      std::cout << endl; \
    } \
    std::cout << endl; \
  } \
}

int main(int argc, char **argv) {
  int iNumProcs=1;
  // initializing P++ virtual machine
  Optimization_Manager::Initialize_Virtual_Machine("",iNumProcs,argc,argv);

  int ic_err;
  // creating communication endpoint
  IC_EndPoint right_left_ep("right:left",iNumProcs,1,IC_EndPoint::IC_COLUMN_MAJOR,IC_EndPoint::IC_COLUMN_MAJOR,ic_err);
  // testing if the endpoint has been properly created
  if (ic_err!=IC_EndPoint::IC_OK) {
    IC_EndPoint::printErrorMessage("ic_endpointconstructor",ic_err);
    return ic_err;
  }

  // array declarations
  intArray INTEGERS(10);
  doubleArray DOUBLES(10,9);
  floatArray FLOATS(3,11,10);

  // array range declarations
  Index I(3,6), J(1,4), K(1,2);

  // array initialization operations
  INTEGERS=33;
  FLOATS=44;

  unsigned i, j;
  for(i=0;i<10;i++) {
    for(j=0;j<9;j++) {
      DOUBLES(i,j)=100+(10*i)+j;
    }
  }

  // playing with 1D array of integers
  std::cout << "before receiving data:" << endl;
  PRINT_1D_ARRAY(INTEGERS);
  // importing array section from the "left" side program
  right_left_ep.importArray(INTEGERS(I),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
  std::cout << "after receiving data:" << endl;
  PRINT_1D_ARRAY(INTEGERS);

  // playing with 2D array of doubles
  std::cout << "before receiving data:" << endl;
  PRINT_2D_ARRAY(DOUBLES);

  // importing array section from the "left" side program
  right_left_ep.importArray(DOUBLES(J,I),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  std::cout << "after receiving data:" << endl;
  PRINT_2D_ARRAY(DOUBLES);

  DOUBLES(J,I)=10;
  std::cout << "before exporting data:" << endl;
  PRINT_2D_ARRAY(DOUBLES);
  // exporting array section to the "left" side program
  right_left_ep.exportArray(DOUBLES(J,I),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  // playing with 3D array of floats
  std::cout << "before receiving data:" << endl;
  PRINT_3D_ARRAY(FLOATS);

  // importing array section from the "left" side program
  right_left_ep.importArray(FLOATS(K,J,I),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  std::cout << "after receiving data:" << endl;
  PRINT_3D_ARRAY(FLOATS);

  FLOATS=55;
  std::cout << "before exporting data:" << endl;
  PRINT_3D_ARRAY(FLOATS(K,J,I));
  // exporting array section to the "left" side program
  right_left_ep.exportArray(FLOATS(K,J,I),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  Optimization_Manager::Exit_Virtual_Machine();
  return 0;
}
