/* 
 * File: communication.c
 * Auth: Jae-Yong Lee (jylee@cs.umd.edu)
 *       University of Maryland, College Park
 * Desc: functions for communication.
 *       Some functions from Meta-Chaos.
 * Date: Feb. 2004
 */

#include "communication.h"
#include "util.h"
#include "fort_wrap.h"
#include "error.h"
#include <stdlib.h>
#include <stdio.h>
#include <pvm3.h>
#include <string.h>


void SendSched(pgme_s *pgme, int to, sched_s *sched, int tag) 
{
  int proc;
  OneSched_s *Osched;
  
  pvm_initsend(GlueEncoding);
  
  pvm_pkint(&(sched->nproc),1,1);
  
  for (proc=0;proc<sched->nproc;proc++)
    {      
      pvm_pkint(&(sched->size[proc]),1,1);
      Osched=sched->sched[proc];
      while (Osched!=NULL)
	{
	  pvm_pkint(&(Osched->offset),1,1);
	  pvm_pkint(&(Osched->size),1,1);
	  Osched=Osched->next;
	}
    }
  pvm_send(TidPgme(pgme,to),tag);      
}

void SendSchedTab(pgme_s *pgme, int to, sched_s** tab_sched, int tag) 
{
  int proc;
  OneSched_s *Osched;
  int num_nodes;
  int i;
  sched_s* sched;
  
  pvm_initsend(GlueEncoding);
  num_nodes = NumNodePgme(pgme);
  pvm_pkint(&num_nodes,1,1);
  for(i=0; i<num_nodes; i++)
    {
      sched = tab_sched[i];
      pvm_pkint(&(sched->nproc),1,1);
      
      for (proc=0;proc<sched->nproc;proc++)
	{ 
	  pvm_pkint(&(sched->size[proc]),1,1);
	  
	  Osched=sched->sched[proc];
	  while (Osched!=NULL)
	    {
	      pvm_pkint(&(Osched->offset),1,1);
	      pvm_pkint(&(Osched->size),1,1);
	      Osched=Osched->next;
	    }
	}
    }
  pvm_send(to,tag);       
}

sched_s *RecvSched(pgme_s *computesched_pgme, sched_s *sched, int tag) 
{ 
  int i,j,proc;
  int MsgSizeInt,offset,size;
  int nproc;
  
  for (proc=0;proc<NumNodePgme(computesched_pgme);proc++)
    {  
      pvm_recv(TidPgme(computesched_pgme,proc),tag);
      pvm_upkint(&nproc,1,1);
      
      if (sched==NULL)
	sched=CreateSched(nproc);
      
      for (i=0;i<nproc;i++)
	{
	  pvm_upkint(&MsgSizeInt,1,1);
	  for (j=0;j<MsgSizeInt;j++)
	    {
	      pvm_upkint(&offset,1,1);
	      pvm_upkint(&size,1,1);
	      AddSched(sched,i,offset,size);
	    }
	}
    }
  return(sched);
}

sched_s *RecvScheds(pgme_s *computesched_pgme1, 
		    pgme_s *computesched_pgme2, 
		    sched_s *sched, 
		    int tag) 
{ 
  int i,j,proc;
  int MsgSizeInt,offset,size;
  int nproc;
  
  for (proc=0;proc<NumNodePgme(computesched_pgme1);proc++)
    {  
      pvm_recv(TidPgme(computesched_pgme1,proc),tag);
      pvm_upkint(&nproc,1,1);
      
      if (sched==NULL)
	sched=CreateSched(nproc);
      
      for (i=0;i<nproc;i++)
	{
	  pvm_upkint(&MsgSizeInt,1,1);
	  for (j=0;j<MsgSizeInt;j++)
	    {pvm_upkint(&offset,1,1);
	    pvm_upkint(&size,1,1);
	    AddSched(sched,i,offset,size);
	    }
	}
    }
  for (proc=0;proc<NumNodePgme(computesched_pgme2);proc++)
    {  
      pvm_recv(TidPgme(computesched_pgme2,proc),tag);
      pvm_upkint(&nproc,1,1);
      
      if (sched==NULL)
	sched=CreateSched(nproc);
      
      for (i=0;i<nproc;i++)
	{
	  pvm_upkint(&MsgSizeInt,1,1);
	  for (j=0;j<MsgSizeInt;j++)
	    {
	      pvm_upkint(&offset,1,1);
	      pvm_upkint(&size,1,1);
	      AddSched(sched,i,offset,size);
	    }
	}
    }
  
  return(sched);
}

sched_s *RecvSchedReverseOrder(pgme_s *computesched_pgme, sched_s *sched, int tag) 
{ 
  int i,j,proc;
  int MsgSizeInt,offset,size;
  int nproc;
  
  for (proc=0;proc<NumNodePgme(computesched_pgme);proc++)
    {  
      pvm_recv(TidPgme(computesched_pgme,proc),tag);
      pvm_upkint(&nproc,1,1);
      
      if (sched==NULL)
        sched=CreateSched(nproc);
      
      for (i=0;i<nproc;i++)
        {
          pvm_upkint(&MsgSizeInt,1,1);
          for (j=0;j<MsgSizeInt;j++)
            {
	      pvm_upkint(&offset,1,1);
	      pvm_upkint(&size,1,1);
	      AddSchedLast(sched,i,offset,size);
	    }
        }
    }
  return(sched);
}

sched_s** RecvSchedTab(pgme_s *computesched_pgme, int from, sched_s** tab_sched, int tag) 
{ 
  int i,j,proc;
  int MsgSizeInt,offset,size;
  int nproc;
  int num_nodes;
  int nodes;
  sched_s* sched;

  pvm_recv(TidPgme(computesched_pgme,from),tag);

  pvm_upkint(&num_nodes,1,1);
  
  tab_sched = (sched_s**)malloc(num_nodes*sizeof(sched_s*));
  
  for (nodes=0;nodes<num_nodes;nodes++)
    { 
      sched = 0; 
      pvm_upkint(&nproc,1,1);
      
      if (sched==NULL)
	sched=CreateSched(nproc);
      
      for (i=0;i<nproc;i++)
	{
	  pvm_upkint(&MsgSizeInt,1,1);
	  for (j=0;j<MsgSizeInt;j++)
	    {
	      pvm_upkint(&offset,1,1);
	      pvm_upkint(&size,1,1);
	      AddSchedLast(sched,i,offset,size);
	    }
	}
      tab_sched[nodes] = sched;
    }
  return(tab_sched);
}

int exch_desctype(pgme_s *other_pgme, int type)
{
  int i;
  int tmp;

  if (MyPosMyPgme() == 0) {
    pvm_initsend(GlueEncoding);
    pvm_pkint(&type, 1, 1);
    for (i = 0; i < NumNodePgme(other_pgme); ++i) {
      pvm_send(TidPgme(other_pgme, i), FLAG_Broadcast2);
    }
  }
  pvm_recv(TidPgme(other_pgme, 0), FLAG_Broadcast2);
  pvm_upkint(&tmp, 1, 1);
  return tmp;
}

int exch_size_regions(pgme_s *other_pgme, int num_data)
{
  int i;
  int tmp;

  if (MyPosMyPgme() == 0) {
    pvm_initsend(GlueEncoding);
    pvm_pkint(&num_data, 1, 1);
    for (i = 0; i < NumNodePgme(other_pgme); ++i) {
      pvm_send(TidPgme(other_pgme, i), FLAG_Broadcast2);
    }
  }
  pvm_recv(TidPgme(other_pgme, 0), FLAG_Broadcast2);
  pvm_upkint(&tmp, 1, 1);
  return tmp;
}

void exch_transfer_info(pgme_s *other_pgme, int type,int num_data, int* other_type, int* other_num_data)
{
  int i;

  if (MyPosMyPgme() == 0) {
    pvm_initsend(GlueEncoding);
    pvm_pkint(&type, 1, 1);
    pvm_pkint(&num_data, 1, 1);
    for (i = 0; i < NumNodePgme(other_pgme); ++i) {
      pvm_send(TidPgme(other_pgme, i), FLAG_Broadcast2);
    }
  }
  pvm_recv(TidPgme(other_pgme, 0), FLAG_Broadcast2);
  pvm_upkint(other_type, 1, 1);
  pvm_upkint(other_num_data, 1, 1);
} 


/* Free info_dd_arSet data structure */
void Free_info_dd_arSet(info_dd_arSet* info) {
  int i, j;
  
  if (info) {
    for (i = 0; i < info->ar_Set->NumRegion; ++i)
      Free_Region(info->ar_Set->regions[i]);
    Free_setOfRegion(info->ar_Set);
    for (i = 0; i < info->numproc; i++) {
      for (j = 0; j < info->num_of_blocks_others[i]; ++j) {
	Free_Block(info->block_array[i][j]);
      }
      free(info->block_array[i]);    
    }
    free(info->block_array);
    free(info->num_of_blocks_others);
    free(info);
  }
}
/* Send a compact descriptor and a set of regions to other program */
void send_compact_DAD_SetOfRegions(HowToSendDAD howsend,
				   int me,
				   setOfRegion_s* ar_Set,
				   void* datadescriptor,
				   pgme_s* my_pgme, 
				   pgme_s* other_pgme)
{
  int num_blocks;
  int nDims;
  
  block** block_array;
  
  int* temp_buff= NULL;
  int temp_count;
  
  num_blocks = get_num_of_blocks(((decomp_Birreg*)datadescriptor));
  nDims = ((decomp_Birreg*)datadescriptor)->nDims;
  
  block_array = (block**)malloc(num_blocks * sizeof(block*));
  store_block_array_ptr(((decomp_Birreg*)datadescriptor),block_array);
   
  if((howsend == SendAllToAll) || ((howsend == SendOneToOne) &&(me == 0))){   
    temp_buff = build_info(nDims,ar_Set, num_blocks, block_array,   ((decomp_Birreg*)( datadescriptor))->col_majeur, &temp_count);
  } 
  
  if(howsend == SendAllToAll){
    send_DD_arSet(my_pgme,other_pgme,temp_buff,temp_count);
  }  
  else if((howsend == SendOneToOne) && (me == 0))
    {
      send_DD_arSet_One2One(my_pgme,other_pgme,temp_buff,temp_count);
    }

  if(temp_buff)
    free(temp_buff);
  free(block_array);
}

/* Recv a compact descriptor and a set of regions to other program */
info_dd_arSet* receive_compact_DAD_SetOfRegions(HowToSendDAD howsend,
						int me,
						pgme_s* my_pgme, 
						pgme_s* other_pgme) 
{
  int i;
  int* buff;
  info_dd_arSet* set;
  
  if (howsend == SendAllToAll) {
    buff = receive_DD_arSet(cal_source_node(my_pgme,other_pgme));
  }
  else if (howsend == SendOneToOne) {
    if (me == 0) {
      buff = receive_DD_arSet(TidPgme(other_pgme,0));
      
      pvm_initsend(GlueEncoding);
      pvm_pkint(buff, buff[0], 1);
      
      for (i = 1; i < NumNodePgme(my_pgme); i++) {
	pvm_send(TidPgme(my_pgme,i), FLAG_DAD);
      }
      
      /* receive info from the node 0 in other side */
      /* send the received info to nodes in my side */
    }
    else {
      /* receive the info from the node 0 in my side */      
      buff = receive_DD_arSet(TidPgme(my_pgme,0));
    }
  }
  
  set = interpret_info(buff, my_pgme, other_pgme);
  
  free(buff);

  return set;
}

/* 
 * Send translation table info to processes in my program 
 * according to the responsible range
 */

void  send_ttable_info(pgme_s* my_pgme,
		       simple_ttable* ttable,
		       setOfRegion_s* my_ar_Set,
		       int overall_size)
{
  int i,j;
  int low, high;
  int global_index;
  int count,index,LZ_index,dest_proc;
  int* data;
  
  int discr;
  int max_size;
  
  low = ((simple_ttable*)ttable)->global_low_bound;
  high = ((simple_ttable*)ttable)->global_high_bound;
  
  discr = overall_size/NumNodePgme(my_pgme);
  max_size = overall_size/NumNodePgme(my_pgme) +NumNodePgme(my_pgme) -1;
  
  /* 1 for size, 4 for a tuple(LZ_index,global,proc_num,local) */
  data = (int*)malloc((1+4*max_size)*sizeof(int));

  index = 1;
  LZ_index = 0;
  count = 0;
  dest_proc = 0;
  
  if(discr==0) {  /* only last processor is responsible */
    
    for(i=0; i< NumNodePgme(my_pgme)-1;i++)
      {
	data[0] =0;
	pvm_initsend(GlueEncoding);
	pvm_pkint(data,1,1);
	
	pvm_send(TidPgme(my_pgme,i),FLAG_TTABLE); 
      }
    dest_proc = NumNodePgme(my_pgme)-1;
    
  }
  
  for(i=0; i< my_ar_Set->NumRegion; i++)
    {
      for(j=0; j < ((region_enum_s*)(my_ar_Set->regions[i]))->size; j++)
	{
	  global_index = ((region_enum_s*)(my_ar_Set->regions[i]))->globalCoord[j];
	  if(global_index >= low   && global_index <= high)
	    {
	      count++;
	      data[index++] = LZ_index;
	      data[index++] = global_index;
	      data[index++] = ((simple_ttable*)ttable)->proc_num[global_index-low];
	      data[index++] = ((simple_ttable*)ttable)->local_offset[global_index-low];
	    }
	  LZ_index++;
	  if( ((dest_proc != NumNodePgme(my_pgme)-1)&&(LZ_index%discr == 0)) 
	      || ( (LZ_index == overall_size) &&( dest_proc == NumNodePgme(my_pgme)-1))
	      )
	    {
	      /* send the data to dest_proc and initialize some indexes */
	      
	      data[0] = count;
	      
	      pvm_initsend(GlueEncoding);
	      pvm_pkint(data,1+count*4,1);
	      
	      pvm_send(TidPgme(my_pgme,dest_proc),FLAG_TTABLE);
	      
	      index = 1;
	      count = 0;
	      dest_proc++;
	    }
	}
    }
  free(data);
}

/* 
 * Recv translation table info from processes in my program 
 * according to the responsible range
 */

simple_ttable* receive_ttable_info(pgme_s* my_pgme, int overall_size)
{
  simple_ttable* table = NULL;
  int** data;
  int i,j;
  int* size;
  int* index;
  int from,to;
  int me;
  int pos;
  int found;
  int nData;

  me = MyPosMyPgme();
  
  data =(int**)malloc(sizeof(int*)*NumNodePgme(my_pgme));
  size = (int*)malloc(sizeof(int)*NumNodePgme(my_pgme));
  
  index =  (int*)malloc(sizeof(int)*NumNodePgme(my_pgme));
  
  for(i=0; i< NumNodePgme(my_pgme); i++)
    {
      pvm_recv(TidPgme(my_pgme,i),FLAG_TTABLE);
      pvm_upkint(&size[i],1,1);
      
      data[i] = (int*)malloc(sizeof(int)*size[i]*4);
      pvm_upkint(data[i],4*size[i],1);
    }
  
  /* build a trans table by simple merging .. to be improved later.. */
  
  nData = 0;
  
  for(i=0;i<NumNodePgme(my_pgme); i++){
    nData += size[i];
    index[i] =0;
  }
  
  table = Allocate_Simple_TTABLE(nData);
  
  from =  me*(overall_size/NumNodePgme(my_pgme));
  if(me==NumNodePgme(my_pgme)-1){
    to = overall_size-1;
  }
  else 
    to =  (me+1)*(overall_size/NumNodePgme(my_pgme)) -1;

  if((to-from+1)!= table->nData) {
    errormsg("Error in calculating the number of data to be received.");
  }

  /* simple merge.. will be improved later.. */
  pos = 0;
  
  for(i = from; i <= to; i++)
    {
      found = 0;
      for(j = 0; j< NumNodePgme(my_pgme); j++)
	{
	  if( data[j][index[j]] == i){
	    /* copy info into a new transtable. */
	    table->global_index[pos] = data[j][index[j]+1];
	    table->proc_num[pos] = data[j][index[j]+2];
	    table->local_offset[pos] = data[j][index[j]+3];
	    index[j] += 4;
	    pos++;
	    found = 1;
	    break;
	  }
	}
    }
  
  /* deallocate memory */
  for(i=0; i< NumNodePgme(my_pgme); i++)
    {
      if(size[i] !=0) /* don't necessary but make it sure */
	free(data[i]);
    }
  
  free(data);
  free(size);
  free(index);
  
  return table;
}

/* 
 * Send translation table info to processes in two programs(mine,other) 
 * according to the responsible range
 */

void  send_ttable_info_two_pgmes(pgme_s* first_pgme, pgme_s* second_pgme, simple_ttable* ttable, setOfRegion_s* my_ar_Set, int overall_size, pgme_s* my_pgme)
{
  int i, j;
  int low, high;
  int global_index;
  int count, index, LZ_index, dest_proc;
  int** data;
  int num_proc_first, num_proc_second, total_num_proc;
  int dest2;  
  int discr;
  int max_size;
  int me;
  int rank;
  int offset_rank;
  pgme_s* dest_pgme;
  int ii;
  
  me = MyPosMyPgme();
  if (first_pgme == my_pgme)
    rank = me;
  else
    rank = NumNodePgme(first_pgme) + me;
  
  low = ((simple_ttable*)ttable)->global_low_bound;
  high = ((simple_ttable*)ttable)->global_high_bound;
  
  num_proc_first = NumNodePgme(first_pgme);
  num_proc_second = NumNodePgme(second_pgme);
  total_num_proc = num_proc_first + num_proc_second;
  
  discr = overall_size/total_num_proc;
  max_size = overall_size/total_num_proc + total_num_proc - 1;
  
  data = (int**)malloc(total_num_proc*sizeof(int*));
  for (i = 0; i < total_num_proc; i++) {
    /* 1 for size, 4 for a tuple (LZ_index,global,proc_num,local) */
    data[i] = (int*)malloc((1+4*max_size)*sizeof(int));
    memset(data[i], 0, (1+4*max_size)*sizeof(int));
  }
  
  index = 1;
  LZ_index = 0;
  count = 0;
  dest_proc = 0;

  if (discr == 0) {
    /* only last processor is responsible */    
    dest_proc = total_num_proc - 1;
  }
  
  for (i = 0; i < my_ar_Set->NumRegion; i++) {
    for (j = 0; j < ((region_enum_s*)(my_ar_Set->regions[i]))->size; j++) {
      global_index = ((region_enum_s*)(my_ar_Set->regions[i]))->globalCoord[j];
      if (global_index >= low && global_index <= high) {
	count++;
	data[dest_proc][index++] = LZ_index;
	data[dest_proc][index++] = global_index;
	data[dest_proc][index++] = ((simple_ttable*)ttable)->proc_num[global_index-low];
	data[dest_proc][index++] = ((simple_ttable*)ttable)->local_offset[global_index-low];
      }
      LZ_index++;
      if (((dest_proc != total_num_proc - 1) && (LZ_index%discr == 0)) ||
	 ((LZ_index == overall_size) && (dest_proc == total_num_proc - 1))) {
	/* send the data to dest_proc and initialize some indexes */
	data[dest_proc][0] = count;
	index = 1;
	count = 0;
	dest_proc++;
      }
    }
  }
  
  i = rank;
  for (ii = 0; ii < total_num_proc; ii++) {    
    if (i < num_proc_first) { 
      dest_pgme = first_pgme;
      offset_rank = 0;
    }
    else {
      dest_pgme = second_pgme;
      offset_rank = num_proc_first;
    }
    
    pvm_initsend(GlueEncoding);
    pvm_pkint(data[i], 1+data[i][0]*4, 1);
    pvm_send(TidPgme(dest_pgme, i-offset_rank), FLAG_TTABLE);
    
    i = (i+1)%total_num_proc; 
  }
  
  for (i = 0; i < total_num_proc; i++)
    free(data[i]);

  free(data);
}

/* 
 * Recv translation table info to processes in a program 
 * according to the responsible range. Very similar to receive_ttable_info.
 * But this function is called when both programs compute schedules.
 */

simple_ttable* receive_ttable_info_two_pgmes(int incr, pgme_s* my_pgme, pgme_s* other_pgme, int overall_size, int my_other)
{
  simple_ttable* table = NULL;
  int** data;
  int i,j;
  int num;
  int* size;
  int* index;
  int from,to;
  int me;
  int pos;
  int found;
  int nData;
  int ii;
  
  me = MyPosMyPgme();
  
  data = (int**)malloc(sizeof(int*)*NumNodePgme(my_pgme));
  size = (int*)malloc(sizeof(int)*NumNodePgme(my_pgme));  
  memset(size, 0, sizeof(int)*NumNodePgme(my_pgme));
  index = (int*)malloc(sizeof(int)*NumNodePgme(my_pgme));
  
  if (my_other == 0)
    ii = me;
  else
    ii = NumNodePgme(my_pgme) - 1;
  
  for (i = 0; i < NumNodePgme(my_pgme); i++) {
    pvm_recv(TidPgme(my_pgme,ii), FLAG_TTABLE);
    pvm_upkint(&size[ii], 1, 1);    
    data[ii] = (int*)malloc(sizeof(int)*size[ii]*4);
    pvm_upkint(data[ii], 4*size[ii], 1);
    ii--;
    if (ii < 0) 
      ii = NumNodePgme(my_pgme) - 1;
  }
  
  /* build a tran table by simple merging .. to be improved later.. */  
  
  nData = 0;  
  for (i = 0; i < NumNodePgme(my_pgme); i++) {
    nData += size[i];
    index[i] = 0;
  }
  
  table = Allocate_Simple_TTABLE(nData);
  
  from = (me+incr)*(overall_size/(NumNodePgme(my_pgme) + NumNodePgme(other_pgme)));
  if ((me+incr) == NumNodePgme(my_pgme) + NumNodePgme(other_pgme) - 1)
    to = overall_size - 1;
  else 
    to = (me+incr+1)*(overall_size/(NumNodePgme(my_pgme) + NumNodePgme(other_pgme))) - 1;
  
  if ((to-from+1) != table->nData) {
    errormsg("Error in calculating the number of data to be received.");
    return 0;
  }
  
  /* simple merge.. will be improved later.. */
  pos = 0;
  
  for (i = from; i <= to; i++) {
    found = 0;
    for (j = 0; j < NumNodePgme(my_pgme); j++) {
      if ((data[j][index[j]] == i) && index[j] < size[j]*4) {
	/* copy info into a new transtable. */
	table->global_index[pos] = data[j][index[j]+1];
	table->proc_num[pos] = data[j][index[j]+2];
	if (data[j][index[j]+2] >= NumNodePgme(my_pgme)) {
	  errormsg("Table assigning data to nonexistent task.");
	  return 0;
	}
	table->local_offset[pos] = data[j][index[j]+3];
	index[j] += 4;
	pos++;
	found = 1;
	break;
      }
    }
  }
  
  /* deallocate memory */
  for (i = 0; i < NumNodePgme(my_pgme); i++) {
    free(data[i]);
  }
  
  free(data);
  free(size);
  free(index);
  
  return table;
}


/* 
 * Flatten info about data blocks and a set of regions.
 * interpret_info function must know how this function
 * pack the data.
 */

int* build_info(int nDims,  
		setOfRegion_s* ar_Set,
		int num_blocks,
		block** block_array,
		int mem_layout,
		int* temp_count)
{
  int i,j;
  int me;
  int* temp_buff;
  int count;
  
  int numproc,other_numproc;
  
  int numRegions;
  
  numRegions = ar_Set->NumRegion;

  /* Decide the buffer size */

  if(numRegions >0){
    temp_buff = (int*)malloc(  ( ( num_blocks*(4*nDims+1)+2) +(1+3*numRegions*(((region_hpf_s*)(ar_Set->regions[0]))->nbDim ) + numRegions + (numRegions+1) ) +1 +1) * sizeof(int));
    *temp_count =  (num_blocks*(4*nDims+1)+2) +(1+3*numRegions*(((region_hpf_s*)(ar_Set->regions[0]))->nbDim ) + numRegions +(numRegions+1))+1 +1;
  }
  else{ 			   
    temp_buff = (int*)malloc(( (( num_blocks*(4*nDims+1)+2    ) +(1 )+1 +1) * sizeof(int)));
    *temp_count = (num_blocks*(4*nDims+1)+2 ) +(1 )+1 +1;
  }

  /* Copy values */
  count = 0;
  temp_buff[count++] = *temp_count;
  temp_buff[count++] = mem_layout;
  temp_buff[count++] = num_blocks;
  temp_buff[count++] = nDims;
  
  for(i = 0 ; i < num_blocks; i++)
    {
      temp_buff[count++] = block_array[i]->proc;
      
      for(j =0; j< nDims; j++){
	temp_buff[count++] =  block_array[i]->size[j];
	temp_buff[count++] = block_array[i]->start_global[j];
	temp_buff[count++] = block_array[i]->end_global[j];
	temp_buff[count++] = block_array[i]->coord[j];
      }
    }
  
  temp_buff[count++] = numRegions;
  
  for(i = 0; i < numRegions ; i++)
    {
      temp_buff[count++] = ((region_hpf_s*)(ar_Set->regions[i]))->nbDim ;
      for(j =0; j <((region_hpf_s*)(ar_Set->regions[i]))->nbDim; j++)
	{
	  temp_buff[count++] = ((region_hpf_s*)(ar_Set->regions[i]))->left[j] ;
	  temp_buff[count++] =  ((region_hpf_s*)(ar_Set->regions[i]))->right[j] ;
	  temp_buff[count++] =  ((region_hpf_s*)(ar_Set->regions[i]))->stride[j] ;
	} 
    }
  
  for(i = 0; i <= numRegions ; i++)
    {
      temp_buff[count++] = ar_Set->BeginGlobalOffset[i];
    }
  return temp_buff;
}

/* 
 *  Send a compact descripor and a set of regions to a disjoint set 
 *  of processes of other program. 
 */
void send_DD_arSet(pgme_s* my_pgme, pgme_s* other_pgme,int* temp_buff,int temp_count)
{
  int i;
  int me;
  
  int numproc,other_numproc;
  
  pvm_initsend (GlueEncoding);
  pvm_pkint(temp_buff,temp_count,1);
  
  numproc = NumNodePgme(my_pgme);
  
  other_numproc = NumNodePgme(other_pgme);
  
  me = MyPosMyPgme();
  
  for(i = me*( (other_numproc -1)/(numproc)+1) ; i < (me+1)*( (other_numproc -1)/(numproc)+1) && i < other_numproc; i++)
    {
      pvm_send(TidPgme(other_pgme,i),FLAG_DAD);
    }
}

/* Send the info to a representative process(process 0) */
void send_DD_arSet_One2One(pgme_s* my_pgme, pgme_s* other_pgme,int* temp_buff,int temp_count)
{
  pvm_initsend (GlueEncoding);
  pvm_pkint(temp_buff,temp_count,1);
  pvm_send(TidPgme(other_pgme,0),FLAG_DAD);
}

/* Recv the info from one process */
int* receive_DD_arSet(int from)
{
  int num_elements;
  int* buff;

  pvm_recv(from,FLAG_DAD); 
  
  pvm_upkint(&num_elements,1,1); 
  
  buff=(int*)malloc(num_elements*sizeof(int)); 
  buff[0] = num_elements; 
  
  pvm_upkint(buff+1,num_elements-1,1); 
  
  return buff;
}

/* 
 * Interpret the received info and build a compact descriptor 
 * and a set of compact(HPF) regions.
 * Must understand how build info function packs the info.
 */

info_dd_arSet* interpret_info(int* buffer, pgme_s* my_pgme, pgme_s* other_pgme)
{
  int num_elements;
  int other_numBlocks;
  int other_nDims;
  int nDims;
  int counter;
  int* da_info;
  info_dd_arSet* res;
  int numRegions_other;
  int dims_of_regions;
  int count;
  int index;
  int i, j;
  int* pos;
  int* left;
  int* right;
  int* stride;
  int numproc;
  int other_numproc;
  int procnum;
  region_s* r;
  int mem_layout;

  other_numproc = NumNodePgme(other_pgme);

  count = 0;	
 
  num_elements = buffer[count++];
  mem_layout = buffer[count++];
  other_numBlocks = buffer[count++];
  other_nDims = buffer[count++];
  
  res = (info_dd_arSet*)malloc(sizeof(info_dd_arSet));
  
  res->numproc = NumNodePgme(other_pgme);
  res->mem_layout = mem_layout;
  da_info = (int*)malloc(other_numBlocks*(4*other_nDims+1)*sizeof(int));	
  for (i = 0; i < other_numBlocks*(4*other_nDims+1); i++)
    da_info[i] = buffer[count++];
  
  res->ar_Set = Alloc_setOfRegion();
  
  numRegions_other = buffer[count++];	
  
  for (i = 0; i < numRegions_other; i++) { 
    dims_of_regions = buffer[count++];
    
    left = (int*)malloc(dims_of_regions*sizeof(int));
    right = (int*)malloc(dims_of_regions*sizeof(int));
    stride = (int*)malloc(dims_of_regions*sizeof(int));
    
    for (j = 0 ; j < dims_of_regions; j++) {
      left[j] = buffer[count++];
      right[j] = buffer[count++];
      stride[j] = buffer[count++];
    }
    
    r = Alloc_R_HPF(dims_of_regions, left, right, stride, R_replicated, Mem_Copy);
    res->ar_Set = Add_Region_setOfRegion(r, res->ar_Set);
    
    free(left);
    free(right);
    free(stride);
  }
  
  for(i = 0; i <= numRegions_other; i++) {
    res->ar_Set->BeginGlobalOffset[i] = buffer[count++];
  }
  
  res->block_array = (block***)malloc(other_numproc*sizeof(block**));
  
  res->num_of_blocks_others = (int*)malloc(other_numproc*sizeof(int));
  
  pos = (int*)malloc(other_numproc*sizeof(int));
  
  count = 0;
  
  for (i = 0; i < other_numproc; i++)
    res->num_of_blocks_others[i] = 0;
  
  for (i = 0; i < other_numBlocks; i++) {
    res->num_of_blocks_others[da_info[count]]++;
    count += 1+4*other_nDims;
  }
  
  for (i = 0; i < other_numproc; i++) {
    res->block_array[i] = (block**)malloc(res->num_of_blocks_others[i]*sizeof(block*));
    pos[i] = 0;
    
    for (j = 0; j < res->num_of_blocks_others[i]; j++) {
      res->block_array[i][j] = Allocate_Block(other_nDims);
    }
  }
  
  count = 0;
  index = 0;
  
  for (i = 0; i < other_numBlocks; i++) {    
    procnum = da_info[count++];
    if (procnum >= NumNodePgme(other_pgme)) {
      errormsg("Decomposition assigning data to nonexistent task.");
      return 0;
    }

    index = pos[procnum];
    res->block_array[procnum][index]->proc = procnum;
    
    for(j = 0; j < other_nDims; j++) {
      res->block_array[procnum][index]->size[j] = da_info[count++];
      res->block_array[procnum][index]->start_global[j] = da_info[count++];
      res->block_array[procnum][index]->end_global[j] = da_info[count++];
      res->block_array[procnum][index]->coord[j] = da_info[count++];
    }
    pos[procnum] += 1;
  }
  
  free(da_info);
  free(pos);
  
  return res;
}

/* to know who sends me the info */
int cal_source_node(pgme_s* my_pgme, pgme_s* other_pgme)
{
  int numproc;
  int other_numproc;
  int me;
  int who;
  
  numproc = NumNodePgme(my_pgme);
  other_numproc = NumNodePgme(other_pgme);
  
  me = MyPosMyPgme();

  who = me/((numproc-1)/other_numproc+1);
 
  return (TidPgme(other_pgme,who));
}
