#include "fort_wrap.h"
#include "intercomm.h"
#include "ic_config.h"
#include "error.h"
#include <stdlib.h>
#include <string.h>

#define IC_INIT_F77 F77_FUNC_(ic_init, IC_INIT)
#define IC_WAIT_F77 F77_FUNC_(ic_wait, IC_WAIT)
#define IC_SYNC_F77 F77_FUNC_(ic_sync, IC_SYNC)

#define IC_CREATE_BDECOMP_DESC_F77 F77_FUNC_(ic_create_bdecomp_desc, IC_CREATE_BDECOMP_DESC)
#define IC_CREATE_TTABLE_DESC_F77  F77_FUNC_(ic_create_ttable_desc, IC_CREATE_TTABLE_DESC)
#define IC_CREATE_ENUM_REGION_F77  F77_FUNC_(ic_create_enum_region, IC_CREATE_ENUM_REGION)
#define IC_CREATE_BLOCK_REGION_F77 F77_FUNC_(ic_create_block_region, IC_CREATE_BLOCK_REGION)
#define IC_COMPUTE_SCHEDULE_F77    F77_FUNC_(ic_compute_schedule, IC_COMPUTE_SCHEDULE)

#define IC_SEND_CHAR_F77   F77_FUNC_(ic_send_char, IC_SEND_CHAR)
#define IC_SEND_SHORT_F77  F77_FUNC_(ic_send_short, IC_SEND_SHORT)
#define IC_SEND_INT_F77    F77_FUNC_(ic_send_int, IC_SEND_INT)
#define IC_SEND_FLOAT_F77  F77_FUNC_(ic_send_float, IC_SEND_FLOAT)
#define IC_SEND_DOUBLE_F77 F77_FUNC_(ic_send_double, IC_SEND_DOUBLE)

#define IC_RECV_CHAR_F77   F77_FUNC_(ic_recv_char, IC_RECV_CHAR)
#define IC_RECV_SHORT_F77  F77_FUNC_(ic_recv_short, IC_RECV_SHORT)
#define IC_RECV_INT_F77    F77_FUNC_(ic_recv_int, IC_RECV_INT)
#define IC_RECV_FLOAT_F77  F77_FUNC_(ic_recv_float, IC_RECV_FLOAT)
#define IC_RECV_DOUBLE_F77 F77_FUNC_(ic_recv_double, IC_RECV_DOUBLE)

#define IC_FREE_SCHED_F77   F77_FUNC_(ic_free_sched, IC_FREE_SCHED)
#define IC_FREE_REGION_F77  F77_FUNC_(ic_free_region, IC_FREE_REGION)
#define IC_FREE_DESC_F77    F77_FUNC_(ic_free_desc, IC_FREE_DESC)
#define IC_FREE_PROGRAM_F77 F77_FUNC_(ic_free_program, IC_FREE_PROGRAM)

#define IC_QUIT_F77 F77_FUNC_(ic_quit, IC_QUIT)

#define IC_PRINT_ERROR_F77 F77_FUNC_(ic_print_error, IC_PRINT_ERROR)

#define IC_CREATE_BDECOMP_TREE_F77 F77_FUNC_(ic_create_bdecomp_tree, IC_CREATE_BDECOMP_TREE)
#define IC_SECTION_F77 F77_FUNC_(ic_section, IC_SECTION)
#define IC_PARTITION_F77 F77_FUNC_(ic_partition, IC_PARTITION)
#define IC_VERIFY_BDECOMP_TREE_F77 F77_FUNC_(ic_verify_bdecomp_tree, IC_VERIFY_BDECOMP_TREE)

#define MAX_POINTERS 1024

void* parray[MAX_POINTERS];
int pnext = 0;

void pinitialize() {
  memset(parray, 0, MAX_POINTERS*sizeof(void*));
}

int pinsert(void* p) {
  int porig;

  if (!p)
    return -1;

  porig = pnext;
  while (parray[pnext]) {
    pnext = (pnext + 1) % MAX_POINTERS;
    if (pnext == porig) {
      errormsg("pointer table full");
      return -1;
    }
  }

  porig = pnext;
  parray[porig] = p;
  pnext = (pnext + 1) % MAX_POINTERS;
  return porig;
}

void* plookup(int i) {
  return parray[i];
}

void* premove(int i) {
  void* p;
  p = parray[i];
  parray[i] = 0;
  return p;
}

char* ftoc(char* s, int l) {
  char* t;
  int i = 0;

  t = (char*)calloc(l+1, sizeof(char));
  while (i < l) {
    t[i] = s[i];
    ++i;
  }
  t[i] = 0;

  return t;
}

void IC_INIT_F77(char* g, int* tasks, int* rank, int* p, int l) {
  char* group;
  IC_Program* this;

  pinitialize();  

  group = ftoc(g, l);

  this = IC_Init(group, *tasks, *rank);

  *p = pinsert((void*)this);
#ifdef IC_DEBUG
  printf("IC_Init(%s, %d, %d)=%d\n", group, *tasks, *rank, *p);
#endif
  free(group);
}

void IC_WAIT_F77(char* g, int* tasks, int* p, int l) {
  char* group;
  IC_Program* other;

  group = ftoc(g, l);

  other = IC_Wait(group, *tasks);

  *p = pinsert((void*)other);
#ifdef IC_DEBUG
  printf("IC_Wait(%s, %d)=%d\n", group, *tasks, *p);
#endif
  free(group);
}

void IC_SYNC_F77(int* p1, int* p2, int* sts) {
  IC_Program* this;
  IC_Program* other;

  this = (IC_Program*)plookup(*p1);
  other = (IC_Program*)plookup(*p2);

  *sts = IC_Sync(this, other);

#ifdef IC_DEBUG
  printf("IC_Sync(%d, %d)=%d\n", *p1, *p2, *sts);
#endif
}

#include "general_bsparti.h"

void IC_CREATE_BDECOMP_DESC_F77(int* rank, int* b, int* tasks, int* count, int* d) {
  int i, j, *blocks;
  IC_Desc* desc;
  decomp_Birreg* decomp;

  blocks = (int*)calloc((*count)*2*(*rank), sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    for (j = 0; j < *rank; ++j) {
      blocks[i*2*(*rank)+j] = b[j*2*(*count)+i] - 1;
      blocks[i*2*(*rank)+(*rank)+j] = b[j*2*(*count)+(*count)+i] - 1;
    }
  }
  
  desc = IC_Create_bdecomp_desc(*rank, blocks, tasks, *count);
  decomp = (decomp_Birreg*)desc->spec;
  decomp->col_majeur = 1;
#ifdef IC_DEBUG
  print_irr(decomp->root);
#endif

  *d = pinsert((void*)desc);
  free(blocks);
  
#ifdef IC_DEBUG
  printf("IC_Create_bdecomp(%d, x, x, %d)=%d\n", *rank, *count, *d);
#endif
}

void IC_CREATE_TTABLE_DESC_F77(int* g, int* l, int* tasks, int* count, int* d) {
  int i, *globals, *locals;
  IC_Desc* desc;

  globals = (int*)calloc(*count, sizeof(int));
  locals = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    globals[i] = g[i] - 1;
    locals[i] = l[i] - 1;
  }
  
  desc = IC_Create_ttable_desc(globals, locals, tasks, *count);

  *d = pinsert((void*)desc);
  free(locals);
  free(globals);
  
#ifdef IC_DEBUG
  printf("IC_Create_ttable(x, x, x, %d)=%d\n", *count, *d);
#endif
}

void IC_CREATE_ENUM_REGION_F77(int* x, int* count, int* r) {
  int i, *indices;
  IC_Region* region;

  indices = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    indices[i] = x[i] - 1;
  }

  region = IC_Create_enum_region(indices, *count);
#ifdef IC_DEBUG
  Print_Region(region);
#endif

  *r = pinsert((void*)region);
  free(indices);
}

void IC_CREATE_BLOCK_REGION_F77(int* rank, int* l, int* u, int* stride, int* r) {
  int i, *lower, *upper;
  IC_Region* region;

  lower = (int*)calloc(*rank, sizeof(int));
  upper = (int*)calloc(*rank, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *rank; ++i) {
    lower[i] = l[i] - 1;
    upper[i] = u[i] - 1;
  }

  region = IC_Create_block_region(*rank, lower, upper, stride);
#ifdef IC_DEBUG
  Print_Region(region);
#endif

  *r = pinsert((void*)region);
  free(upper);
  free(lower);
}

void IC_COMPUTE_SCHEDULE_F77(int* p1, int* p2, int* d, int* r, int* num_of_regions, int* s) {
  IC_Program* this;
  IC_Program* other;
  IC_Desc* desc;
  IC_Region** regions;
  IC_Sched* sched;
  int i;

  this = (IC_Program*)plookup(*p1);
  other = (IC_Program*)plookup(*p2);
  desc = (IC_Desc*)plookup(*d);
  regions = (IC_Region**)calloc(*num_of_regions, sizeof(IC_Region*));
  for (i = 0; i < *num_of_regions; ++i) {
    regions[i] = (IC_Region*)plookup(r[i]);
  }

  sched = IC_Compute_schedule(this, other, desc, regions, *num_of_regions);

  *s = pinsert((void*)sched);
  free(regions);
}

void IC_SEND_CHAR_F77(int* p, int* s, char* data, int* tag, int* sts) {
  IC_Program* to;
  IC_Sched* sched;

  to = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Send_char(to, sched, data, *tag);  
}

void IC_SEND_SHORT_F77(int* p, int* s, short* data, int* tag, int* sts) {
  IC_Program* to;
  IC_Sched* sched;

  to = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Send_short(to, sched, data, *tag);
}

void IC_SEND_INT_F77(int* p, int* s, int* data, int* tag, int* sts) {
  IC_Program* to;
  IC_Sched* sched;

  to = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Send_int(to, sched, data, *tag);
}

void IC_SEND_FLOAT_F77(int* p, int* s, float* data, int* tag, int* sts) {
  IC_Program* to;
  IC_Sched* sched;

  to = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Send_float(to, sched, data, *tag);
}

void IC_SEND_DOUBLE_F77(int* p, int* s, double* data, int* tag, int* sts) {
  IC_Program* to;
  IC_Sched* sched;

  to = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Send_double(to, sched, data, *tag);
}

void IC_RECV_CHAR_F77(int* p, int* s, char* data, int* tag, int* sts) {
  IC_Program* from;
  IC_Sched* sched;

  from = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Recv_char(from, sched, data, *tag);
}

void IC_RECV_SHORT_F77(int* p, int* s, short* data, int* tag, int* sts) {
  IC_Program* from;
  IC_Sched* sched;

  from = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Recv_short(from, sched, data, *tag);
}

void IC_RECV_INT_F77(int* p, int* s, int* data, int* tag, int* sts) {
  IC_Program* from;
  IC_Sched* sched;

  from = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Recv_int(from, sched, data, *tag);
}

void IC_RECV_FLOAT_F77(int* p, int* s, float* data, int* tag, int* sts) {
  IC_Program* from;
  IC_Sched* sched;

  from = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Recv_float(from, sched, data, *tag);
}

void IC_RECV_DOUBLE_F77(int* p, int* s, double* data, int* tag, int* sts) {
  IC_Program* from;
  IC_Sched* sched;

  from = (IC_Program*)plookup(*p);
  sched = (IC_Sched*)plookup(*s);

  *sts = IC_Recv_double(from, sched, data, *tag);
}

void IC_FREE_SCHED_F77(int* s) {
  IC_Sched* sched;

  sched = (IC_Sched*)premove(*s);

  IC_Free_sched(sched);
}

void IC_FREE_REGION_F77(int* r) {
  IC_Region* region;

  region = (IC_Region*)premove(*r);

  IC_Free_region(region);
}

void IC_FREE_DESC_F77(int* d) {
  IC_Desc* desc;

  desc = (IC_Desc*)premove(*d);

  IC_Free_desc(desc);
}

void IC_FREE_PROGRAM_F77(int* p) {
  IC_Program* other;

  other = (IC_Program*)premove(*p);

  IC_Free_program(other);
}

void IC_QUIT_F77(int* p) {
  IC_Program* this;

  this = (IC_Program*)premove(*p);

  IC_Quit(this);
}

void IC_PRINT_ERROR_F77(char* m, int l) {
  char* msg;
  
  msg = ftoc(m, l);
  IC_Print_error(msg);
  free(msg);
}

void IC_CREATE_BDECOMP_TREE_F77(int* r) {
  IC_Tree* root;
  root = IC_Create_bdecomp_tree();
  *r = pinsert((void*)root);
}

void IC_SECTION_F77(int* r, int* dim, int* count, int* x) {
  int i, *indices;
  IC_Tree* root;

  indices = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    indices[i] = x[i] - 1;
  }

  root = (IC_Tree*)plookup(*r);
  IC_Section(root, *dim, *count, indices);
  free(indices);
}

void IC_PARTITION_F77(int* r, int* dim, int* b, int* count, int* x, int* p) {
  int i, *indices, *block;
  IC_Tree* root;

  indices = (int*)calloc(*count, sizeof(int));
  /* fortran arrays start at 1 */
  for (i = 0; i < *count; ++i) {
    indices[i] = x[i] - 1;
  }
  block = (int*)calloc(*dim, sizeof(int));
  for (i = 0; i < *dim; ++i) {
    block[i] = b[i] - 1;
  }

  root = (IC_Tree*)plookup(*r);
  IC_Partition(root, *dim, block, *count, indices);
  free(indices);
  free(block);
}

void IC_VERIFY_BDECOMP_TREE_F77(int* r, int* ndims, int* size, int* tasks, int* count, int* d) {
  IC_Tree* root;
  IC_Desc* desc;
  root = (IC_Tree*)premove(*r);
  if (!(desc = IC_Verify_bdecomp_tree(root, *ndims, size, tasks, *count))) {
    *r = pinsert((void*)root);
    *d = 0;
  } else {
    *r = 0;
    *d = pinsert((void*)desc);
  }
}
