/** 
 * This is the public interface of InterComm.
 * @file intercomm.h
 * @author Christian Hansen (chansen@cs.umd.edu)
 *         University of Maryland, College Park
 */

#ifndef INTERCOMM_H
#define INTERCOMM_H

#define IC_BDECOMP 1
#define IC_TTABLE 2

/* Built-in types */

/** A program handle */
typedef struct {
  int nbNode;
  char* name;
  int* tid;
  int myPos;
  char* joined;
} IC_Program;

struct leaf_node;

/** A partial decomposition descriptor */
typedef struct decomp_TREE {
  int nDims;
  int num_of_partitions;
  struct decomp_TREE* children;
  int* keys;
  struct leaf_node* block_info;
} IC_Tree;

/** A decomposition descriptor */
typedef struct {
  int type;
  void* spec;
} IC_Desc;

typedef enum { R_Enum, R_HPF } T_RegionType;
typedef enum { R_replicated, R_distributed, R_local } T_RegionDistribution;
typedef enum { Mem_Copy, Mem_Pointer } T_MemOp;

/** An array region */
typedef struct {
  T_RegionType type;
  T_RegionDistribution distribution;
  T_MemOp mem_type;
  int size;
} IC_Region;

typedef struct OneSched_ {
  struct OneSched_ *next;
  int offset;
  int size;
} OneSched_s;

/** A communication schedule */
typedef struct {
  int nproc;
  int *size;
  OneSched_s **sched;
  OneSched_s **last;
} IC_Sched;

/**
 * Initialize the communication library
 * @param name   a unique program name
 * @param tasks  the number of tasks in this program
 * @param rank   the rank of this task
 * @return a handle for this program
 */
IC_Program* IC_Init(char* name, int tasks, int rank);

/**
 * Contact another program
 * @param name   the name of the other program (as given in its IC_Init call)
 * @param tasks  the number of tasks in the other program
 * @return a handle for the other program
 */
IC_Program* IC_Wait(char* name, int tasks);

/** 
 * Synchronizes two programs
 * @param myprog  a handle to this program
 * @param prog    a handle to the other program
 * @return status
 */
int IC_Sync(IC_Program* myprog, IC_Program* prog);

/**
 * Create a block decomposition
 * @param ndims   the number of dimensions
 * @param blocks  an array of block bound specifications
 * @param tasks   the task assignments for each block
 * @param count   the number of blocks
 * @return the decomposition descriptor
 */
IC_Desc* IC_Create_bdecomp_desc(int ndims, int* blocks, int* tasks, int count);

/**
 * Create a translation table
 * @param globals  a list of global indices
 * @param locals   the local index for each global index
 * @param tasks    the task assignment for each global index
 * @param count    the number of global indices
 * @return the decomposition descriptor
 */
IC_Desc* IC_Create_ttable_desc(int* globals, int* locals, int* tasks, int count);

/**
 * Allocate a partial decomposition
 * @return the partial descriptor
 */
IC_Tree* IC_Create_bdecomp_tree();

/**
 * Create a partition for a partial descriptor
 * @param the partial descriptor
 * @param the dimension to partition
 * @param the number of partitions
 * @param the upper bounding index for each partition
 */
void IC_Section(IC_Tree* root, int dim, int count, int* indices);

/**
 * Create a partition for a partial descriptor
 * @param the partial descriptor
 * @param the dimension to partition
 * @param the block to partition
 * @param the number of partitions
 * @param the upper bounding index for each partition
 * @return an array of partial descriptors which can then be partitioned further
 */
void IC_Partition(IC_Tree* root, int dim, int* block, int count, int* indices);

/**
 * Verify a partial descriptor and generate a complete descriptor
 * @param the partial descriptor
 * @param the number of dimensions
 * @param the size of the global array
 * @param the task assignments for each block
 * @param the number of blocks
 * @return the decomposition descriptor
 */
IC_Desc* IC_Verify_bdecomp_tree(IC_Tree* root, int ndims, int* size, int* tasks, int count);

/**
 * Allocate a block region structure
 * @param ndims   the number of dimensions
 * @param lower   the lower bounds
 * @param upper   the upper bounds
 * @param stride  the stride
 * @return the array region
 */
IC_Region* IC_Create_block_region(int ndims, int* lower, int* upper, int* stride);

/**
 * Allocate an enumeration region structure
 * @param indices  a list of global indices
 * @param count    the number of indices
 * @return the array region
 */
IC_Region* IC_Create_enum_region(int* indices, int count);

/**
 * Compute a schedule for the given regions
 * @param myprog      a handle to this program
 * @param prog        a handle to the other program
 * @param desc        a decomposition descriptor
 * @param region_set  a list of regions
 * @param set_size    the number of regions
 * @return the communication schedule
 */
IC_Sched* IC_Compute_schedule(IC_Program* myprog, IC_Program* prog, IC_Desc* desc, 
			      IC_Region** region_set, int set_size);

/* These TYPE functions are added as placeholders for Doxygen */
#define TYPE void

/**
 * Send a region to another program.  There is a send call for each C TYPE (char, 
 * short, int, long, float, double).
 * @li <b>int IC_Send_char(IC_Program* to, IC_Sched* sched, char* data, int tag)</b>
 * @li <b>int IC_Send_short(IC_Program* to, IC_Sched* sched, short* data, int tag)</b>
 * @li <b>int IC_Send_int(IC_Program* to, IC_Sched* sched, int* data, int tag)</b>
 * @li <b>int IC_Send_long(IC_Program* to, IC_Sched* sched, long* data, int tag)</b>
 * @li <b>int IC_Send_float(IC_Program* to, IC_Sched* sched, float* data, int tag)</b>
 * @li <b>int IC_Send_double(IC_Program* to, IC_Sched* sched, double* data, int tag)</b>
 * @param to     the receiving program
 * @param sched  the communication schedule
 * @param data   the local portion of the global array
 * @param tag    a message tag
 * @return status
 */
int IC_Send_TYPE(IC_Program* to, IC_Sched* sched, TYPE* data, int tag);

int IC_Send_char(IC_Program* to, IC_Sched* sched, char* data, int tag);
int IC_Send_short(IC_Program* to, IC_Sched* sched, short* data, int tag);
int IC_Send_int(IC_Program* to, IC_Sched* sched, int* data, int tag);
int IC_Send_long(IC_Program* to, IC_Sched* sched, long* data, int tag);
int IC_Send_float(IC_Program* to, IC_Sched* sched, float* data, int tag);
int IC_Send_double(IC_Program* to, IC_Sched* sched, double* data, int tag);

/** 
 * Receive a region from a program.  There is a receive call for each C TYPE (char,
 * short, int, long, float, double).
 * @li <b>int IC_Recv_char(IC_Program* from, IC_Sched* sched, char* data, int tag)</b>
 * @li <b>int IC_Recv_short(IC_Program* from, IC_Sched* sched, short* data, int tag)</b>
 * @li <b>int IC_Recv_int(IC_Program* from, IC_Sched* sched, int* data, int tag)</b>
 * @li <b>int IC_Recv_long(IC_Program* from, IC_Sched* sched, long* data, int tag)</b>
 * @li <b>int IC_Recv_float(IC_Program* from, IC_Sched* sched, float* data, int tag)</b>
 * @li <b>int IC_Recv_double(IC_Program* from, IC_Sched* sched, double* data, int tag)</b>
 * @param from   the sending program
 * @param sched  the communication schedule
 * @param data   the local portion of the global array
 * @param tag    a message tag
 * @return status
 */
int IC_Recv_TYPE(IC_Program* from, IC_Sched* sched, TYPE* data, int tag);

int IC_Recv_char(IC_Program* from, IC_Sched* sched, char* data, int tag);
int IC_Recv_short(IC_Program* from, IC_Sched* sched, short* data, int tag);
int IC_Recv_int(IC_Program* from, IC_Sched* sched, int* data, int tag);
int IC_Recv_float(IC_Program* from, IC_Sched* sched, float* data, int tag);
int IC_Recv_double(IC_Program* from, IC_Sched* sched, double* data, int tag);

/**
 * Frees a schedule
 * @param sched  a communication schedule 
 */
void IC_Free_sched(IC_Sched* sched);

/**
 * Frees a region
 * @param region  an array region
 */
void IC_Free_region(IC_Region* region);

/**
 * Frees a descriptor
 * @param desc  an array partitioning descriptor
 */
void IC_Free_desc(IC_Desc* desc);

/**
 * Fress a program handle
 * @param prog  a program handle
 */
void IC_Free_program(IC_Program* prog);

/**
 * Frees up internal data structures and
 * free the program handle for this program
 * @param myprog  the handle for this program
 */
void IC_Quit(IC_Program* myprog);

/**
 * Prints an error message for the last call that failed
 * @param msg  a message that will be prepended to the internal error message  
 */
void IC_Print_error(char* msg);

#endif /* INTERCOMM_H */
