#include "region.h"
#include "global.h"
#include "util.h"
#include "error.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

IC_Region* IC_Create_enum_region(int* indices, int size) {
  setcall("IC_Create_enum_region");
  if (indices == 0 || size < 1) {
    errormsg("invalid parameters");
    return 0;
  }
  return Alloc_R_Enum(size, indices, R_replicated, Mem_Copy);
}

IC_Region* IC_Create_block_region(int rank, int* lower, int* upper, int* stride) {
  setcall("IC_Create_block_region");
  if (rank < 1 || lower == 0 || upper == 0 || stride == 0) {
    errormsg("invalid parameters");
    return 0;
  }
  return Alloc_R_HPF(rank, lower, upper, stride, R_replicated, Mem_Copy);
}

void IC_Free_region(IC_Region* region) {
  if (region) {
    Free_Region(region);
  }
}

void Free_Region(region_s *region) {
  switch (region->type) {
  case R_Enum: 
    Free_R_Enum((region_enum_s*)region);
    break;
  case R_HPF:
    Free_R_HPF((region_hpf_s *)region);
    break;
  default:
    fprintf(stderr, "invalid region type\n");
    break;
  }
}  

void Print_Region(region_s *region) {
  if (region!=NULL) {
    switch (region->type) {
    case R_Enum: 
      Print_R_Enum((region_enum_s *)region);
      break;
    case R_HPF:
      Print_R_HPF((region_hpf_s *)region);
      break;
    default:
      fprintf(stderr, "invalid region type");
      break;
    }
  } else {
    printf("\n Region : NULL ");
  }  
}

void getRegionName(region_s *region, char *name) {
  if (region!=NULL) {
    switch (region->type) {
    case R_Enum: 
      strcpy(name,"R_Enum");
      break;
    case R_HPF:
      strcpy(name,"R_HPF");
      break;
    default:
      fprintf(stderr, "invalid region type");
      break;
    }
  } else {
    strcpy(name," [region == NULL] ");
  } 
} 
  
setOfRegion_s *Alloc_setOfRegion() {
  setOfRegion_s *setR;
  setR = malloc(sizeof(setOfRegion_s));
  setR->NumRegion = 0;
  setR->regions = NULL;
  setR->BeginGlobalOffset = (int *)malloc(sizeof(int) * 1);
  setR->BeginGlobalOffset[0] = 0;
  return setR;
}

setOfRegion_s *Add_Region_setOfRegion(region_s *region, setOfRegion_s *setOfRegion) {
  if (region == NULL)
    return(setOfRegion);
  
  if (setOfRegion == NULL) 
    setOfRegion = Alloc_setOfRegion();

  setOfRegion->regions = (region_s**)realloc(setOfRegion->regions, (setOfRegion->NumRegion+1)*sizeof(region_s*));
  setOfRegion->BeginGlobalOffset = (int*)realloc(setOfRegion->BeginGlobalOffset, ((setOfRegion->NumRegion)+2)*sizeof(int));
  setOfRegion->regions[setOfRegion->NumRegion] = region;
  setOfRegion->BeginGlobalOffset[setOfRegion->NumRegion+1] = setOfRegion->BeginGlobalOffset[setOfRegion->NumRegion]+region->size;
  (setOfRegion->NumRegion)++;

  return setOfRegion;
}

setOfRegion_s *Add_LocalRegion_setOfRegion(region_s *region, setOfRegion_s *setOfRegion, int Goffset) {
  if (region == NULL)
    return(setOfRegion);

  if (setOfRegion == NULL) 
    setOfRegion = Alloc_setOfRegion();

  setOfRegion->regions = (region_s**)realloc(setOfRegion->regions, (setOfRegion->NumRegion+1)*sizeof(region_s*));  
  setOfRegion->BeginGlobalOffset = (int*)realloc(setOfRegion->BeginGlobalOffset, ((setOfRegion->NumRegion)+2)*sizeof(int));
  setOfRegion->regions[setOfRegion->NumRegion] = region;
  setOfRegion->BeginGlobalOffset[setOfRegion->NumRegion] = Goffset;
  setOfRegion->BeginGlobalOffset[setOfRegion->NumRegion+1] = Goffset+region->size;
  (setOfRegion->NumRegion)++;

  return setOfRegion;
}

void Free_setOfRegion(setOfRegion_s *setOfRegion) {
  if (setOfRegion != NULL) {
    free(setOfRegion->regions);
    free(setOfRegion->BeginGlobalOffset);
    free(setOfRegion);
  }
}

void Print_setOfRegion(setOfRegion_s *s) {
  int i;
  
  printf("\n");
  printf("\nSet Of Region ");
  if (s!=NULL) {
    printf("\n   NumRegion=%i ",s->NumRegion);
    printf("\n   BeginGlobalOffset=[");
    for (i=0;i<s->NumRegion+1;i++)
      printf(" %i",s->BeginGlobalOffset[i]);
    printf(" ]");
    for (i=0;i<s->NumRegion;i++)
      Print_Region(s->regions[i]);     
  } else
    printf("Set Of Region : NULL \n");
  printf("\n");
}

int SizeSetOfRegion(setOfRegion_s *setOfRegion) {
  if (setOfRegion->BeginGlobalOffset == NULL)
    return 0;

  return setOfRegion->BeginGlobalOffset[NbRegionSetOfRegion(setOfRegion)];
}

int NbRegionSetOfRegion(setOfRegion_s *setOfRegion) {
  return setOfRegion->NumRegion;
}

region_s *Alloc_R_Enum(int NumElem, int *globalCoord, T_RegionDistribution distribution, T_MemOp mem_type) {
  region_enum_s *r;

  r = malloc(sizeof(region_enum_s));
  r->type = R_Enum;
  r->distribution = distribution;
  r->size = NumElem;
  r->mem_type = mem_type;
  switch (r->mem_type) {
  case Mem_Copy:
    r->globalCoord = (int*)malloc(sizeof(int)*NumElem);
    memcpy(r->globalCoord,globalCoord,sizeof(int)*NumElem);
    break;
  case Mem_Pointer:
    r->globalCoord = globalCoord;
    break;
  default:
    warning("Unknown Memory Operation");
    break;
  }
  
  return (region_s*)r;
}

void Free_R_Enum(region_enum_s* region) {
  if (region != NULL) {
    switch (region->mem_type) {
    case Mem_Copy:
      free(region->globalCoord);
      break;
    case Mem_Pointer:	
      break;
    default:
      warning("Unkown Memory Operation");
      break;
    }  
    free(region);
  }
}

void Print_R_Enum(region_enum_s *r) {
  int i;
  
  if (r != NULL) {
    printf("\n");
    printf("\nRegion type: Enum ");
    printf("\n  distribution=%i", r->distribution);
    printf("\n  size=%i ", r->size);
    printf("\n  mem_type=%i ", r->mem_type);
    printf("\n  globalCoord=[");
    for (i = 0; i < r->size; i++)
      printf(" %i", r->globalCoord[i]);
    printf(" ]\n");
  } else {
    printf("\n Region : NULL\n ");
  }
}

region_s *Alloc_R_HPF(int nbDim, int *left, int *right, int *stride, T_RegionDistribution distribution, T_MemOp mem_type) {
  region_hpf_s *r = malloc(sizeof(region_hpf_s));
  int taille[MAX_lbB_DIM];
  int i;
  
  r->type = R_HPF;
  r->distribution = distribution;
  r->nbDim = nbDim;
  
  r->mem_type = mem_type;
  switch (r->mem_type) {
  case Mem_Copy:
    r->left = (int *)malloc(sizeof(int)*nbDim);
    r->right = (int *)malloc(sizeof(int)*nbDim);
    r->stride = (int *)malloc(sizeof(int)*nbDim);
    memcpy(r->left  ,left  ,sizeof(int)*nbDim);
    memcpy(r->right ,right ,sizeof(int)*nbDim);
    memcpy(r->stride,stride,sizeof(int)*nbDim);
    break;
  case Mem_Pointer:
    r->left = left;
    r->right = right; 
    r->stride = stride;
    break;
  default:
    warning("Unkown Memory Operation");
    break;
  }
  
  for (i = 0; i < nbDim; i++){
    /*taille[i] = (right[i]-left[i]+1)/stride[i]; */ /* A bug */
    taille[i] = (right[i]-left[i])/stride[i] +1;
  }

  r->size = 1;
  for (i = 0; i < nbDim; i++)
    r->size = taille[i]*r->size;

  return (region_s *)r;
}

void Free_R_HPF(region_hpf_s *region) {
  if (region != NULL) {
    switch (region->mem_type) {
    case Mem_Copy:
      free(region->left);
      free(region->right);
      free(region->stride);
      break;
    case Mem_Pointer:	
      break;
    default:
      warning("Unkown Memory Operation");
      break;
    }  
    free(region);  
  }
}

void Print_R_HPF(region_hpf_s *r)
{ 
  int i;

  if (r != NULL) {
    printf("\n");  
    printf("\nRegion type: HPF");
    printf("\n  distribution=%i",r->distribution);
    printf("\n  size=%i ",r->size);
    printf("\n  nbDim=%i",r->nbDim);
    printf("\n  mem_type=%i ",r->mem_type);
    printf("\n  left  =[");
    for (i=0;i<r->nbDim;i++)
      printf(" %3i",r->left[i]);
    printf(" ]");
    printf("\n  right =[");
    for (i=0;i<r->nbDim;i++)
      printf(" %3i",r->right[i]);
    printf(" ]");
    printf("\n  stride=[");
    for (i=0;i<r->nbDim;i++)
      printf(" %3i",r->stride[i]);
    printf(" ]\n");
  } else {
    printf("\n Region : NULL \n");
  }
}


