// This code is coupled with the two sides of the simulation of the Wave 
// equation. It is used to collect the simulation snapshots and to visualize
// the solution
//
// It requires running two endpoint objects. Each one dealing with a 
// different "side" of the simulation
#include <A++.h>
#include <iostream>
#include <stdlib.h>
#include "FILE_EndPoint.h"
#include "IC_EndPoint.h"
#include "IC_support.h"
#include "IC_util.h"

#if !defined(FILEENDPOINT) && !defined(ICENDPOINT)
#error "you have to define the endpoint to use in the Makefile"
#endif

// this setting allows the data to be modified upon receipt
// #define AMPLIFY -- it has been undefined recently to cleanup the interface

const char localconf[]="waveeq.conf";

int main(int argc, char* argv[]) {
  unsigned nright, nleft;
  int iNumProcs;
  Optimization_Manager::Initialize_Virtual_Machine("",iNumProcs,argc,argv);
  ConfFile conf;
  int ret=conf.readConfFile(localconf);
  if (ret!=CF_OK) {
    std::cerr << "Configuration file '" << localconf << "' missing!\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  // reads the simulation parameters to gather information about how
  // it should be displayed
  int iNumX, iNumY, iNSteps;
  const char *ptr=conf.getValue4("SimulationParameters","XGridSize");
  if (ptr) {
    iNumX=atoi(ptr);
  }
  else {
    std::cerr << "Missing value 'XGridSize'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }
  ptr=conf.getValue4("SimulationParameters","YGridSize");
  if (ptr) {
    iNumY=atoi(ptr);
  }
  else {
    std::cerr << "Missing value 'YGridSize'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }
  ptr=conf.getValue4("SimulationParameters","NTimeSteps");
  if (ptr)
    iNSteps=atoi(ptr);
  else {
    std::cerr << "Missing value 'NTimeSteps'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  ptr=conf.getValue4("SimulationParameters","LeftNProcessors");
  if (ptr)
    nleft=atoi(ptr);
  else {
    std::cerr << "Missing value 'LeftNProcessors'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  ptr=conf.getValue4("SimulationParameters","RightNProcessors");
  if (ptr)
    nright=atoi(ptr);
  else {
    std::cerr << "Missing value 'RightNProcessors'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  doubleArray daUn(iNumX+2,iNumY+2);

  // creates the communication endpoints and the interpolator objects
#ifdef ICENDPOINT
  int ic_err;
  IC_EndPoint left_vis_ep("vis:left",1,nleft,IC_EndPoint::IC_COLUMN_MAJOR,IC_EndPoint::IC_COLUMN_MAJOR,ic_err);
  if (ic_err!=IC_EndPoint::IC_OK) {
    std::cerr << "ic_endpointconstructor: " << IC_EndPoint::IC_errors[-ic_err] << endl;
    return ic_err;
  }
  IC_EndPoint right_vis_ep("vis:right",1,nright,IC_EndPoint::IC_COLUMN_MAJOR,IC_EndPoint::IC_COLUMN_MAJOR,ic_err);
  if (ic_err!=IC_EndPoint::IC_OK) {
    std::cerr << "ic_endpointconstructor: " << IC_EndPoint::IC_errors[-ic_err] << endl;
    return ic_err;
  }

#else
  FILE_EndPoint left_vis_ep("vis:left");
  FILE_EndPoint right_vis_ep("vis:right");
#endif

  // collects the simulation snapshots from the two "sides" of the
  // simulation
  int i;
  int xhalf=(iNumX/2);
  for(i=0;i<iNSteps;i++) {
    // gather data from left-hand side of the simulation
    Index Il(0,xhalf+2), Jl(0,iNumY+2);
    const doubleArray& templ=daUn(Il,Jl);
#ifdef AMPLIFY
    left_vis_ep.importArray(templ,ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
    // it is a very simple example just to show that the Interpolator can do
    // tricks with the array
    left_vis_ep.amplify(templ,1.2);
#else
    left_vis_ep.importArray(templ,ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
#endif

//    daUn.display("daUn -- after from left");

    // gather data from the right-hand side of the simulation
    Index Ir(xhalf+1,iNumX-xhalf+1), Jr(0,iNumY+2);
    const doubleArray& tempr=daUn(Ir,Jr);
#ifdef AMPLIFY
    right_vis_ep.importArray(tempr,ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
    // it is a very simple example just to show that the Interpolator can do
    // tricks with the array
    right_vis_ep.amplify(tempr,0.2);
#else
    right_vis_ep.importArray(tempr,ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
#endif

//    daUn.display("daUn -- after from right");

    // generates a pbm file with the "overall" snapshot
    gnuplot("vis",daUn,0,(20.0/iNumX)*0.1,0,(20.0/iNumY)*0.1,i);
  }
  Optimization_Manager::Exit_Virtual_Machine();

  return 0;
}
