// Solves the Wave Eq, Utt - c Uxx = 0, using P++
//
// Henrique Andrade (based on code originally written by Mike Wiltberger)
//
// This code is responsible for processing the right side of the simulation
// grid. That is, the grid is divided into two halves in the x-axis. This
// code handles the grid points to the right of the middle point in the
// x-axis
#include <A++.h>
#include <iostream>
#include <stdlib.h>
#include "IC_util.h"
#include "IC_support.h"
#include <time.h>
#include <math.h>
#include "FILE_EndPoint.h"
#include "IC_EndPoint.h"

#if (defined(ICENDPOINT) && defined(FILEENDPOINT)) || (!defined(ICENDPOINT) && !defined(FILEENDPOINT))
#error You have to define the endpoint to use in the Makefile
#endif

#define VIS

const char localconf[]="waveeq.conf";

int main(int argc, char **argv) {
  int iNumProcs, myRank=0;
  Optimization_Manager::Initialize_Virtual_Machine("",iNumProcs,argc,argv);

  // Length of physical dimensions and time
  double dLenX, dLenY;
  // Steps sizes
  double dDX, dDY, dDT,dC;
  int gridSizeX, gridSizeY;

  ConfFile conf;
  int ret=conf.readConfFile(localconf);
  if (ret!=CF_OK) {
    std::cerr << "Configuration file '" << localconf << "' missing!\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  // reads the simulation parameters from a configuration file
  int iNumX, iNumY, iNSteps;
  const char *ptr=conf.getValue4("SimulationParameters","XGridSize");
  if (ptr) {
    gridSizeX=atoi(ptr);
    iNumX=atoi(ptr)-(atoi(ptr)/2);
  }
  else {
    std::cerr << "Missing value 'XGridSize'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }
  ptr=conf.getValue4("SimulationParameters","YGridSize");
  if (ptr) {
    gridSizeY=atoi(ptr);
    iNumY=atoi(ptr);
  }
  else {
    std::cerr << "Missing value 'YGridSize'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }
  ptr=conf.getValue4("SimulationParameters","NTimeSteps");
  if (ptr)
    iNSteps=atoi(ptr);
  else {
    std::cerr << "Missing value 'NTimeSteps'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }
  ptr=conf.getValue4("SimulationParameters","TimeStep");
  if (ptr)
    dDT = atof(ptr);
  else
    dDT = 1.0e-2;

  int nright, nleft;

  ptr=conf.getValue4("SimulationParameters","LeftNProcessors");
  if (ptr)
    nleft=atoi(ptr);
  else {
    std::cerr << "Missing value 'LeftNProcessors'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  ptr=conf.getValue4("SimulationParameters","RightNProcessors");
  if (ptr)
    nright=atoi(ptr);
  else {
    std::cerr << "Missing value 'RightNProcessors'...\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

  if (nright!=iNumProcs) {
    std::cerr << "The entry RightNProcessors in " << localconf << " is inconsistent with -np parameter in mpirun!\n";
    Optimization_Manager::Exit_Virtual_Machine();
    return -1;
  }

#ifdef ICENDPOINT
  ret=MPI_Comm_rank(MPI_COMM_WORLD,&myRank);
  assert(ret==MPI_SUCCESS);
#endif

  std::cout << "Running on " << iNumProcs << " processors. My rank is " << myRank << endl;

  int i,j;

  // Set Parameters
  dLenX = 1.0; dLenY = 1.0;
  dDX = dLenX/iNumX;
  dDY = dLenY/iNumY;
  dC = 1.0;

  // Arrays to hold solution at steps n-1,n,and n+1
  // Size is NumX+2 to allow for BC on either side which we are periodic
  doubleArray daUnm1(iNumX+2,iNumY+2),daUn(iNumX+2,iNumY+2);
  doubleArray daUnp1(iNumX+2,iNumY+2);

  doubleArray daX(iNumX+2), daY(iNumY+2);
  for(i=1;i<iNumX+1;i++) {
    daX(i) = 1+(i-1)*dDX;
//    printf("i - x: %d %.3g\n",i,daX(i));
  }
#ifdef DISPLAY
  daX.display("X");
#endif
  for(j=1;j<iNumY+1;j++)
    daY(j) = (j-1)*dDY;

  Index I(1,iNumX), J(1,iNumY); // Indices for computational domain
//  Index If,Jf; // Indices for entire domain
// computational domain should exclude the ghost cells
  Index If(1,iNumX+1),Jf; // Indices for entire domain

  // Impose an initial wave in the form sin(wt+x/L) with constraint that 
  // w = c/L
  double dW;
  dW = dC/dLenX;
  double dTime;
  dTime = -1.0*dDT;
  double dPi;
  dPi = M_PI;
  for(j=1;j<iNumY+1;j++) {
    daUnm1(I,j) = sin(dW*dTime + (daX(I)*2*dPi)/dLenX); 
#ifdef DISPLAY
    daUnm1.display("Unm1");
#endif
    daUn(If,j)  = sin(dW*0 + (daX(If)*2*dPi)/dLenX); 
#ifdef DISPLAY
    daUn.display("Un");
#endif
  }

  // Apply BC

  // Since the simulation has been broken into two halves in the x-axis.
  // Applying the boundary conditions requires exchanging data between the
  // two halves.
  // The data exchange happens using a communication endpoint and an
  // interpolator object.
  // In fact, below, two endpoints are defined as well as two interpolator
  // objects. One pair is responsible for handling the communication between
  // the two halves (it is a bi-directional endpoint/interpolator) and the
  // the other one handles the communication with the visualization tool 
  // (vis.cc). The visualization tool is unidirectional, since it only
  // gathers the data generated by the simulation.
  int ic_err;
#ifdef ICENDPOINT
  IC_EndPoint left_right_ep("right:left",iNumProcs,nleft,IC_EndPoint::IC_COLUMN_MAJOR,IC_EndPoint::IC_COLUMN_MAJOR,ic_err);
  if (ic_err!=IC_EndPoint::IC_OK) {
    std::cerr << "ic_endpointconstructor: " << IC_EndPoint::IC_errors[-ic_err] << endl;
    return ic_err;
  }
#ifdef VIS
  IC_EndPoint right_vis_ep("right:vis",iNumProcs,1,IC_EndPoint::IC_COLUMN_MAJOR,IC_EndPoint::IC_COLUMN_MAJOR,ic_err);
  if (ic_err!=IC_EndPoint::IC_OK) {
    std::cerr << "ic_endpointconstructor: " << IC_EndPoint::IC_errors[-ic_err] << endl;
    return ic_err;
  }
#endif
#else
  FILE_EndPoint left_right_ep("right:left");
#ifdef VIS
  FILE_EndPoint right_vis_ep("right:vis");
#endif
#endif

  // receive column iNumx from left side
  // daUnm1(0,J) = daUnm1(iNumX,J);
  left_right_ep.exportArray(daUnm1(iNumX,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
  left_right_ep.importArray(daUnm1(0,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  // receive column iNumX+1 from left side
  // daUnm1(iNumX+1,J) = daUnm1(1,J);
  left_right_ep.exportArray(daUnm1(1,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
  left_right_ep.importArray(daUnm1(iNumX+1,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  daUnm1(I,0) = daUnm1(I,iNumY);
  daUnm1(I,iNumY+1) = daUnm1(I,1);

  // receive column iNumX from right side
  // daUn(0,J) = daUn(iNumX,J);
  left_right_ep.exportArray(daUn(iNumX,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
  left_right_ep.importArray(daUn(0,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  // daUn(iNumX+1,J) = daUn(1,J);
  // send column 1 to right side
  left_right_ep.exportArray(daUn(1,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
  left_right_ep.importArray(daUn(iNumX+1,J),ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);

  daUn(I,0) = daUn(I,iNumY);
  daUn(I,iNumY+1) = daUn(I,1);

#ifdef DISPLAY
  daUnm1.display("Unm1");
#endif

  // Evolve a step forward in time
  dTime = dDT;

#ifdef VIS
  // TOVIS: we do not want to send the ghost cells
  Index I_noghost(1,iNumX+1);
  const doubleArray& temp_noghost=daUn(I_noghost,Jf);
#endif
  for(i=1;i<iNSteps;i++) {
    // sends data to the visualization tool 
#ifdef VIS
    right_vis_ep.exportArray(temp_noghost,ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
#endif

    // computes the next time step
    daUnp1(I,J) = ((dC*dC*dDT*dDT)/(dDX*dDX))*
                  (daUn(I-1,J)-2*daUn(I,J)+daUn(I+1,J)) +
                  2*daUn(I,J) - daUnm1(I,J);

    // Apply BC

    // receive column iNumx from left side
    // daUnp1(0,J) = daUnp1(iNumX,J);
    left_right_ep.exportArray(daUnp1(iNumX,J),ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
    left_right_ep.importArray(daUnp1(0,J),ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);

    // receive column iNumX+1 from left side
    // daUnp1(iNumX+1,J) = daUnp1(1,J);
    left_right_ep.exportArray(daUnp1(1,J),ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);
    left_right_ep.importArray(daUnp1(iNumX+1,J),ic_err);
    assert(ic_err==IC_EndPoint::IC_OK);

    daUnp1(I,0) = daUnp1(I,iNumY);
    daUnp1(I,iNumY+1) = daUnp1(I,1);

#ifdef DISPLAY
    char caTmp[80];
    sprintf(caTmp,"U at %f sec",dTime);
    daUnp1.display(caTmp);
#endif

    daUnm1 = daUn;
    daUn = daUnp1;
    dTime += dDT;
  }
  // sends the last snapshot to the visualization tool
#ifdef VIS
  right_vis_ep.exportArray(temp_noghost,ic_err);
  assert(ic_err==IC_EndPoint::IC_OK);
#endif
  Optimization_Manager::Exit_Virtual_Machine();

  return 0;
}
