#include <iostream>
#include <assert.h>
#include "IC_EndPoint.h"

#ifdef PPLUSPLUS
#include <mpi.h>
#endif

// pointer to his program
// the communication may happen between this program and many others, however
// we only need a single pointer to the current program
IC_Program* IC_EndPoint::thisProgram=NULL;
unsigned IC_EndPoint::nendpoints=0;

const char * const IC_EndPoint::IC_errors[] = {
    "no error",
    "generic error",
    "invalid number of array dimensions",
    "can't allocate InterComm array regions",
    "can't obtain distributed array descriptor",
    "can't compute communication schedule",
    "communication (send/recv) failure",
    "invalid endpoint name",
    "this program's initialization failed",
    "can't connect to remote program",
    "PVM error. Is PVM running?",
    "MPI error",
    "internal pointer table error",
};

IC_EndPoint::IC_EndPoint(const char* endpointName, 
  const unsigned mynproc, const unsigned onproc, const unsigned myao, 
        const unsigned oao, int& stat) : EndPoint(endpointName), 
  otherProgram(NULL), thisProgramAO(myao), otherProgramAO(oao) {

  assert(myao==IC_COLUMN_MAJOR || myao==IC_ROW_MAJOR);
  assert(oao==IC_COLUMN_MAJOR || oao==IC_ROW_MAJOR);
  char myPName[IC_MAX_STR], otherPName[IC_MAX_STR];

  // let's parse the interpolator name and get the info we need
  int ret=sscanf(endpointName,"%[A-z0-9]%*[ :]%[A-z0-9]",myPName,otherPName);
  if (ret!=2) {
    stat=IC_INVALID_ENDPOINT_NAME;
    return;
  }

  std::cout << "initializing endpoint " << myPName << "(" << mynproc <<  ")/" << otherPName << "(" << onproc << ")\n";
  if (!thisProgram) {
    int myrank;
#ifdef PPLUSPLUS
    int ret=MPI_Comm_rank(MPI_COMM_WORLD,&myrank);
    if (ret!=MPI_SUCCESS) {
      stat=IC_MPI_ERROR;
      return;
    }
#else
    myrank=0;
#endif
    thisProgram=IC_Init(myPName,mynproc,myrank);
    if (!thisProgram) {
      stat=IC_INITIALIZATION_FAILURE;
      return;
    }
    std::cout << "local program [" << myPName << "] has started...\n";
    ++nendpoints;
  }
  otherProgram=IC_Wait(otherPName,onproc);
  if (!otherProgram) {
    stat=IC_CANT_CONNECT_TO_REMOTE;
    return;
  }
  std::cout << "remote program [" << otherPName << "] has started...\n";

  stat=IC_OK;
}

void IC_EndPoint::printErrorMessage(const char* msg, const int ic_err) {
  unsigned nerrs=sizeof(IC_errors)/sizeof(char*);
  if (-ic_err<0 || -ic_err>nerrs) {
    std::cerr << "invalid error code\n";
    return;
  }
  if (msg)
    std::cerr << msg << ": " << IC_errors[-ic_err] << std::endl;
  else
    std::cerr << IC_errors[-ic_err] << std::endl;
}

IC_EndPoint::~IC_EndPoint() {
  --nendpoints;
  IC_Free_program(otherProgram);
  if (nendpoints==0) {
    IC_Quit(thisProgram);
  }
}
