/* 
 * File: schedule.c
 * Auth: Jae-Yong Lee (jylee@cs.umd.edu)
 *       University of Maryland, College Park
 * Desc: functions for communication schedules.
 *       Some functions from Meta-Chaos.
 * Date: Feb. 2004
 */

#include "schedule.h"
#include "communication.h"
#include <stdlib.h>
#include <stdio.h>

void IC_Free_sched(IC_Sched* sched) {
  if (sched) {
    FreeSched(sched);
  }
}

/* Not used in InterComm */
void sendScheduleLocal(sched_s **tab_sched_recv, 
		       sched_s **tab_sched_send, 
		       pgme_s *pgme, 
		       sched_s **sched_send, 
		       sched_s **sched_recv) 
{
  int i;
  int me=MyPosPgme(pgme);
  int nproc=NumNodePgme(pgme);
  int i_1,proc;
  sched_s *sched;
  OneSched_s *Osched;
  sched_s *sched_sendPt;
  
  *sched_send=sched_sendPt=CreateSched(nproc);
  
  for (i_1=0;i_1<NumNodePgme(pgme);i_1++)
    {
      sched=tab_sched_send[i_1];
      for (proc=0;proc<sched->nproc;proc++)
	{
	  Osched=sched->sched[proc];
	  while (Osched!=NULL)
	    {
	      AddSched(*sched_send,proc,Osched->offset,Osched->size);
	      Osched=Osched->next;
	    }
	}
    }
  
  for (i=0;i<NumNodePgme(pgme);i++)
    {
      SendSched(pgme,i,tab_sched_recv[i],FLAG_SCHEDULE);      
    }
  
  *sched_recv=ReceiveSchedule(pgme);
}


/* Send schedules to sender and receiver processes */
void sendSchedule(sched_s **tab_sched_recv, sched_s **tab_sched_send, pgme_s *pgme_recv, pgme_s *pgme_send)
{
  int i;
  
  for (i=0;i<NumNodePgme(pgme_send);i++)
    {
      SendSched(pgme_send,i,tab_sched_send[i],FLAG_SCHEDULE);      
    }
  
  for (i=0;i<NumNodePgme(pgme_recv);i++)
    {
      SendSched(pgme_recv,i,tab_sched_recv[i],FLAG_SCHEDULE);      
    }
}

/* Send schedules to all processes of a program */
void sendScheduleToPgme(sched_s** tab_sched, pgme_s *pgme)
{
  int i;
  
  for (i=0;i<NumNodePgme(pgme);i++)
    {
      SendSched(pgme,i,tab_sched[i],FLAG_SCHEDULE);      
    }
}


/* Send a table of schedules to one process */
void sendScheduleTableToOneNode(sched_s **tab_sched, pgme_s *pgme, int to)
{
  SendSchedTab(pgme,to,tab_sched,FLAG_SCHEDULE);      
}

/* Receive schedules from two programs and merge them */
sched_s *ReceiveSchedules(pgme_s *pgme1, pgme_s *pgme2)
{
  sched_s *sched=NULL;

  sched=RecvScheds(pgme1,pgme2,sched,FLAG_SCHEDULE);
  return(sched);
}

/* Receive schedules from one program */
sched_s *ReceiveSchedule(pgme_s *pgme1)
{
  int i;
  sched_s *sched=NULL;
  sched=RecvSched(pgme1,sched,FLAG_SCHEDULE);
  return(sched);
}

/* Receive schedules from one program and rebuild them in reverse order */
sched_s *ReceiveScheduleReverseOrder(pgme_s *pgme1)
{
  sched_s *sched=NULL;

  sched=RecvSchedReverseOrder(pgme1,sched,FLAG_SCHEDULE);
  return(sched);
}

/* Receive a table of schedules form a program */
sched_s** ReceiveScheduleTab(pgme_s *pgme1, int from)
{
  sched_s** tab_sched=NULL;

  tab_sched = RecvSchedTab(pgme1,from,tab_sched,FLAG_SCHEDULE);
  return(tab_sched);
}

/* Merge tables of schedules into one table of schedules */
sched_s** merge_schedules(sched_s*** tab,int num,pgme_s* other_pgme, pgme_s* my_pgme)
{
  sched_s** tab_sched;
  int i,j,k;
  OneSched_s* trace;
  
  tab_sched=(sched_s **)malloc(sizeof(sched_s *)*NumNodePgme(my_pgme));
  
  for (i=0;i<NumNodePgme(my_pgme);i++)   
    tab_sched[i]=CreateSched(NumNodePgme(other_pgme)); 
  
  for(i=0; i < num ; i++)
    {
      for(j=0; j < NumNodePgme(my_pgme); j++)
	{
	  tab_sched[j]->nproc = NumNodePgme(other_pgme);
	  for(k=0; k < NumNodePgme(other_pgme); k++)
	    {
	      if(tab_sched[j]->sched[k]== NULL){
		tab_sched[j]->sched[k] = tab[i][j]->sched[k];
	      }
	      else{
		tab_sched[j]->last[k]->next = tab[i][j]->sched[k];
	      }
	      if(tab[i][j]->last[k])
		tab_sched[j]->last[k] = tab[i][j]->last[k]; 
	      tab_sched[j]->size[k] += tab[i][j]->size[k];
	      tab[i][j]->size[k] = 0;
	      tab[i][j]->sched[k] = tab[i][j]->last[k] = NULL; 
	    }     
	}
    }
  return tab_sched;
} 

sched_s* Receive_Distribute_Schedules(pgme_s* my_pgme,pgme_s* other_pgme,int me)
{
  sched_s* scheds;
  int numproc;
  int other_numproc;
  int max_num_sources;
  int num_sources;
  sched_s*** tab_tab_schedules;
  sched_s** merged_tab_schedules;
  sched_s** tab_schedules;
  
  int count;
  int i,j;
  
  /* 
   * receive from non-compact nodes which this process sent 
   * a compact descriptor 
   */
  numproc = NumNodePgme(my_pgme);
  other_numproc = NumNodePgme(other_pgme);
  
  max_num_sources = other_numproc -  me*( (other_numproc -1)/(numproc)+1);
  num_sources = (other_numproc -1)/(numproc)+1;
  
  if((max_num_sources > 0 )&& (max_num_sources< num_sources)) num_sources = max_num_sources;
  if(max_num_sources <=0) num_sources = 0;
  
  /*
   * can be improved..
   * do not need to build tab_tab_schedules and merge them
   * we can merge raw data 
   * but it's complicated and needs many memory copies.
   */
  tab_tab_schedules = NULL;    
  tab_tab_schedules = (sched_s***)malloc(sizeof(sched_s**)*num_sources);
  
  count =0;
  
  for(i = me*( (other_numproc -1)/(numproc)+1) ; i < (me+1)*( (other_numproc -1)/(numproc)+1) && i < other_numproc; i++)
    {
      tab_schedules = ReceiveScheduleTab(other_pgme,i);
      tab_tab_schedules[count++]= tab_schedules;
    }
  
  merged_tab_schedules = merge_schedules(tab_tab_schedules,num_sources,other_pgme,my_pgme);
  
  for(i=0;i<NumNodePgme(my_pgme); i++){
    SendSched(my_pgme,i,merged_tab_schedules[i],FLAG_SCHEDULE);
  }
  
  scheds = ReceiveSchedule(my_pgme);
  
  /* deallocate tab_tab_schedules and the merged_tab_schedules */
  
  if(num_sources != 0){
    for(i =0; i< num_sources; i++){
      for(j =0; j < NumNodePgme(my_pgme); j++)
	FreeSched(tab_tab_schedules[i][j]);
      free(tab_tab_schedules[i]);
    }
    
    free(tab_tab_schedules);
    
    for(i =0; i < NumNodePgme(my_pgme); i++)
      FreeSched(merged_tab_schedules[i]);
    free(merged_tab_schedules);
  }
  return scheds;
  
}


sched_s *CreateSched(int nproc)
{
  sched_s *sched;
  int i;
  
  sched=malloc(sizeof(sched_s));
  sched->sched=(OneSched_s **)malloc(sizeof(OneSched_s *)*nproc);
  sched->size=(int *)malloc(sizeof(int)*nproc);
  sched->nproc=nproc;
  sched->last = (OneSched_s **)malloc(sizeof(OneSched_s *)*nproc);
  
  for (i=0;i<nproc;i++)
    {
      sched->size[i]=0;
      sched->sched[i]=NULL;
      sched->last[i] = NULL;
    }
  return(sched);
}

sched_s** CreateTabScheds(int entries, int nproc)
{
  int i;
  sched_s** tab_sched;
  
  tab_sched=(sched_s **)malloc(sizeof(sched_s *)*entries);
  
  for (i=0;i<entries;i++) 
    tab_sched[i]=CreateSched(nproc); 
  
  return tab_sched;
}

sched_s *AddSched(sched_s *sched, int proc, int offset, int size)
{
  int i;
  OneSched_s *Osched;
 
  Osched = malloc(sizeof(OneSched_s));
  Osched->offset=offset;
  Osched->size=size;
  
  if (sched->sched[proc] == NULL) { /* added in June,2003(jylee) */
    sched->last[proc] = Osched;
  }
  
  Osched->next=sched->sched[proc];
  sched->sched[proc]=Osched;
  (sched->size[proc])++;
  
  return sched;
}

sched_s *AddSchedLast(sched_s *sched, int proc, int offset, int size)
{
  int i;
  OneSched_s *Osched;
  OneSched_s *temp;
  
  Osched=malloc(sizeof(OneSched_s));
  Osched->offset=offset;
  Osched->size=size;
  
  Osched->next = 0;
  
  if(sched->sched[proc] == NULL) {
    sched->sched[proc] = Osched;
  }
  else{
    sched->last[proc]->next = Osched;
  }
  
  sched->last[proc] = Osched;
  (sched->size[proc])++;
  
  return(sched);
}

void FreeSched(sched_s *sched)
{
  int proc;
  OneSched_s *Osched, *NextOsched;
  if (sched!=NULL)
    {
      for (proc=0;proc<sched->nproc;proc++)
	{
	  Osched=sched->sched[proc];
	  NextOsched=Osched;
	  
	  while (Osched!=NULL)
	    {
	      NextOsched=Osched->next;
	      free(Osched);
	      Osched=NextOsched;
	    }
	}
      free(sched->size);
      free(sched->sched);
      free(sched->last);
      
      free(sched);
    }
}

void FreeTabScheds(sched_s** tab_sched, int num)
{
  int i;
  
  for (i=0; i< num ;i++){
    FreeSched( tab_sched[i]);
  }
  
  free(tab_sched);
}

void PrintSched(sched_s *sched)
{
  int proc;
  OneSched_s *Osched;  
  
  printf("\n-- Sched --\n nproc=%i ",sched->nproc);
  for (proc=0;proc<sched->nproc;proc++)
    {
      printf("\n [%i] size=%i ",proc,sched->size[proc]);
      Osched=sched->sched[proc];
      while (Osched!=NULL)
	{
	  printf(" (%i,%i)",Osched->offset,Osched->size);
	  Osched=Osched->next;
	}
    }
  printf("\n");
}

void PrintSched1(sched_s *sched)
{
  int proc;
  OneSched_s *Osched;  
  int size=0;
  
  for (proc=0;proc<sched->nproc;proc++)
    {
      Osched=sched->sched[proc];
      while (Osched!=NULL)
	{
	  size+=Osched->size;
	  Osched=Osched->next;
	}
    }
  printf("\n-- Sched -- nproc=%i -- size %i %i  ",sched->nproc,sched->size,size);
}
