#include <unistd.h> /* read */
#include <errno.h>
#include <assert.h>
#include <stdio.h>
#include "ioutil.h"

/* generic routines that keep reading until an error occurs or until
   all of the requested chars have been retrieved. */

ssize_t readn(int fd, char * buf, size_t count)
{
  size_t nleft = count;
  ssize_t nread;
  char * ptr = buf;

  while (nleft > 0) {
    if ((nread = read(fd, ptr, nleft)) < 0) {
      if (errno == EINTR)
	nread = 0;
      else
	return -1;
    }
    else if (nread == 0) /* EOF */
      break;
    nleft -= nread;
    ptr += nread;
  }
  return (count - nleft); /* return >= 0 */
}

ssize_t writen(int fd, const char * buf, size_t count)
{
  size_t nleft = count;
  ssize_t nwritten;
  const char *ptr = buf;

  while (nleft > 0) {
    if ((nwritten = write(fd, ptr, nleft)) <= 0) {
      if (errno == EINTR)
	nwritten = 0;
      else
	return -1;
    }
    nleft -= nwritten;
    ptr += nwritten;
  }
  return count;
}

#define MAXLINE 1024

struct readstate {
  int read_cnt;
  char * read_ptr;
  char read_buf[MAXLINE];
};

static ssize_t my_read(int fd, char * ptr, struct readstate * rs)
{
  if (rs->read_cnt <= 0) {
  again:
    if ((rs->read_cnt = read(fd, rs->read_buf, MAXLINE)) < 0) {
      if (errno == EINTR)
	goto again;
      return -1;
    }
    else if (rs->read_cnt == 0)
      return 0;
    rs->read_ptr = rs->read_buf;
  }
  rs->read_cnt--;
  *ptr = *(rs->read_ptr)++;
  return 1;
}

static struct readstate rs;

ssize_t readline(int fd, char * buf, size_t maxlen)
{
  int n, rc;
  char c = 0, * ptr = buf;
  
  for (n = 1; n < maxlen; n++) { /* loop starts at 1 */
    if ((rc = my_read(fd, &c, &rs)) == 1) {
      *ptr++ = c;
      if (c == '\n')
	break; /* newline is stored, like fgets() */
    }
    else if (rc == 0) {
      if (n == 1)
	return 0;  /* EOF, no data read */
      else
	break;     /* EOF, some data was read */
    }
    else
      return -1;   /* error; errno was set by read() */
  }
  *ptr = 0; /* null terminate like fgets() */
  return n;
}
