#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/select.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <assert.h>
#include "socket.h"
#include "packet.h"
#include "ioutil.h"
#include "timeval_util.h"

#ifndef MAX
#define MAX(x,y) (x) > (y) ? (x) : (y)
#endif

/* default parameters */
#define BUFLEN 65535
static unsigned short outport = 5001;
static unsigned short inport = 5000;
static int bufsz = 64;
static int nbufs = 10;

// send outgoing pkt
int sendPkt(struct timeval *tm, int outfd, char * buf, int sz) {
  int ret;

  gettimeofday(tm,NULL);
  ret = sendPacket(outfd,buf,0,sz);
  if (ret <= 0) {
    if (ret == 0 || errno == EPIPE || errno == ECONNABORTED) /* reset */
      return 0;
    else {
      fprintf(stderr,"Errno = %d\n",errno);
      perror("sendPacket");
      exit(1);
    }
  }
  return 1;
}

// read incoming packet
char * readPkt(struct timeval *tm, int infd, int *sz) {
  char * inbuf = NULL;
  int charsRead = 0;
  int ret;
  do {
    ret = readPacket(infd,&inbuf,sz,charsRead);
    gettimeofday(tm,NULL);
    if (ret == 0)
      return NULL;
    else if (ret < 0) {
      perror("readPacket");
      exit(1);
    }
    charsRead += ret;
  } while (charsRead != *sz);

  return inbuf;
}

void usage(const char *argv0) {
  fprintf(stderr,"usage: %s [-s n sz] [-r] [-p send-port] [-r recv-port] [host]\n  -s n sz   ping sender: send n packets (def=%d) of sz bytes, n>0, 0<sz<%d (def=%d)\n  -r        ping receiver\n  -p port   port to send outgoing packets on (def=%d)\n  -i port   port to receive return packets on (def=%d)\n",argv0, nbufs, BUFLEN, bufsz, outport, inport);
  exit(1);
}

int main(int argc, char **argv) {
  char defhost[10] = 
    { 'l', 'o', 'c', 'a', 'l', 'h', 'o', 's', 't', '\0' };
  char * host = defhost;
  char buf[BUFLEN];
  int outfd, infd;
  struct timeval starttm, endtm, delta;
  int do_sender = 0, do_receiver = 0;
  int i, argn, state;
  struct sockaddr_in sa;
  int szbuf;

  for (i=0; i<BUFLEN; i++) buf[i] = (i % 26) + 'A';

  argn = 1;
  while ( argn < argc && argv[argn][0] == '-' )  {
    if ( strcmp( argv[argn], "-?" ) == 0 )
      usage(argv[0]);
    else if ( strcmp( argv[argn], "-s" ) == 0 && argn + 2 < argc ) {
      do_sender = 1;
      ++argn;
      nbufs = atoi(argv[argn]);
      ++argn;
      bufsz = atoi(argv[argn]);
      if (nbufs <= 0 || bufsz <= 0 || bufsz > BUFLEN)
	usage(argv[0]);
    }
    else if ( strcmp( argv[argn], "-p" ) == 0 && argn + 1 < argc ) {
      ++argn;
      outport = (unsigned short)atoi(argv[argn]);
    }
    else if ( strcmp( argv[argn], "-i" ) == 0 && argn + 1 < argc ) {
      ++argn;
      inport = (unsigned short)atoi(argv[argn]);
    }
    else if ( strcmp( argv[argn], "-r" ) == 0 ) {
      do_receiver = 1;
    }
    else
      usage(argv[0]);
    ++argn;
  }
  if ( argn != argc )
    if ( argn == (argc-1) ) {
      host = argv[argn];
    }
    else
      usage(argv[0]);

  if ((!do_sender && !do_receiver))
    do_sender = 1;
  else if (do_sender && do_receiver) {
    fprintf(stderr,"Must choose either -s (default) or -r, but not both\n");
    usage(argv[0]);
  }

  signal(SIGPIPE, SIG_IGN);		/* get EPIPE instead */

  /* keep reading lines from stdin until we hit EOF, and write
     each line as a frame on the connection. */
  
#define Do_Sender_Outport 0
#define Do_Sender_Inport  1
#define Done       2
    /* open channels */ 
    for (state = Do_Sender_Outport; state != Done; ++state) {
      int fd;
      int port = 
	state == Do_Sender_Outport ? 
	(do_sender ? outport : inport) : (do_sender ? inport : outport);
      if (do_sender) {
	/* connect on outgoing port */
	fprintf(stderr,"connecting to host %s:%d\n", host, port);
	fd = persistentOpen(port,host,0);
	fprintf(stderr,"connected\n");
      }
      else {
	int listenfd, ret;
	/* listen on incoming port first */
	fprintf(stderr,"Listening on address %s:%d\n", host, port);
	ret = OpenListen(port,host,0,&listenfd);
	if (ret == -1) {
	  fprintf(stderr,"failed to open listen\n");
	  exit(1);
	}
	fprintf(stderr,"waiting for connection\n");
	fd = RespondAccept(listenfd);
	if (fd == -1) {
	  fprintf(stderr,"failed to accept\n");
	  exit(1);
	}
	fprintf(stderr,"connected\n");
	close(listenfd);
      }
      if (state == Do_Sender_Outport)
	if (do_sender) outfd = fd;
	else infd = fd;
      else
	if (do_sender) infd = fd;
	else outfd = fd;
    }

    /* do timed pings */
    while (1) {
      if (do_sender) {
	char * inbuf;
	int sz;

	/* set up the outgoing buffer */
	if (!nbufs) break;
	--nbufs;
    
	/* send it out */
	if (!sendPkt(&starttm, outfd, buf, bufsz)) break;

	/* wait for the response */
	inbuf = readPkt(&endtm,infd,&sz);
	if (inbuf == NULL) break;
      
	/* print the difference in time */
	subTimeval(&endtm,&starttm,&delta);
	printf("%d bytes %d usecs\n", sz, 
	       delta.tv_sec * 1000000 + delta.tv_usec);
	free(inbuf);
      }
      else {
	/* wait for first packet */
	char * inbuf;
	int sz;

	inbuf = readPkt(&endtm,infd, &sz);
	if (inbuf == NULL) break;
	
	/* turn it around */
	if (!sendPkt(&starttm, outfd, inbuf, sz)) break;
	free(inbuf);
      }
    }

    /* clean up */
    close(outfd);
    close(infd);

  return 0;
}
