/*------------------------------------------------------------------------
 * Usage : mpgsender hostname port inputMPG
 *
 * Decodes an MPEG video dumps the frames it finds one by one to the
 *   destination host on the given TCP port.  Each GOP header is put in
 *   a separate frame (as well as the SEQ header).
 *
 *------------------------------------------------------------------------
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/fcntl.h>
#include "dvmbasic.h"
#include "dvmmpeg.h"
#include "dvmcolor.h"
#include "dvmpnm.h"
#include "bitparser.h"
#include "socket.h"

/*
 * operations on timevals for setting up the timing properly
 */

/* for comparing timevals (tv1 < tv2 -> -1, tv1 = tv2 -> 0, tv1 > tv2 -> 1) */
int timevalCmp(struct timeval *tv1, struct timeval *tv2) 
{
  if (tv2 == NULL) return -1; /* special case */

  if (tv1->tv_sec > tv2->tv_sec) return 1;
  else if (tv1->tv_sec == tv2->tv_sec)
    if (tv1->tv_usec > tv2->tv_usec) return 1;
    else if (tv1->tv_usec == tv2->tv_usec) return 0;
    else return -1;
  else return -1;
}

/* [tv1] - [tv2] = [res] */
void subTimeval(struct timeval *tv1, struct timeval *tv2, 
		struct timeval *res)
{
  if (tv1 == NULL) { /* special case */
    res->tv_sec = 0;
    res->tv_usec = 0;
  }
  else {
    res->tv_usec = tv1->tv_usec - tv2->tv_usec;
    res->tv_sec = tv1->tv_sec - tv2->tv_sec;
    if (res->tv_usec < 0) {
      res->tv_usec += 1000000;
      res->tv_sec -= 1;
    }
  }
}

/* for adding < 1 sec to a timeval */
void incTimeval(struct timeval *tv, unsigned int usec) 
{
  if (usec >= 1000000) {
    fprintf(stderr,"incTimeval: increment %d too large\n",usec);
    return;
  }
  tv->tv_usec += usec;
  while (tv->tv_usec >= 1000000) {
    tv->tv_sec++;
    tv->tv_usec -= 1000000;
  }
}

static void printTimeval(struct timeval *tv)
{
  printf("%d.%06d", tv->tv_sec, tv->tv_usec);
}

extern int sendPacket(int fd, unsigned char *s, int ofs, int len);
extern void int2buf(int x, char *buf);

static unsigned short port = 5000;
static char defhost[10] = 
  { 'l', 'o', 'c', 'a', 'l', 'h', 'o', 's', 't', '\0' };
static char * host = defhost;

static unsigned int bytesSent = 0;

/* whether to use non-blocking sockets for testing purposes */
static int nonBlock = 0;

/* whether dumping each frame to a file */
static int dumpFrame = 0;
static char *name = "";
/* opens file for the frame */
static FILE *openFrame() {
  char frameNameBuf[500];
  FILE *fp;

  sprintf(frameNameBuf,"dumpedFrames/fofs%08d-%s",bytesSent,name);
  fp = fopen(frameNameBuf,"wb");
  if (fp == NULL) {
    fprintf(stderr, "unable to open %s for writing\n", frameNameBuf);
    exit(1);
  }
  return fp;
}

#define FRAME_BUF_SIZE 25000
static unsigned char framebuf[FRAME_BUF_SIZE];
static unsigned int seqnum = 0;

/* set to true if receiver can't keep up, and wants to reconnect */
static int waitForClose = 0;
void sig_urg(int signo) {
  printf("SIGURG received!\n");
  waitForClose = 1;
}

/* sends a "flush" packet to the upstream node, and then waits
   for it to close the connection.  Then it reconnects and
   continues.  This really isn't fair at the moment, since the
   movie should keep playing ...
*/
static int closeAndReopen(int fd) {
  int r;
  char buf[1];
  int newfd;
  struct timeval t, u;

  /* send the flush */
  r = sendPacket(fd,"",0,0);
  printf("sent flush packet\n");

/*   /\* wait for the close and reconnect *\/ */
/*   r = read(fd,buf,1); */
/*   if (r == 1) { */
/*     fprintf(stderr,"Oops! read data!\n"); */
/*     exit(1); */
/*   } */
/*   close(fd); */
  gettimeofday(&t,NULL);
  printf("%d.%06d: connecting to host %s:%d\n", 
	 t.tv_sec % 1000, t.tv_usec, host, port);
  fflush(stdout);

  newfd = persistentOpen(port,host,nonBlock);
  signal(SIGURG, sig_urg);
  fcntl(newfd,F_SETOWN,getpid());

  gettimeofday(&u,NULL);
  subTimeval(&u,&t,&t);
  printf("%d.%06d: connected (fd = %d) elapsed time %d.%06d seconds\n", 
	 u.tv_sec % 1000, u.tv_usec, newfd,
	 t.tv_sec % 1000, t.tv_usec);
  fflush(stdout);

  return newfd;
}  

int SendFrame(int fd, unsigned char *buf, int bufsize, int add_seq) {
  int ret, ofs = 0;
  FILE *fp = NULL;

  /* if receiver is reconfiguring, we'll pause so it can keep up */
  if (waitForClose) {
    fd = closeAndReopen(fd);
    waitForClose = 0;
  }

  if (add_seq) {
    if (bufsize > (FRAME_BUF_SIZE-4)) {
      fprintf(stderr,"Can't add seqno: buffer size is too large\n");
      exit(1);
    }
    memcpy(framebuf+4,buf,bufsize);
    int2buf(seqnum++,framebuf);
    buf = framebuf;
    bufsize += 4;
    printf("[%d] ",seqnum-1);
  }

  printf("sending packet ");
  if (dumpFrame)
    fp = openFrame();
  bytesSent += bufsize;
  while (1) {
    printf("(%d) ",bufsize);
    ret = sendPacket(fd,buf,ofs,bufsize);
    if (ret <= 0)
      if (nonBlock && (errno == EWOULDBLOCK || errno == EAGAIN)) {
	fprintf(stderr,"would block!\n");
	exit(1);
      }
/*        else if (ret == 0 || errno == EPIPE || errno == ECONNABORTED) */
/*  	goto reset; */
      else {
	struct timeval t;

  	printf("\n");
	fprintf(stderr,"Errno = %d, fd = %d\n",errno, fd);
	perror("sendPacket");
/*        reset: */
    	close(fd);

	gettimeofday(&t,NULL);
	printf("%d.%06d: connecting to host %s:%d\n", 
	       t.tv_sec % 1000, t.tv_usec, host, port);
	fflush(stdout);

  	fd = persistentOpen(port,host,nonBlock);
	signal(SIGURG, sig_urg);
	fcntl(fd,F_SETOWN,getpid());

	gettimeofday(&t,NULL);
	printf("%d.%06d: connected (fd = %d)\n", 
	       t.tv_sec % 1000, t.tv_usec, fd);

	if (ofs != 0) { /* sent partial packet */
	  fprintf(stderr,"Dropping remainder of packet\n");
	  break;
	}
	else
	  continue;

/*    	exit(1); */
      }
    else if (ret != bufsize) { /* didn't write it all */
      if (fp != NULL) 
	fwrite( buf+ofs, 1, ret, fp );
      bufsize -= ret;
      ofs += ret;
      continue;
    }
    else {
      if (fp != NULL)
	fwrite( buf+ofs, 1, ret, fp );
      printf("\n");
      break;
    }
  }
  if (fp != NULL)
    fclose(fp);
  return fd;
}

/*
 * This proc make sure that there are at least size bytes of 
 * data in the bitstream bs, which is attached to bitparser bp.  
 * If there is not enough data, fill up the bitstream by reading 
 * from tcl channel chan.
 */
void CheckBitStreamUnderflow (bs, bp, chan, size)
    BitStream *bs;
    BitParser *bp;
    FILE *chan;
    int size;
{
    int off  = BitParserTell(bp);
    int left = BitStreamBytesLeft(bs, off);
    if (left < size) {
        BitStreamShift (bs, off);
        BitStreamFileRead (bs, chan, left);
        BitParserSeek (bp, 0);
    }
}

/* statistics about stream being sent */
static int bytes[3] = { 0, 0, 0 };
static int num[3] = { 0, 0, 0 };
static int numGopHdrs = 0, numPics = 0;

/* information about frame sizes of the given MPEG stream */
static int halfw, halfh, w, h, picSize;
static int mbw, mbh;

/*
 * Creates a P-frame that is based on, and duplicates, the given
 * I-frame.  The bitparser bp is situated at the beginning of
 * the frame data, and the header has already been read.  The result
 * is returned by reference in buf and bufsz.
 */
//  #define FORWARD_F_CODE      3
//  int MakeDeadFrame(BitStream *bs, BitParser *bp, MpegSeqHdr *sh,
//  		  MpegPicHdr *IpicHdr, BitStream *obs)
//  {
//    int sliceInfo[] = {1000};       /* this should be enough for one frame */
//    int sliceInfoLen = 1;
//    int gop, pic, temporalRef, bufsz;

//    ByteImage *y, *u, *v;
//    ByteImage *qScale;
//    ScImage *scY, *scU, *scV;
//    VectorImage *fmv;                       /* forward motion vector */

//    MpegPicHdr *picHdr = MpegPicHdrNew();
//    BitParser *obp = BitParserNew();        /* output bitstream */
//    BitParserWrap(obp, obs);

//    y       = ByteNew(w, h);
//    u       = ByteNew(halfw, halfh);
//    v       = ByteNew(halfw, halfh);
//    qScale  = ByteNew(mbw, mbh);
//    scY     = ScNew(mbw*2, mbh*2);
//    scU     = ScNew(mbw, mbh);
//    scV     = ScNew(mbw, mbh);
//    fmv     = VectorNew(mbw, mbh);
//    ByteSet(qScale, 4);

//    /* set up Pic Header */
//    MpegPicHdrSetVBVDelay(picHdr, 0);
//    MpegPicHdrSetFullPelBackward(picHdr, 0);
//    MpegPicHdrSetBackwardFCode(picHdr, 0);
//    MpegPicHdrSetTemporalRef(picHdr, temporalRef);
//    MpegPicHdrSetType(picHdr, P_FRAME);
//    MpegPicHdrSetFullPelForward(picHdr, 1);
//    MpegPicHdrSetForwardFCode(picHdr, FORWARD_F_CODE);

//    MpegPicHdrEncode (picHdr, obp);

//    MpegPicIParse(bp, sh, IpicHdr, scY, scU, scV);
//    ScIToByte(scY, y);
//    ScIToByte(scU, u);
//    ScIToByte(scV, v);

//    BytePMotionVecSearch(picHdr, y, y, NULL, fmv);
//    ByteYToScP (y, y, fmv, qScale, MPEG_INTRA, MPEG_NON_INTRA, scY);
//    ByteUVToScP(u, u, fmv, qScale, MPEG_INTRA, MPEG_NON_INTRA, scU);
//    ByteUVToScP(v, v, fmv, qScale, MPEG_INTRA, MPEG_NON_INTRA, scV);
//    MpegPicPEncode (picHdr, scY, scU, scV, fmv, qScale, sliceInfo, sliceInfoLen, obp);

//    bufsz = BitParserTell(obp);

//    /* lots of things to free up */
//    BitParserFree(obp);
//    MpegPicHdrFree(picHdr);
//    ByteFree(y);
//    ByteFree(u);
//    ByteFree(v);
//    ByteFree(qScale);
//    ScFree(scY);
//    ScFree(scU);
//    ScFree(scV);
//    VectorFree(fmv);

//    return bufsz;
//  }

/*
 * Creates a forward motion vector that essentially duplicates the
 * prior frame. 
 */
static VectorImage *nullVector(int vecw, int vech) {
  VectorImage *fmv;
  int i, num = vecw * vech;

  fmv = VectorNew(vecw, vech);
  for (i=0; i<num; i++) {
    fmv->firstVector[i].exists = 1;
    fmv->firstVector[i].down = 0;
    fmv->firstVector[i].right = 0;
  }
  return fmv;
}

/*
 * Creates an "empty" scImage the depends entirely on the prior frame.
 */
static ScImage *nullScImage(int scw, int sch) {
  ScImage *sc;
  int i, num = scw * sch;

  sc = ScNew(scw, sch);
  for (i=0; i<num; i++) {
    sc->firstBlock[i].intracoded = 0;
    sc->firstBlock[i].skipMB = 0;
    sc->firstBlock[i].skipBlock = 1;
  }
  return sc;
}

/*
 * Creates a "dead" P-frame of the given size.
 */
#define FORWARD_F_CODE      3
int MakeDeadFrameX(BitStream *obs)
{
  int sliceInfo[] = {1000};       /* this should be enough for one frame */
  int sliceInfoLen = 1;
  int gop, pic, temporalRef, bufsz;

  ByteImage *qScale;
  ScImage *scY, *scU, *scV;
  VectorImage *fmv;                       /* forward motion vector */

  MpegPicHdr *picHdr = MpegPicHdrNew();
  BitParser *obp = BitParserNew();        /* output bitstream */
  BitParserWrap(obp, obs);

  /* Create the underlying data for the "dead" frame */
  scY     = nullScImage(mbw*2, mbh*2);
  scU     = nullScImage(mbw, mbh);
  scV     = nullScImage(mbw, mbh);
  fmv     = nullVector(mbw, mbh);
  qScale  = ByteNew(mbw, mbh);
  ByteSet(qScale, 4);

  /* set up Pic Header */
  MpegPicHdrSetVBVDelay(picHdr, 0);
  MpegPicHdrSetFullPelBackward(picHdr, 0);
  MpegPicHdrSetBackwardFCode(picHdr, 0);
  MpegPicHdrSetTemporalRef(picHdr, 0);
  MpegPicHdrSetType(picHdr, P_FRAME);
  MpegPicHdrSetFullPelForward(picHdr, 1);
  MpegPicHdrSetForwardFCode(picHdr, FORWARD_F_CODE);

  /* Encode it */
  MpegPicHdrEncode (picHdr, obp);
  MpegPicPEncode (picHdr, scY, scU, scV, fmv, qScale, sliceInfo, sliceInfoLen, obp);

  bufsz = BitParserTell(obp);

  /* lots of things to free up */
  BitParserFree(obp);
  MpegPicHdrFree(picHdr);
  ByteFree(qScale);
  ScFree(scY);
  ScFree(scU);
  ScFree(scV);
  VectorFree(fmv);

  return bufsz;
}

int main(int argc, char **argv) {
  BitParser *bp = BitParserNew();
  BitStream *bs = BitStreamNew(65536);
#define BUFFER_SIZE         5000
  BitStream *deadFrameS = BitStreamNew(BUFFER_SIZE);
  BitParser *deadFrameP = BitParserNew();
  int deadFrameSz = 0;
  MpegSeqHdr *sh;
  MpegPicHdr *fh;

  FILE *file;
  FILE *framefp;

  unsigned char * framestart;
  int ret, fd, sz, type;
  int currCode, len, temporalRef;

  unsigned int usec_frame_delay;
  struct timeval nextDeadline, currentTime;

  int repeatn = 1, dataStart = 0;
  int dropB = 0, dropP = 0, addSeq = 0;
  int gotDeadFrame = 0;
  int bufsz;
  int bufszsz = sizeof(bufsz);

  /**** parse arguments ****/

  gettimeofday(&currentTime,NULL);
  printf("%d.%06d: starting\n", 
	 currentTime.tv_sec % 1000, currentTime.tv_usec);
  fflush(stdout);

  if (argc != 5 && argc != 6) {
    fprintf(stderr,"usage: %s hostname port mpgfile BPseq [num repeats]\n",argv[0]);
    exit(1);
  }

  host = argv[1];
  port = (unsigned short)atoi(argv[2]);
  if (strlen(argv[4]) < 3) {
    fprintf(stderr, "Flags not correct\n");
    exit(1);
  }    
  dropB = (argv[4][0] == '1');
  dropP = (argv[4][1] == '1');
  addSeq = (argv[4][2] == '1');
  file = fopen(argv[3], "rb");
  if (file == NULL) {
    fprintf(stderr, "unable to open %s for reading.\n", argv[3]);
    exit(1);
  }
  name = argv[3];
  if (argc == 6) {
    repeatn = atoi(argv[5]);
    if (repeatn <= 0)
      fprintf(stderr,"will repeat forever\n");
    else
      fprintf(stderr,"repeatn set to %d\n",repeatn);
  }

  signal(SIGPIPE, SIG_IGN);		/* get EPIPE instead */

  /**** connect to remote host ****/

  gettimeofday(&currentTime,NULL);
  printf("%d.%06d: connecting to host %s:%d\n", 
	 currentTime.tv_sec % 1000, currentTime.tv_usec, host, port);
  fflush(stdout);

  fd = persistentOpen(port,host,nonBlock);
  signal(SIGURG, sig_urg);
  fcntl(fd,F_SETOWN,getpid());

  /* print out the send buffer size */
  if (getsockopt(fd, SOL_SOCKET, SO_SNDBUF, &bufsz, &bufszsz) < 0) {
    perror("setsockopt SO_SNDBUF");
    fprintf(stderr,"errno=%d\n",errno);
    exit(1); /* XXX */
  }
  else
    fprintf(stderr,"snd buf is size %d\n",bufsz);

  gettimeofday(&currentTime,NULL);
  printf("%d.%06d: connected (fd = %d)\n", 
	 currentTime.tv_sec % 1000, currentTime.tv_usec, fd);
    
  /**** read mpeg data ****/

  BitStreamFileRead(bs, file, 0);
  BitParserWrap(bp, bs);

  /*
   * Allocate a new sequence header, skips the initial garbage (if any)
   * in the input file, and read in the sequence header.
   */
  sh = MpegSeqHdrNew();
  MpegSeqHdrFind(bp);
  framestart = bp->offsetPtr;
  MpegSeqHdrParse(bp, sh);
  
  /*
   * Find the width and height of the video frames.  If the width
   * and height are not multiple of 16, round it up to the next 
   * multiple of 16.
   */
  w = MpegSeqHdrGetWidth(sh);
  h = MpegSeqHdrGetHeight(sh);
  picSize = MpegSeqHdrGetBufferSize(sh);

//    remw = seqw % 16;
//    remh = seqh % 16;
//    if (remw != 0) {
//      w = seqw + 16 - remw;
//    } else {
//      w = seqw;
//    }
//    if (remh != 0) {
//      h = seqh + 16 - remh;
//    } else {
//      h = seqh;
//    }

  halfw = w/2;
  halfh = h/2;
  mbw = (w + 15) / 16;
  mbh = (h + 15) / 16;

  printf("MPEG picsize %d, dimensions %d x %d (H x W in pixels)\n",
	 picSize, h, w);
  printf("     pic-rate %f, bitrate %f Mbps\n",
	 MpegSeqHdrGetPicRate(sh), MpegSeqHdrGetBitRate(sh) * 400 / 1024.0);
  usec_frame_delay = (unsigned int)(1000000.0 / MpegSeqHdrGetPicRate(sh));

  /**** send the seq header frame ****/

  sz = bp->offsetPtr - framestart;
  fd = SendFrame(fd, framestart, sz, addSeq);
  dataStart = bp->offsetPtr - bp->bs->buffer; 

  /**** start the timer (send frames with correct spacing) ****/

  gettimeofday(&nextDeadline, NULL);
  incTimeval(&nextDeadline, usec_frame_delay);

  /* now just keep going through, and print out the headers types
     that are found */
  fh = MpegPicHdrNew();
  do {
    CheckBitStreamUnderflow(bs, bp, file, picSize);
    currCode = MpegGetCurrStartCode(bp);
    switch (currCode) {
    case PIC_START_CODE:
      { 
	int dropit, i;

	framestart = bp->offsetPtr;
	MpegPicHdrParse(bp, fh);
	type = MpegPicHdrGetType(fh);
	if (type == I_FRAME) {
	  printf("I");
	  if (!gotDeadFrame) {
	    int currPos = BitParserTell(bp);
//  	    deadFrameSz = MakeDeadFrame(bs,bp,sh,fh,deadFrameS);
  	    deadFrameSz = MakeDeadFrameX(deadFrameS);
	    BitParserSeek(bp,currPos);
	    gotDeadFrame = 1;
	  }
	  dropit = 0;
	  temporalRef = fh->temporal_reference;
	} else if (type == P_FRAME) {
	  printf("P");
	  if (dropP) 
	    dropit = 1;
	  else {
	    dropit = 0;
	    temporalRef = fh->temporal_reference;
	  }
	} else {
	  printf("B");
	  if (dropB) 
	    dropit = 1;
	  else {
	    dropit = 0;
	    temporalRef = fh->temporal_reference;
	  }
	}

	num[type]++; bytes[type] += sz;
	MpegPicSkip(bp);

	sz = bp->offsetPtr - framestart;
	numPics++;

	gettimeofday(&currentTime, NULL);
	if (timevalCmp(&currentTime,&nextDeadline) < 0) {
	  /* wait until deadline before continuiing */
	  subTimeval(&nextDeadline, &currentTime, &currentTime);
	  select(0,NULL,NULL,NULL,&currentTime);
	}
	if (dropit) {
//  	  temporalRef++;
//  	  /* update the temporal ref for the frame */
//  	  BitParserWrap(deadFrameP, deadFrameS);
//  	  Bp_PutInt(deadFrameP, PIC_START_CODE); // XXX redundant
//  	  Bp_PutBits(deadFrameP, 10, temporalRef);
//  	  SendFrame(fd,deadFrameS->buffer,deadFrameSz);
	}
	else
	  fd = SendFrame(fd,framestart,sz,addSeq);
	incTimeval(&nextDeadline, usec_frame_delay);
      }
      break;
      
    case GOP_START_CODE:
      { 
	framestart = bp->offsetPtr;
	MpegGopHdrSkip(bp);
	sz = bp->offsetPtr - framestart;	
	printf("\nG");
	numGopHdrs++;
	fd = SendFrame(fd,framestart,sz,addSeq);
      }
      break;
      
    case SEQ_START_CODE:
      printf("Got SEQ_START_CODE---no good\n");
    case PACK_START_CODE:
      printf("Got PACK_START_CODE\n");
    case SYS_START_CODE:
      printf("Got SYS_START_CODE\n");
    case EXT_START_CODE:
      printf("Got EXT_START_CODE\n");
    case USER_START_CODE:
      printf("Got USER_START_CODE\n");
    case SLICE_MIN_START_CODE:
      printf("Got SLICE_MIN_START_CODE\n");
    case SLICE_MAX_START_CODE:
      printf("Got SLICE_MAX_START_CODE\n");
    case PACKET_MIN_START_CODE:
      printf("Got PACKET_MIN_START_CODE\n");
    case PACKET_MAX_START_CODE:
      printf("Got PACKET_MAX_START_CODE\n");
      exit(1);
      
    case SEQ_END_CODE:
      printf("\n");
      break;
      
    case ISO_11172_END_CODE:
      printf("Got ISO_11172_END_CODE---expecting start code\n");
      exit(1);
      
    default:
      printf("Got unknown code %#x\n", currCode);
      exit(1);
    }
    if (currCode == SEQ_END_CODE) {
      printf("==== repeating\n");
      repeatn--;
      if (repeatn == 0) /* done */
	break;
      else { /* reset and send again */
	if (fseek(file,dataStart,SEEK_SET) != 0) {
	  fprintf(stderr,"Failed to seek to beginning of file\n");
	  exit(1);
	}
	BitStreamFileRead(bs, file, 0);
	BitParserWrap(bp, bs);
      }
    }
    
  } while (1);
  
  MpegSeqHdrFree(sh);
  BitStreamFree(bs);
  BitParserFree(bp);
  
  printf("%d frames, %d GOP headers, %d bytes sent\n",
	 numPics,numGopHdrs,bytesSent);
  printf("%d I-frames (avg %d bytes)\n%d P-frames (avg %d bytes)\n%d B-frames (avg %d bytes)\n",num[I_FRAME],bytes[I_FRAME] / num[I_FRAME], num[P_FRAME], bytes[P_FRAME] / num[P_FRAME], num[B_FRAME], bytes[B_FRAME] / num[B_FRAME]);

  return 0;
}
