#include <stdio.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#ifndef FALSE
#define FALSE 0
#endif

/* ---------------------------------------------------------------- */
struct sockaddr_in GetAddr(unsigned short port, char *hostname) 
{
  struct hostent* he;
  struct sockaddr_in sa;

  sa.sin_family = AF_INET;
  if (strlen(hostname) == 0)
    sa.sin_addr.s_addr = htonl(INADDR_ANY);
  else {
    if (inet_aton(hostname, &sa.sin_addr) == 0) /* invalid */ {
      he = gethostbyname(hostname);
      if (he == NULL) {
	perror("gethostbyname");
	exit(1);
      }
      if (he->h_addrtype != AF_INET ||
	  he->h_length != sizeof(sa.sin_addr)) {
	fprintf(stderr, "%.80s - non-IP network address\n", hostname);
	exit(1);
      }
      sa.sin_addr = *(struct in_addr *)he->h_addr;
    }
  }
  sa.sin_port = htons(port);

  return sa;
}

static int setNonBlock(int fd) {
  int res;
#ifdef __CYGWIN__
  //  #define FIONBIO 0x8004667e /* To be compatible with termiost version */
  int nonblocking = 1;
  res = ioctl(fd, FIONBIO, &nonblocking);
#else
  res = fcntl(fd, F_GETFL, 0);
  if (res < 0) {
    perror("fcntl(GETFL)");
    return -1;
  }
  res = fcntl(fd, F_SETFL, res | (long)O_NONBLOCK);
#endif
  return res;
}

int OpenSocket(int nonBlock)
{
  int fd;
 
  /* Create socket. */
  fd = socket(AF_INET, SOCK_STREAM, 0);
  if (fd < 0) {
    perror("creating accept socket");
    return -1;
  }
  
  if (nonBlock) {
    /* Set the accept socket file descriptor to no-delay mode. */
    int res;
    res = setNonBlock(fd);
    if (res < 0) {
      perror("can't get accept into no-delay");
      return -1;
    }
  }
  
  return fd;
}

int OpenConnect(unsigned short port, char *hostname, int nonBlock, int *retfd)
{
  int fd;
  struct sockaddr_in sa;

/*    fprintf(stderr,"opening connection %s:%d\n",hostname,port); */

  fd = OpenSocket(nonBlock);
  if (fd == -1) {
    *retfd = -1;
    return -1;
  }
  else {
    sa = GetAddr(port,hostname);
    if (connect(fd, (struct sockaddr *)&sa, (socklen_t)sizeof(sa)) < 0) {
      *retfd = fd;
      return -1;
    }
    *retfd = fd;
    return fd;
  }
}

int persistentOpen(unsigned short port, char *host, int nonBlock) {
  int retrying = 0;
  while (1) {
    int fd;
    if (OpenConnect(port,host,FALSE,&fd) != -1) {
      if (nonBlock) {
	int res = setNonBlock(fd);
	if (res < 0) {
	  perror("can't get opened fd into no-delay");
	  return -1;
	}
      }
      return fd;
    }
    else {
      if (!retrying) {
	perror("Connect failed (retrying):");
	retrying = 1;
      }
      close(fd);
      continue;
    }
  }
}

int OpenListen(unsigned short port, char * hostname, int nonBlock, int *retfd)
{
  int fd;
  int on;
  int i;
  struct sockaddr_in sa;

  fd = OpenSocket(nonBlock);
  if (fd == -1) {
    *retfd = -1;
    return -1;
  }
  else {

    /* Allow reuse of local addresses. */
    on = 1;
    if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, 
		   &on, (socklen_t)(sizeof(on))) < 0) {
      perror("setsockopt SO_REUSEADDR");
      exit(1); /* XXX */
    }
  
    /* Bind to it. */
    sa = GetAddr(port,hostname);
    if (bind(fd, (struct sockaddr *)&sa, sizeof(sa)) < 0) {
      perror("binding accept socket");
      *retfd = fd;
      return -1;
    }
  
    /* Start a listen going. */
    if (listen(fd, 1024) < 0) {
      perror("listen on accept socket");
      *retfd = fd;
      return -1;
    }
    return fd;
  }
}

int RespondAccept(int fd) 
{
  struct sockaddr sin;
  socklen_t sz = sizeof(sin);
  int newConnFD;

  printf("doing accept\n");
  newConnFD = accept(fd, &sin, &sz);
  printf("did accept\n");
  if (newConnFD < 0) {
    perror("accept");
    return -1;
  }

  /* we assume that accept() inherits the O_NDELAY
     settings, so we don't need to do it ourselves
     if (fcntl(hc->fd, F_SETFL, O_NDELAY) < 0)
     syslog(LOG_ERR, "fcntl O_NDELAY - %m");
     */
  
  return newConnFD;
}
