#include "dvmbasic.h"
#include "dvmmpeg.h"
#include "bitparser.h"

/*
 * Creates a forward motion vector that essentially duplicates the
 * prior frame. 
 */
static VectorImage *nullVector(int vecw, int vech) {
  VectorImage *fmv;
  int i, num = vecw * vech;

  fmv = VectorNew(vecw, vech);
  for (i=0; i<num; i++) {
    fmv->firstVector[i].exists = 1;
    fmv->firstVector[i].down = 0;
    fmv->firstVector[i].right = 0;
  }
  return fmv;
}

/*
 * Creates an "empty" scImage the depends entirely on the prior frame.
 */
static ScImage *nullScImage(int scw, int sch) {
  ScImage *sc;
  int i, num = scw * sch;

  sc = ScNew(scw, sch);
  for (i=0; i<num; i++) {
    sc->firstBlock[i].intracoded = 0;
    sc->firstBlock[i].skipMB = 0;
    sc->firstBlock[i].skipBlock = 1;
  }
  return sc;
}

/*
 * Creates a "dead" P-frame of the given size.
 */
#define FORWARD_F_CODE      3
static int MakeDeadFrame(int h, int w, BitStream *obs)
{
  int sliceInfo[] = {1000};       /* this should be enough for one frame */
  int sliceInfoLen = 1;
  int gop, pic, temporalRef, bufsz;
  int mbw, mbh;

  ByteImage *qScale;
  ScImage *scY, *scU, *scV;
  VectorImage *fmv;                       /* forward motion vector */

  MpegPicHdr *picHdr = MpegPicHdrNew();
  BitParser *obp = BitParserNew();        /* output bitstream */
  BitParserWrap(obp, obs);

  /* Macroblock sizes */
  mbw = (w + 15) / 16;
  mbh = (h + 15) / 16;

  /* Create the underlying data for the "dead" frame */
  scY     = nullScImage(mbw*2, mbh*2);
  scU     = nullScImage(mbw, mbh);
  scV     = nullScImage(mbw, mbh);
  fmv     = nullVector(mbw, mbh);
  qScale  = ByteNew(mbw, mbh);
  ByteSet(qScale, 4);

  /* set up Pic Header */
  MpegPicHdrSetVBVDelay(picHdr, 0);
  MpegPicHdrSetFullPelBackward(picHdr, 0);
  MpegPicHdrSetBackwardFCode(picHdr, 0);
  MpegPicHdrSetTemporalRef(picHdr, 0);
  MpegPicHdrSetType(picHdr, P_FRAME);
  MpegPicHdrSetFullPelForward(picHdr, 1);
  MpegPicHdrSetForwardFCode(picHdr, FORWARD_F_CODE);

  /* Encode it */
  MpegPicHdrEncode (picHdr, obp);
  MpegPicPEncode (picHdr, scY, scU, scV, fmv, qScale, sliceInfo, sliceInfoLen, obp);

  bufsz = BitParserTell(obp);

  /* lots of things to free up */
  BitParserFree(obp);
  MpegPicHdrFree(picHdr);
  ByteFree(qScale);
  ScFree(scY);
  ScFree(scU);
  ScFree(scV);
  VectorFree(fmv);

  return bufsz;
}

static BitStream *BitStreamFromBuf(char *buf, int bufsz)
{
  BitStream *bs = BitStreamNew(0);
  bs->isVirtual = 1;
  bs->buffer = buf;
  bs->size = bufsz;
  bs->endDataPtr = bs->buffer;
  bs->endBufPtr = bs->buffer + bufsz;
  return bs;
}

/*----------------------------------------------------------------------*/

int makeNullPFrame(char *buf, int bufsz, short w, short h)
{
  BitStream *bs = BitStreamFromBuf(buf,bufsz);
  int framesz = MakeDeadFrame(h,w,bs);
  BitStreamFree(bs);
  return framesz;
}

int parseMpegSeqHdr(char *buf, int bufsz, MpegSeqHdr *sh)
{
  int ret;
  BitStream *bs = BitStreamFromBuf(buf,bufsz);
  BitParser *bp = BitParserNew();
  BitParserWrap(bp, bs);

//    if (MpegSeqHdrFind(bp) == -1)
//      return -1;
  ret = MpegSeqHdrParse(bp, sh);
  printf("seqhdrparse returned %d\n",ret);
  BitParserFree(bp);
  BitStreamFree(bs);
  return ret;
}
