/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#ifndef _ARRQUEUE_H_
#define _ARRQUEUE_H_

#include <core.h> // for bool

namespace ArrQueue {

  /*** \subsection{\texttt{<arrqueue.h>}} */
  /*** Defines namespace ArrQueue, which implements generic imperative
       queues of fixed-size, using an array as the underlying
       implementation.  The current functions have been targeted at
       elements that are possibly unique. */

extern struct ArrQueue<`a::TB,`r::TR>;
typedef struct ArrQueue<`a,`r> @`r aqueue_t<`a,`r>;
  /** A value of type [aqueue_t<`a,`r>] is a first-in, first-out queue
      of elements of type [`a]; the queue data structures are
      allocated in region [`r]. */

extern bool is_empty(aqueue_t<`a::TB>);
  /** [is_empty(q)] returns true if [q] contains no elements, and
      returns false otherwise. */

extern aqueue_t<`a::TA *`r::TR,`H> create(unsigned int sz);
  /** [create(sz)] heap-allocates a new, empty queue of size [sz]
      whose elements are possibly-null pointers. */

extern aqueue_t<`a::TA *`r2::TR,`r> rcreate(region_t<`r::TR> r,
					    unsigned int sz);
  /** [rcreate()] region-allocates into [r] a new, empty queue of size
      [sz] whose elements are possibly-null pointers. */

extern datatype exn { extern Full };
  /** [Full] is an exception raised by [add] and [push]. */
extern void add(aqueue_t<`a::TB,`r::TR>,`a x);
  /** [add(q,x)] adds [x] to the end of [q] (by side effect). */

extern void push(aqueue_t<`a::TB,`r::TR> q,`a x);
  /** [push(q,x)] adds [x] to the front of [q] (by side effect). */

extern datatype exn { extern Empty };
  /** [Empty] is an exception raised by [take]. */
extern `a *`r take(aqueue_t<`a::TA *`r::TR> q);
  /** [take(q)] removes the element from the front on [q] and returns
      it; if [q] is empty, exception [Empty] is thrown. */
extern `a *`r take_match(aqueue_t<`a::TA *`r::TR> q,
			 bool (@f)(`b,`a *`r) __attribute__((noconsume(2))),
			 `b env);
  /** [take_match(r,q,f,c)] looks through the queue (starting from the
      front) and returns the element [x] for which [f(x,c)] returns
      true. */

extern int length(aqueue_t<`a::TB>);
  /** [length(q)] returns the number of elements in [q]. */

extern int size(aqueue_t<`a::TB> q);
  /** [size(q)] returns the maximum number of elements [q] can hold. */

}

#endif
