/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* Defines the component structure, where one component implements a low-level
   operation, and can be connected to other components. */

/* A component can have N inputs and M outputs, which can be dynamically
   connected to other components.  Any number of outputs can be connected
   to a single input, but one output may not go to multiple inputs so
   that we avoid aliasing (this might change).

   As with Click, rather than assume queues between components, we make
   queues a special kind of component.  We similarly create "rate-control"
   components that alter the rates of operations.  We assume that for a
   given stream (that is, a path of input data through some components),
   that we schedule it only as often as the input stream arrives.

   n fps   +-------------+    +------------+
   ------->| rate 15 fps |--->|            |  15 fps
           +-------------+    | Picture in |--------->
   15 fps                     | Picture    |
   -------------------------->|            |
                              +------------+

   This rate reducer will be a separately scheduled item.  That is,
   it will have deadlines that determine when it runs, and it will
   have to queue its input.  Therefore, its input ports will
   essentially be queueing functions.
*/


#ifndef _COMPONENT_H_
#define _COMPONENT_H_

#include <core.h>
#include <fn.h>
#include <list.h>
#include "frame.h"

namespace Component {

using Fn;

/** The function used to implement component input ports
 **/
extern struct InportFn<`r::R>;
typedef struct InportFn<`r> @`r inportfn_t<`r>;

extern inportfn_t<`r>
make_inportfn(region_t<`r> r,
	      int (@`H f)(`env,frame_t),
	      `env x : regions(`env) > `r);
extern int apply_inportfn(inportfn_t f, frame_t x);

/** A component is an element that has a fixed number of inputs
    and a variable number of outputs, where each output receives
    a copy of the computation's data.  Each component is
    scheduled to run at [rate] Hz. 
**/
struct Component<`r::R> {
  string_t<`r> id;
  double rate;
  /** [rate] specifies how often a component should be scheduled **/
  inportfn_t<`r> ?`r inports;
  /** [inports] is an array of closures to be called
      by those components that connect to this component.  The
      closures take a frame (plus a hidden environment) as
      their argument.
  **/
  inportfn_t<`r> ?`r outports;
  /** [outports] is an array of closure pointers (each possibly null)
      pointing to inports of other components this component
      is connected to.  Each outport has the same output value,
      and each is "copied" to the corresponding inport.
  **/
  fn_t<inportfn_t<`r> ?`r @`r,int,`r> schedule_f;
  /** [schedule_f] is the function called at each schedule point,
      which takes the inports and outports as its arguments
  **/
};
typedef struct Component<`r> @`r component_t<`r>;

/** stuff for putting regions into dynamic regions */

struct DynComponent { <`r>
  Core::region_key_t<`r,`U> dyn;
  Component::component_t<`r> component;
};
typedef struct DynComponent dyncomponent_t;

struct DynComponentList { <`r>
  Core::region_key_t<`r,`U> dyn;
  List::list_t<Component::component_t<`r>,`r> components;
};
typedef struct DynComponentList *`r dyncomp_list_t<`r::TR>;

/** generic component usage functions **/

extern void connectComponents(region_t<`r> r,
			      component_t<`r> source,
			      component_t<`r> dest, 
			      int ? connections);
extern void invokeComponent(component_t<`r> component);
extern int forwardToComponent(inportfn_t ? @outports, 
			      frame_t stb);

/** useful generic components **/

extern int inNoop(int unusedEnv, frame_t unusedpkt);
extern int schedNoop(int unusedEnv, inportfn_t ? @outports);
extern component_t<`r> new_NullComponent(region_t<`r> r, string_t id);
/** [new_NullComponent creates a "null" component that discards any 
    input frames received 
**/

extern component_t<`r>
new_QueueComponent(region_t<`r> r, string_t id,
		   unsigned int maxqlen, double rate, bool doFlush);
/** [new_QueueComponent(id,len,r,flush) creates a queueing component.
    [r] times per second it is scheduled and its packets sent.  If [r]
    is 0, then it simply forwards its packets on through.  If the
    queue size ever exceeds [len], then packets are dropped.  If
    [flush] is true, then each time it is invoked, the component
    flushes its entire queue.  Otherwise it just sends one packet.
**/

extern component_t<`r> new_PrintComponent(region_t<`r> r, string_t id);
/** [new_PrintComponent(r,id)] creates a component that prints out
    all of the buffers it receives. **/
}
#endif
