/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* Defines functions and components for monitoring the resource usage
   of the network and local machine. */

#ifndef _MONITOR_H_
#define _MONITOR_H_

#include "component.h"
#include <sys/time.h>

using Component;

extern component_t<`r> 
new_PingComponent(region_t<`r> r, string_t id,
		  double rate, int maxOut, double maxOutTime);
/** [PingComponent(id,rate,maxOut,maxOutTime)] is scheduled [rate] times
    per second to send out a ping packet.  Packets that arrive on its
    inport are correlated with those sent, and the timing difference
    is logged. */

extern $(component_t<`r>, component_t<`r>) 
new_TimerComponents(region_t<`r> r, string_t idstart, string_t idstop,
		    string_t timerId);
/** [TimerComponent(startId,endId,timerId)] returns two components,
    the first is the starttimer and the second is the endtimer. */

extern component_t<`r>
new_SysinfoComponent(region_t<`r> r, string_t id, double rate);
/** [SysinfoComponent(id,rate) is scheduled [rate] times per second
    to query the current CPU activity and log the result. */

struct countState<`r::R> {
  struct timeval startTime;
  unsigned int totalBytesCounted;
  struct timeval lastTime;
  unsigned int bytesCounted;
  string_t<`r> id;
};

extern void countBytes(struct countState @countInfo,
		       streambuff_t stb,  int opt_count);
/** [countBytes(i,v,c)] counts the number of bytes in packet [v] and
    stores the information in [i], if [c] == 0; otherwise assumes
    it was a partial write/read, and logs [c] bytes. */

extern void printCounter(struct countState @countInfo);
/** [printCounter(c)] prints the accumulated state, relative to
    the current time */

extern component_t<`r> new_CountComponent(region_t<`r> r, 
					  string_t id, double rate);
/** [CountComponent(id) counts the number of bytes that pass through
    its inports, noting the total count and current bandwidth. */

extern void registerByteCounter(struct countState<`H> @`H sentCounter,
				struct countState<`H> @`H droppedCounter,
				double historyTime);
/** [registerByteCounter(s,d,h)] registers a byte counter for a send
    component, where [s] is the sent byte counter, [d] is the dropped
    byte counter, and [h] is the period over which to average, in
    seconds. */

extern void resetByteCounter(struct countState<`H> @`H sentCounter);
/** [resetByteCounter(s)] purges the counting information associated
    with counter [s]. **/

extern component_t<`r> new_MonitorComponent(region_t<`r> r,
					    string_t id, double rate,
					    string_t<`H> hostname,
					    unsigned short port,
					    string_t<`H> uri,
					    string_t<`H> version);
/** [MonitorComponent(id,r,h,p,u,v)] is the "syslog" equivalent for medianet.
    [r] times per second, it creates a monitoring report that it sends
    to the global scheduler at http://[h]:[p]/[u].  It includes the
    version [v] in the report. */

extern component_t<`r> new_PacketPairSendComponent(region_t<`r> r,
						   string_t id, double rate);
extern component_t<`r> new_PacketPairRecvComponent(region_t<`r> r, string_t id);
#endif
