/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse.

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#ifdef WIN32

#include <winsock2.h>

#else /* WIN32 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>

#define closesocket(s)	close((s))

#endif /* WIN32 */

#include <stdlib.h>
#include <stdio.h>
#include "oslib.h"

struct ev_channel {
	struct activity *activity;
	bool_t blocked;
	void (*handler)(void *env, ev_t ev);
	void *env;
};

struct activity {
	struct activity *next;
	int outstanding;			/* # outstanding events */
	int (*select_mask)(void *env, fd_set *read_set, fd_set *write_set);
	void (*select_check)(void *env, fd_set *read_set, fd_set *write_set);
	void *env;					/* global state */
};
static bool_t events_queued;

static ev_t run_queue, *run_last;
static struct activity *io_activities;

bool_t act_running = true;
evt_t evt_root, evt_close, evt_error, evt_data, evt_data_src, evt_client;

static void chunk_release(void *p){
	struct chunk *chunk = p;

	if (chunk->data != 0) {
		free(chunk->data);
	}
	free((char *) chunk);
}

/* Used to release ternary search tree maps.
 */
void map_release(void *map){
	tst_release((map_t *) &map, mem_release, 0);
}

void do_free(void *p){
	free(p);
}

ev_t evt_error_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_error));
}

ev_t evt_close_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_close));
}

ev_t evt_data_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_data));
}

void evt_data_release(evt_t evt, ev_t ev){
	struct evt_data *evd = ev_cast(ev, evt_data);

	if (evd->chunk != 0) {
		chunk_release(evd->chunk);
		evd->chunk = 0;
	}
}

ev_t evt_data_src_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_data_src));
}

void evt_data_src_release(evt_t evt, ev_t ev){
	struct evt_data_src *evds = ev_cast(ev, evt_data_src);

	evt_data_release(evt, ev);
	if (evds->addr != 0) {
		free(evds->addr);
		evds->addr = 0;
	}
}

ev_t evt_client_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_client));
}

void evt_client_release(evt_t evt, ev_t ev){
	/* TODO ??? */
}

/* Initialize this module.  Builds some basic event types.
 */
void act_init(void){
	run_last = &run_queue;

	evt_root = evt_create(0, "", 0, 0);

	evt_error = evt_create(evt_root, "error",
					evt_error_create, 0);

	evt_close = evt_create(evt_root, "close",
					evt_close_create, 0);

	evt_data = evt_create(evt_root, "data",
					evt_data_create, evt_data_release);

	evt_data_src = evt_create(evt_data, "src",
					evt_data_src_create, evt_data_src_release);

	evt_client = evt_create(evt_root, "client",
					evt_client_create, evt_client_release);
}

void act_done(void){
	evt_release(evt_client);
	evt_release(evt_close);
	evt_release(evt_data);
	evt_release(evt_error);
	evt_release(evt_root);
}

void ev_set_blocked(struct ev_channel *chan, bool_t blocked){
	chan->blocked = blocked;
}

bool_t ev_blocked(struct ev_channel *chan){
	return chan->blocked;
}

/* Enqueue an event, on behalf of the given activity, on the given channel.
 */
void ev_send(struct ev_channel *release, struct ev_channel *chan, ev_t ev){
	ev->release = release;
	ev->dst = chan;

	/* Enqueue event.
	 */
	ev->next = 0;
	*run_last = ev;
	run_last = &ev->next;

	/* Reduce scheduling priority.
	 */
	release->activity->outstanding++;

	events_queued = true;
}

/* Called from ev_release.
 */
void ev_released(ev_t ev){
	if (--ev->release->activity->outstanding < 0) {
		err_warning("ev_released: outstanding count underflow???");
		ev->release->activity->outstanding = 0;
	}
}

/* Should be called when event has been handled.  Much like ev_send back to the source.
 */
void ev_handled(ev_t ev){
	ev->dst = ev->release;
	ev->next = 0;
	*run_last = ev;
	run_last = &ev->next;
	events_queued = true;
}

/* Pick an event to handle.  Initially prefer activities with no outstanding events.
 */
ev_t pick_job(bool_t all){
	ev_t ev, *pev;

	for (pev = &run_queue; (ev = *pev) != 0; pev = &ev->next) {
		/* Skip events for blocked event channels.
		 */
		if (ev->dst->blocked) {
			continue;
		}

		/* If the activity has no outstanding events, this would be a great
		 * choice.  However, sometimes there aren't any, and we don't want
		 * to get into a deadlock situation, so we can't always be picky.
		 */
		if (ev->dst->activity->outstanding == 0 || all) {
			if ((*pev = ev->next) == 0) {
				run_last = pev;
			}
			return ev;
		}
	}

	return 0;
}

/* Dispatch the event.
 */
void dispatch(ev_t ev){
	struct ev_channel *chan = ev->dst;

	(*chan->handler)(chan->env, ev);
}

bool_t run_block(void){
	ev_t ev;
	fd_set read_set, write_set;
	struct activity *act;
	int n;
	bool_t block = false, blocked = false;
	struct timeval tv;

	memset(&tv, 0, sizeof(tv));
	do {
		/* First deal with as many events as possible, but for flow control purposes, don't
		 * give events to activities that have outstanding events.
		 */
		while ((ev = pick_job(false)) != 0) {
			dispatch(ev);
		}

		/* If there are no events left, it's ok to block.
		 */
		if (run_queue == 0) {
			block = true;
		}

		/* Run through all the I/O activities to see if they want input/output.
		 */
		FD_ZERO(&read_set); FD_ZERO(&write_set);
		if (act_running) {
			n = 0;
			for (act = io_activities; act != 0; act = act->next) {
				n += (*act->select_mask)(act->env, &read_set, &write_set);
			}

			/* Wait for input/output.
			 */
			if (n > 0) {
				n = select(64, &read_set, &write_set, 0, block ? 0 : &tv);
				if (n < 0) {
					err_fatal("run_block: select");
				}
			}
		}
		else {
			n = 0;
		}

		/* See if we blocked.
		 */
		if (block) {
			blocked = true;
		}

		/* Run through all the I/O activities again to pick up their input/generate their output.
		 * This can create new events, and we're interested in this, so we keep track of that
		 * in the global variable events_queued.
		 */
		events_queued = false;
		for (act = io_activities; act != 0; act = act->next) {
			(*act->select_check)(act->env, &read_set, &write_set);
		}

		/* If the I/O activities created new events, deal with those first as if we're
		 * entering the loop for the first time.
		 */
		if (events_queued) {
			block = blocked = false;
			continue;
		}

		/* Try to pick a job from the remaining list in order to break cycles.
		 */
		if ((ev = pick_job(true)) != 0) {
			dispatch(ev);
			block = blocked = false;
			continue;
		}

		if (!act_running) {
			return false;
		}

		/* At this point, we should block.
		 */
		block = true;
		
		/* Run loop until we've blocked.
		 */
	} while (!blocked);

	return true;
}

/* Allocate a new activity.
 */
struct activity *act_alloc(void *env){
	struct activity *act = (struct activity *) calloc(1, sizeof(*act));

	act->env = env;
	return act;
}

void *act_env(struct activity *act){
	return act->env;
}

void act_release(struct activity *act){
	struct activity **pa, *a;

	/* Unlink out of io_activities.
	 */
	for (pa = &io_activities; (a = *pa) != 0;) {
		if (a == act) {
			*pa = a->next;
			break;
		}
		else {
			pa = &a->next;
		}
	}

	free((char *) act);
}

/* See if the given activity has any outstanding events.
 */
bool_t act_outstanding(struct activity *act){
	return act->outstanding > 0;
}

/* Allocate a new event channel.
 */
struct ev_channel *ev_channel_alloc(struct activity *act,
						void (*handler)(void *env, ev_t ev), void *env){
	struct ev_channel *chan = (struct ev_channel *) calloc(1, sizeof(*chan));

	chan->activity = act;
	chan->handler = handler;
	chan->env = env;
	return chan;
}

void ev_channel_release(struct ev_channel *chan){
	free((char *) chan);
}

/* Convert an string IP address of the form "a.b.c.d:port" to a sockaddr_in address.
 */
static bool_t net_get_addr(struct sockaddr_in *sin, const char *addr){
	char *ip, *delim;
	unsigned short p;
	struct hostent *h;
	
	/* Disect the address.
	 */
	if ((delim = strchr(addr, ':')) == 0) {
		err_warning("net_get_addr: bad address format");
		return false;
	}
	if (sscanf(delim + 1, "%hu", &p) != 1 || p == 0) {
		err_warning("net_get_addr: bad port");
		return false;
	}
	ip = mem_string_copy(addr, delim - addr);

	memset(sin, 0, sizeof(*sin));
	sin->sin_family = AF_INET;
	sin->sin_port = htons(p);

	/* See if it's a DNS name.
	 */
	if (*ip < '0' || *ip > '9') {
		if ((h = gethostbyname(ip)) == 0) {
			err_warning("net_get_addr: gethostbyname '%s' failed", ip);
			free(ip);
			return false;
		}
		sin->sin_addr = * (struct in_addr *) h->h_addr_list[0];
	}
	else {
		sin->sin_addr.s_addr = inet_addr(ip);
	}

	free(ip);
	return true;
}

/***************************************************************************/
// TCP output activity.

struct socket_info {
	int refcnt;				/* one for sender, one for receiver */
	int socket;
};

struct tcp_output {
	struct activity *activity;
	struct ev_channel *in;
	struct socket_info *si;
	char *buffer;
	int size;
	int offset;
};

void socket_release(struct socket_info *si){
	if (--si->refcnt == 0) {
		free((char *) si);
	}
}

/* Set the output bit if we're interested in whether output is possible.
 */
int tcp_output_mask(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_output *to = env;

	/* Ignore if I'm not blocked.  There no need to check to see if output is
	 * possible then, because we know it is.
	 */
	if (!to->in->blocked) {
		return 0;
	}

	/* Also ignore if socket has been closed.
	 */
	if (to->si->socket < 0) {
		return 0;
	}

	FD_SET(to->si->socket, write_set);
	return 1;
}

/* See if output is possible.
 */
void tcp_output_check(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_output *to = env;
	int n;

	/* Ignore if I'm not blocked.  There no need to check to see if output is
	 * possible then, because we know it is.
	 */
	if (!to->in->blocked) {
		return;
	}

	/* Ignore if socket has been closed.
	 */
	if (to->si->socket < 0) {
		return;
	}

	if (FD_ISSET(to->si->socket, write_set)) {
		to->in->blocked = false;
		if (to->size != 0) {
			n = send(to->si->socket, to->buffer + to->offset, to->size, 0);
			if (n < 0) {
				err_warning("tcp_output_check: send %d failed", to->si->socket);
				closesocket(to->si->socket);
				to->si->socket = -1;
				free(to->buffer);
				to->buffer = 0;
				to->size = 0;
				to->offset = 0;
				return;
			}
			to->offset += n;
			to->size -= n;
			if (to->size > 0) {
				to->in->blocked = true;
			}
		}
	}
}

/* Events for the TCP output activity come in here.
 */
void tcp_output_handler(void *env, ev_t ev){
	struct tcp_output *to = env;
	struct chunk *chunk;
	int n;

	if (ev->type == evt_close) {
		if (to->si->socket >= 0) {
			closesocket(to->si->socket);
			to->si->socket = -1;
		}
		if (to->buffer != 0) {
			free(to->buffer);
		}
		socket_release(to->si);
		ev_channel_release(to->in);
		act_release(to->activity);
		free((char *) to);
	}
	else if (ev->type != evt_data) {
		err_warning("tcp_output_handler: not a data event");
	}
	else {
		chunk = ev_cast(ev, evt_data)->chunk;
		n = send(to->si->socket, chunk->data, chunk->size, 0);

		/* TODO.  Buffer the rest and set blocked flag if so.
		 */
		// to->in->blocked = true;
	}

	ev_handled(ev);
}

/* Create a TCP output activity on the given socket.
 */
struct ev_channel *tcp_output_create(struct socket_info *si){
	struct tcp_output *to;
	struct activity *act;

	/* Create the activity.
	 */
	to = (struct tcp_output *) calloc(1, sizeof(*to));
	to->activity = act = act_alloc(to);
	to->in = ev_channel_alloc(act, tcp_output_handler, to);
	to->si = si;
	
	/* Add activity to I/O activities.
	 */
	act->select_mask = tcp_output_mask;
	act->select_check = tcp_output_check;
	act->next = io_activities;
	io_activities = act;

	return to->in;
}

/***************************************************************************/
// TCP input.

#define CHUNK_SIZE		1024

struct tcp_input {
	struct activity *activity;
	struct ev_channel *release;
	struct ev_channel *dst;
	struct socket_info *si;
};

/* Set the read bit if we're interested in input on our socket.
 */
int tcp_input_mask(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_input *ti = env;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ti->activity) || ev_blocked(ti->dst)) {
		return 0;
	}

	/* Also ignore if socket has been closed.
	 */
	if (ti->si->socket < 0) {
		return 0;
	}

	FD_SET(ti->si->socket, read_set);
	return 1;
}

/* Check if there's input on our socket.
 */
void tcp_input_check(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_input *ti = env;
	char *data;
	int n;
	struct chunk *chunk;
	ev_t ev;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ti->activity) || ev_blocked(ti->dst)) {
		return;
	}

	/* Ignore if socket has been closed.
	 */
	if (ti->si->socket < 0) {
		return;
	}

	if (act_running) {
		/* If there's no input, we're done.
		 */
		if (!FD_ISSET(ti->si->socket, read_set)) {
			return;
		}

		/* Read a chunk.
		 */
		data = malloc(CHUNK_SIZE);
		n = recv(ti->si->socket, data, CHUNK_SIZE, 0);
		if (n < 0) {
			err_warning_socket("tcp_input_check: error");
			free(data);
			ev = ev_create(evt_error);
			ev_cast(ev, evt_error)->descr = "tcp_input_check error";
			ev_send(ti->release, ti->dst, ev);
		}
		else if (n == 0) {
#ifdef debug
			log_printf("tcp_input_check: eof");
#endif
			free(data);
		}
		else {
			chunk = (struct chunk *) calloc(1, sizeof(*chunk));
			chunk->data = data;
			chunk->size = n;
			ev = ev_create(evt_data);
			ev_cast(ev, evt_data)->chunk = chunk;
			ev_send(ti->release, ti->dst, ev);
		}
	}
	else {
		n = 0;
	}

	/* Clean up on EOF or ERROR.
	 */
	if (n <= 0) {
		closesocket(ti->si->socket);
		ti->si->socket = -1;
		ev = ev_create(evt_close);
		ev_send(ti->release, ti->dst, ev);
	}
}

void tcp_input_event_release(void *env, ev_t ev){
	struct tcp_input *ti = env;

	ev_release(ev);

	/* See if we can disappear.
	 */
	if (ti->si->socket < 0 && act_outstanding(ti->activity) == 0) {
		socket_release(ti->si);
		ev_channel_release(ti->release);
		act_release(ti->activity);
		free((char *) ti);
	}
}

/* Create a TCP input activity on the given socket.  Output
 * events should go to the given event handler.
 */
bool_t tcp_input_create(struct socket_info *si, struct ev_channel *dst){
	struct tcp_input *ti;
	struct activity *act;

	/* Create the activity.
	 */
	ti = (struct tcp_input *) calloc(1, sizeof(*ti));
	ti->activity = act = act_alloc(ti);
	ti->release = ev_channel_alloc(act, tcp_input_event_release, ti);
	ti->dst = dst;
	ti->si = si;
	
	/* Add activity to I/O activities.
	 */
	act->select_mask = tcp_input_mask;
	act->select_check = tcp_input_check;
	act->next = io_activities;
	io_activities = act;

	return true;
}

/***************************************************************************/
// TCP server.

struct tcp_server {
	struct activity *activity;
	struct ev_channel *release;
	struct ev_channel *dst;
	int socket;
};

/* See if we're interested in new clients.
 */
int tcp_server_mask(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_server *ts = env;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ts->activity) || ev_blocked(ts->dst)) {
		return 0;
	}

	/* Also ignore if socket has been closed.
	 */
	if (ts->socket < 0) {
		return 0;
	}

	FD_SET(ts->socket, read_set);
	return 1;
}

/* If there is a new client, send an event to the output event handler.
 */
void tcp_server_check(void *env, fd_set *read_set, fd_set *write_set){
	struct tcp_server *ts = env;
	struct sockaddr_in sin;
	int len, skt;
	ev_t ev;
	struct socket_info *si;
	bool_t do_close = false;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ts->activity) || ev_blocked(ts->dst)) {
		return;
	}

	/* If socket has been closed, ignore.
	 */
	if (ts->socket < 0) {
		return;
	}

	if (act_running) {
		/* If there's no new client, we're done.
		 */
		if (!FD_ISSET(ts->socket, read_set)) {
			return;
		}

		/* Accept connection.
		 */
		len = sizeof(sin);
		skt = accept(ts->socket, (struct sockaddr *) &sin, &len);
		if (skt < 0) {
			ev = ev_create(evt_error);
			ev_cast(ev, evt_error)->descr = "tcp_server_check error";

			/* Clean up.
			 */
			do_close = true;
		}
		else {
			si = (struct socket_info *) calloc(1, sizeof(*si));
			si->refcnt = 2;
			si->socket = skt;
			ev = ev_create(evt_client);
			ev_cast(ev, evt_client)->si = si;
		}
		ev_send(ts->release, ts->dst, ev);
	}
	else {
		do_close = true;
	}

	if (do_close) {
		closesocket(ts->socket);
		ts->socket = -1;
		ev = ev_create(evt_close);
		ev_send(ts->release, ts->dst, ev);
	}
}

void tcp_server_event_release(void *env, ev_t ev){
	struct tcp_server *ts = env;

	ev_release(ev);

	/* See if we can disappear.
	 */
	if (ts->socket < 0 && act_outstanding(ts->activity) == 0) {
		ev_channel_release(ts->release);
		act_release(ts->activity);
		free((char *) ts);
	}
}

/* Create a new TCP server activity on the given port.  When clients
 * announce themselves, an event is sent to the given event handler.
 */
bool_t tcp_server_create(unsigned short port, struct ev_channel *dst){
	int s, on = 1;
	struct sockaddr_in sin;
	struct tcp_server *ts;
	struct activity *act;
	
	/* Create and bind the socket.
	 */
	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("tcp_server_create: socket");
		return false;
	}
	setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *) &on, sizeof(on));
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(port);
	if (bind(s, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		perror("tcp_server_create: bind");
		closesocket(s);
		return false;
	}
	(void) listen(s, 16);

	/* Create the activity.
	 */
	ts = (struct tcp_server *) calloc(1, sizeof(*ts));
	ts->activity = act = act_alloc(ts);
	ts->release = ev_channel_alloc(act, tcp_server_event_release, ts);
	ts->dst = dst;
	ts->socket = s;
	
	/* Add activity to I/O activities.
	 */
	act->select_mask = tcp_server_mask;
	act->select_check = tcp_server_check;
	act->next = io_activities;
	io_activities = act;
	return true;	
}

/***************************************************************************/
// TCP client.

/* Connect to a TCP server.
 *
 * TODO.  Make non-blocking.
 */
struct socket_info *tcp_client_create(char *addr){
	int s;
	struct sockaddr_in sin;
	struct socket_info *si;

	/* See if this is a reasonable address.
	 */
	if (!net_get_addr(&sin, addr)) {
		return 0;
	}
	
	/* Create and connect to the socket.
	 */
	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("tcp_client_create: socket");
		return 0;
	}
	if (connect(s, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		perror("tcp_client_create: bind");
		closesocket(s);
		return 0;
	}

	si = (struct socket_info *) calloc(1, sizeof(*si));
	si->refcnt = 2;
	si->socket = s;
	return si;
}

/***************************************************************************/
// UDP output activity.

struct udp_output {
	struct activity *activity;
	struct ev_channel *in;
	struct socket_info *si;
};

/* Set the output bit if we're interested in whether output is possible.
 */
int udp_output_mask(void *env, fd_set *read_set, fd_set *write_set){
	struct udp_output *uo = env;

	/* Ignore if I'm not blocked.  There no need uo check uo see if output is
	 * possible then, because we know it is.
	 */
	if (!uo->in->blocked) {
		return 0;
	}

	/* Also ignore if socket has been closed.
	 */
	if (uo->si->socket < 0) {
		return 0;
	}

	FD_SET(uo->si->socket, write_set);
	return 1;
}

/* See if output is possible.
 */
void udp_output_check(void *env, fd_set *read_set, fd_set *write_set){
	struct udp_output *uo = env;

	/* Ignore if I'm not blocked.  There no need uo check uo see if output is
	 * possible then, because we know it is.
	 */
	if (!uo->in->blocked) {
		return;
	}

	/* Ignore if socket has been closed.
	 */
	if (uo->si->socket < 0) {
		return;
	}

	if (FD_ISSET(uo->si->socket, write_set)) {
		uo->in->blocked = false;
	}
}

/* Events for the TCP output activity come in here.
 */
void udp_output_handler(void *env, ev_t ev){
	struct udp_output *uo = env;
	struct chunk *chunk;
	int n;
	char *addr;
	struct sockaddr_in sin;

	if (ev->type == evt_close) {
		if (uo->si->socket >= 0) {
			closesocket(uo->si->socket);
			uo->si->socket = -1;
		}
		ev_channel_release(uo->in);
		act_release(uo->activity);
		free((char *) uo);
	}
	else if (ev->type != evt_data_src) {
		err_warning("udp_output_handler: not a data/src event");
	}
	else {
		chunk = ev_cast(ev, evt_data)->chunk;
		addr = ev_cast(ev, evt_data_src)->addr;
		if (!net_get_addr(&sin, addr)) {
			err_fatal("udp_output_handler: bad address");
		}
		n = sendto(uo->si->socket, chunk->data, chunk->size, 0,
							(struct sockaddr *) &sin, sizeof(sin));
printf("SENT %d bytes to %d\n", n, htons(sin.sin_port));
	}

	ev_handled(ev);
}

/* Create a UDP output activity on the given socket.
 */
struct ev_channel *udp_output_create(struct socket_info *si){
	struct udp_output *uo;
	struct activity *act;

	/* Create the activity.
	 */
	uo = (struct udp_output *) calloc(1, sizeof(*uo));
	uo->activity = act = act_alloc(uo);
	uo->in = ev_channel_alloc(act, udp_output_handler, uo);
	uo->si = si;
	
	/* Add activity uo I/O activities.
	 */
	act->select_mask = udp_output_mask;
	act->select_check = udp_output_check;
	act->next = io_activities;
	io_activities = act;

	return uo->in;
}

/***************************************************************************/
// UDP input activity.

struct udp_input {
	struct activity *activity;
	struct ev_channel *release;
	struct ev_channel *dst;
	struct socket_info *si;
};

/* See if we're interested in input.
 */
int udp_input_mask(void *env, fd_set *read_set, fd_set *write_set){
	struct udp_input *ui = env;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ui->activity) || ev_blocked(ui->dst)) {
		return 0;
	}

	/* Also ignore if socket has been closed.
	 */
	if (ui->si->socket < 0) {
		return 0;
	}

	FD_SET(ui->si->socket, read_set);
	return 1;
}

/* Check if there's input on our socket.
 */
void udp_input_check(void *env, fd_set *read_set, fd_set *write_set){
	struct udp_input *ui = env;
	char *data;
	int n, len;
	struct sockaddr_in sin;
	struct chunk *chunk;
	ev_t ev;

	/* Ignore if my output has not been consumed yet, or the output channel is blocked.
	 */
	if (act_outstanding(ui->activity) || ev_blocked(ui->dst)) {
		return;
	}

	/* Ignore if socket has been closed.
	 */
	if (ui->si->socket < 0) {
		return;
	}

	if (act_running) {
		/* If there's no input, we're done.
		 */
		if (!FD_ISSET(ui->si->socket, read_set)) {
			return;
		}

		/* Read a chunk.
		 */
		data = malloc(CHUNK_SIZE);
		len = sizeof(sin);
		n = recvfrom(ui->si->socket, data, CHUNK_SIZE, 0, (struct sockaddr *) &sin, &len);
		if (n < 0) {
			err_warning_socket("udp_input_check: error");
			free(data);
			ev = ev_create(evt_error);
			ev_cast(ev, evt_error)->descr = "udp_input_check error";
			ev_send(ui->release, ui->dst, ev);
		}
		else if (n == 0) {
			log_printf("udp_input_check: eof");
			free(data);
		}
		else {
			chunk = (struct chunk *) calloc(1, sizeof(*chunk));
			chunk->data = data;
			chunk->size = n;
			ev = ev_create(evt_data_src);
			ev_cast(ev, evt_data)->chunk = chunk;
			ev_cast(ev, evt_data_src)->addr = mem_print("%s:%d",
						inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
			ev_send(ui->release, ui->dst, ev);
		}
	}
	else {
		n = 0;
	}

	/* Clean up on EOF or ERROR.
	 */
	if (n <= 0) {
		closesocket(ui->si->socket);
		ui->si->socket = -1;
		ev = ev_create(evt_close);
		ev_send(ui->release, ui->dst, ev);
	}
}

void udp_input_event_release(void *env, ev_t ev){
	struct udp_input *ui = env;

	ev_release(ev);

	/* See if we can disappear.
	 */
	if (ui->si->socket < 0 && act_outstanding(ui->activity) == 0) {
		ev_channel_release(ui->release);
		act_release(ui->activity);
		free((char *) ui);
	}
}

/* Create a new UDP input activity on the given socket.  Packets should be sent to the
 * given event channel.
 */
bool_t udp_input_create(struct socket_info *si, struct ev_channel *dst){
	struct udp_input *ui;
	struct activity *act;

	/* Create the activity.
	 */
	ui = (struct udp_input *) calloc(1, sizeof(*ui));
	ui->activity = act = act_alloc(ui);
	ui->release = ev_channel_alloc(act, udp_input_event_release, ui);
	ui->dst = dst;
	ui->si = si;
	
	/* Add activity to I/O activities.
	 */
	act->select_mask = udp_input_mask;
	act->select_check = udp_input_check;
	act->next = io_activities;
	io_activities = act;

	return true;
}

struct socket_info *udp_create(unsigned short port){
	int s, on = 1;
	struct sockaddr_in sin;
	struct socket_info *si;
	
	/* Create and bind the socket.
	 */
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("udp_input_create: socket");
		return 0;
	}
	setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *) &on, sizeof(on));
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(port);
	if (bind(s, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		perror("udp_input_create: bind");
		closesocket(s);
		return 0;
	}

	si = (struct socket_info *) calloc(1, sizeof(*si));
	si->refcnt = 2;
	si->socket = s;
	return si;
}

/***************************************************************************/
// Sink activity.

struct sink_act {
	struct activity *activity;
	struct ev_channel *in;
};

void sink_handler(void *env, ev_t ev){
	struct sink_act *sa = env;

	if (ev->type == evt_close) {
		ev_channel_release(sa->in);
		act_release(sa->activity);
		free((char *) sa);
	}

	ev_handled(ev);
}

struct ev_channel *sink_create(void){
	struct sink_act *sa = (struct sink_act *) calloc(1, sizeof(*sa));

	sa->activity = act_alloc(sa);
	sa->in = ev_channel_alloc(sa->activity, sink_handler, sa);
	return sa->in;
}

/***************************************************************************/
// Print activity.

struct print_act {
	struct activity *activity;
	struct ev_channel *in;
	char *descr;
	bool_t binary;
};

void print_handler(void *env, ev_t ev){
	struct print_act *pa = env;
	struct chunk *chunk;
	int i, c;

	printf("%s: %s\n", pa->descr, ev->type->name);
	if (ev->type == evt_data) {
		chunk = ev_cast(ev, evt_data)->chunk;
		printf("================== %d bytes\n", chunk->size);
		if (pa->binary) {
			for (i = 0; i < chunk->size; i++) {
				putchar(' ');
				c = chunk->data[i] & 0xFF;
				switch (c) {
				case 0:		printf("\\0"); break;
				case '\n':	printf("\\n"); break;
				case '\t':	printf("\\t"); break;
				case '\r':	printf("\\r"); break;
				case '\b':	printf("\\b"); break;
				default:
					if (' ' <= c && c <= '~') {
						putchar(c);
						putchar(' ');
					}
					else {
						printf("%2x", c);
					}
				}
				if (i % 16 == 15) {
					putchar('\n');
				}
			}
			putchar('\n');
		}
		else {
			printf("%.*s", chunk->size, chunk->data);
		}
		printf("==================\n");
	}
	else if (ev->type == evt_close) {
		ev_channel_release(pa->in);
		act_release(pa->activity);
		free(pa->descr);
		free((char *) pa);
	}

	ev_handled(ev);
}

struct ev_channel *print_create(char *descr, bool_t binary){
	struct print_act *pa = (struct print_act *) calloc(1, sizeof(*pa));

	pa->activity = act_alloc(pa);
	pa->in = ev_channel_alloc(pa->activity, print_handler, pa);
	pa->descr = mem_string_copy(descr, strlen(descr));
	pa->binary = binary;
	return pa->in;
}

/***************************************************************************/
// File input.

struct file_input {
	struct activity *activity;
	struct ev_channel *release;
	struct ev_channel *dst;
	int fd;
};

void file_input_read(struct file_input *fi){
	char *data;
	int n;
	ev_t ev;
	struct chunk *chunk;

	/* Read a chunk.
	 */
	data = malloc(CHUNK_SIZE);
	n = read(fi->fd, data, CHUNK_SIZE);
	if (n < 0) {
		err_warning_socket("file_input_check: error");
		free(data);
		ev = ev_create(evt_error);
		ev_cast(ev, evt_error)->descr = "file_input_check error";
		ev_send(fi->release, fi->dst, ev);
	}
	else if (n == 0) {
		free(data);
	}
	else {
		chunk = (struct chunk *) calloc(1, sizeof(*chunk));
		chunk->data = data;
		chunk->size = n;
		ev = ev_create(evt_data);
		ev_cast(ev, evt_data)->chunk = chunk;
		ev_send(fi->release, fi->dst, ev);
	}

	/* Clean up on EOF or ERROR.
	 */
	if (n <= 0) {
		close(fi->fd);
		fi->fd = -1;
		ev = ev_create(evt_close);
		ev_send(fi->release, fi->dst, ev);
	}
}

void file_input_event_release(void *env, ev_t ev){
	struct file_input *fi = env;

	ev_release(ev);

	/* If there's more, read the next chunk.
	 */
	if (fi->fd >= 0) {
		file_input_read(fi);
		return;
	}

	/* Otherwise see if we can disappear.
	 */
	if (act_outstanding(fi->activity) == 0) {
		ev_channel_release(fi->release);
		act_release(fi->activity);
		free((char *) fi);
		return;
	}
}

/* Create a file input activity on the given file descriptor.  Output
 * events should go to the given event handler.
 */
bool_t file_input_create(int fd, struct ev_channel *dst){
	struct file_input *fi;
	struct activity *act;

	/* Create the activity.
	 */
	fi = (struct file_input *) calloc(1, sizeof(*fi));
	fi->activity = act = act_alloc(fi);
	fi->release = ev_channel_alloc(act, file_input_event_release, fi);
	fi->dst = dst;
	fi->fd = fd;

	file_input_read(fi);

	return true;
}
