%{

  /* Parses files output by the BRITE network topology generator.
     See http://www.cs.bu.edu/brite/ for general information,
     and http://www.cs.bu.edu/brite/user_manual/node29.html for
     an example file. */

#include <string.h>
#include <stdlib.h>
#include "mmsched.h"

extern void yyerror(char *s);
extern int yylex();
extern struct topology *topology;

struct list {
  struct list *next;
  void *x;
};

struct nodelist {
  struct nodelist *next;
  struct node *node;
};

struct netlist {
  struct netlist *next;
  struct network *network;
};

static int length(struct list *l) {
  int len = 0;
  while (l != NULL) {
    len++;
    l = l->next;
  }
  return len;
} 

static int compress_netlist(struct netlist *l, struct network **result) {
  int len = length((struct list *)l);
  struct network *nets;
  int i = 0;

  nets = malloc(sizeof(struct network) * (len+1));

  while (l != NULL) {
    nets[i++] = *(l->network);
    l = l->next;
  }
  nets[i].name = 0;
  nets[i].capacity = 0;

  *result = nets;
  return len;
}

static int compress_nodelist(struct nodelist *l, struct node **result) {
  int len = length((struct list *)l);
  struct node *nodes;
  int i = 0;

  nodes = malloc(sizeof(struct node) * (len+1));
  
  while (l != NULL) {
    nodes[i++] = *(l->node);
    l = l->next;
  }
  nodes[i].name = 0;
  nodes[i].capacity = 0;

  *result = nodes;
  return len;
}

%}

%union {
  double floatval;
  int intval;
  struct topology *topology;
  struct nodelist *nodevals;
  struct netlist *netvals;
  struct node *nodeval;
  struct network *netval;
}

%token TOPOLOGY, NODES, EDGES, MODEL, COLON, LPAREN, RPAREN, COMMA, ID
%token <floatval> FLOAT
%token <intval> INT

%type <intval> anything_list anything
%type <nodevals> node_list nodes
%type <netvals>  edge_list edges
%type <nodeval> node
%type <netval>  edge
%type <topology> file

%%

file : topology model nodes edges
	{ struct topology *t = malloc(sizeof(struct topology));
	  int num_nodes;
	  int num_nets;
	  struct node *nodes;
	  struct network *networks;
	  num_nodes = compress_nodelist($3,&nodes);
	  num_nets = compress_netlist($4,&networks);
	  t->nodes = nodes;
	  t->num_nodes = num_nodes;
	  t->nets = networks;
	  t->num_nets = num_nets;
	  $$ = t; 
	  topology = t;
	};

parenstring : LPAREN anything_list RPAREN;

anything : INT { $$ = 0; } | FLOAT { $$ = 0; } | ID { $$ = 0; };
anything_list : { $$ = 0; } | anything anything_list { $$ = 0; };

topology : TOPOLOGY COLON LPAREN INT NODES COMMA INT EDGES RPAREN;

model : MODEL parenstring COLON anything_list;

nodes : NODES COLON LPAREN INT RPAREN node_list	{ $$ = $6; };

node_list : /* epsilon */
	{ $$ = NULL; }
	  | node node_list 
	{ struct nodelist *l;
	  l = malloc(sizeof(struct nodelist *));
	  l->next = $2;
	  l->node = $1;
	  $$ = l;
	};

node : INT INT INT INT INT INT ID
	{ struct node *n = calloc(sizeof(struct node),1);
	  char *name = malloc(20);
	  sprintf(name,"pc%d",$1 + 1);
	  n->name = name;
	  n->capacity = 1e8;
	  n->host = "";
	  $$ = n;
	};

edges : EDGES COLON LPAREN INT RPAREN edge_list { $$ = $6; };

edge_list : /* epsilon */
	{ $$ = NULL; }
	  | edge edge_list 
	{ struct netlist *l;
	  l = malloc(sizeof(struct netlist *));
	  l->next = $2;
	  l->network = $1;
	  $$ = l;
	};

edge : INT INT INT FLOAT FLOAT FLOAT INT INT ID ID
	{ struct network *n = calloc(sizeof(struct network),1);
	  char *name = malloc(20);
	  char *nodes = malloc(40);
	  sprintf(name,"net%d",$1);
	  sprintf(nodes,"pc%d pc%d",$2+1,$3+1);
	  n->name = name;
	  n->capacity = n->link_capacity = $6 * 1000.0;
	  n->latency = $5;
	  n->nodes = nodes;
	  $$ = n;
	};

%%

void yyerror(char *s)          {printf("%s\n",s);}

