/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include "oslib.h"

#define get_short(p)	((((p)[0] << 8) & 0xFF00) | ((p)[1] & 0xFF))
#define get_long(p)		((((p)[0] << 24) & 0xFF000000) | (((p)[1] << 16) & 0xFF0000) | \
						 (((p)[2] << 8) & 0xFF00) | ((p)[3] & 0xFF))

struct dns_act {
	struct activity *activity;
	struct ev_channel *in;
	struct ev_channel *release;
	struct ev_channel *output;
};

static int evt_data_data, evt_data_src_addr;
static int evt_error_descr;

static char *read_labels(char *p, struct print_channel *pc, char *data, int size){
	int n;

	for (;;) {
		n = *p++ & 0xFF;
		if (n == 0) {
			return p;
		}
		switch (n & 0xC0) {
		case 0x00:
			while (n--) {
				pc_putchar(pc, *p++);
			}
			pc_putchar(pc, '.');
			break;
		case 0xC0:
			(void) read_labels(&data[n & 0x3F], pc, data, size);
			return p;
		default:
			err_warning("dns_got_message: bad label");
			return 0;
		}
	}
}

static void dns_make_message(struct dns_act *da, char *id, char **msg, int *size){
	struct print_channel *pc = mc_open();
	unsigned char flags;

	/* First build the header.
	 */
	pc_puts(pc, id, 2);

	flags = 0;
	flags |= (1 << 7);		/* QR = 1 */
	flags |= (1 << 2);		/* AA = 1 */
	flags |= (1 << 0);		/* RD = 1 */
	pc_putchar(pc, flags);

	flags = 0;
	flags |= (1 << 7);		/* RA = 1 */
	pc_putchar(pc, flags);

	pc_putchar(pc, 0); pc_putchar(pc, 0);	/* QDCOUNT = 0 */
	pc_putchar(pc, 0); pc_putchar(pc, 1);	/* ANCOUNT = 1 */
	pc_putchar(pc, 0); pc_putchar(pc, 0);	/* NSCOUNT = 0 */
	pc_putchar(pc, 0); pc_putchar(pc, 0);	/* ARCOUNT = 0 */

	/* Now add the Answer Section.
	 */
	pc_putchar(pc, 2); pc_puts(pc, "xx", 2);
	pc_putchar(pc, 5); pc_puts(pc, "rnets", 5);
	pc_putchar(pc, 3); pc_puts(pc, "com", 3);
	pc_putchar(pc, 0);

	pc_putchar(pc, 0); pc_putchar(pc, 1);	/* TYPE = 1 */
	pc_putchar(pc, 0); pc_putchar(pc, 1);	/* CLASS = 1 */

	pc_putchar(pc, 0); pc_putchar(pc, 0);	/* TTL = 0 */
	pc_putchar(pc, 0); pc_putchar(pc, 0);	/* ditto */

	pc_putchar(pc, 0); pc_putchar(pc, 4);	/* RDLENGTH */
	pc_putchar(pc, 2); pc_putchar(pc, 'a');
	pc_putchar(pc, 'b'); pc_putchar(pc, 0);

	/* Return the result.
	 */
	*size = mc_size(pc);
	*msg = mc_close(pc);
}

static void dns_udp_send(struct dns_act *da, struct ev_channel *chan, char *addr, char *msg, int size){
	struct chunk *chunk;
	ev_t ev;

	chunk = (struct chunk *) calloc(1, sizeof(*chunk));
	chunk->data = msg;
	chunk->size = size;

	ev = ev_create(evt_data_src);
	ev_set_ptr(ev, evt_data_data, chunk);
	ev_set_ptr(ev, evt_data_src_addr, mem_print("%s", addr));

	ev_send(da->release, da->output, ev);
}

static void dns_print_message(struct dns_act *da, char *data, int size){
	struct print_channel *pc;
	int i, qdcount, ancount, rdlength;
	char *p;

	printf("==============================\n");
	printf("ID: %x\n", * (short *) data & 0xFFFF);
	printf("QR: %d\n", (data[2] >> 7) & 1);
	printf("OP: %d\n", (data[2] >> 3) & 0xF);
	printf("AA: %d\n", (data[2] >> 2) & 1);
	printf("TC: %d\n", (data[2] >> 1) & 1);
	printf("RD: %d\n", (data[2] >> 0) & 1);
	printf("RA: %d\n", (data[3] >> 7) & 1);
	printf("Z:  %d\n", (data[3] >> 4) & 0x7);
	printf("RC: %d\n", (data[3] >> 0) & 0xF);
	printf("QD: %d\n", get_short(&data[4]));
	printf("AN: %d\n", get_short(&data[6]));
	printf("NS: %d\n", get_short(&data[8]));
	printf("AR: %d\n", get_short(&data[10]));

	qdcount = get_short(&data[4]);
	p = &data[12];
	for (i = 0; i < qdcount; i++) {
		pc = mc_open();
		p = read_labels(p, pc, data, size);
		printf("QNAME: '%s'\n", mc_close(pc));
		printf("QTYPE: %d\n", get_short(p));
		printf("QCLAS: %d\n", get_short(p + 2));
		p += 4;
	}

	ancount = get_short(&data[6]);
	for (i = 0; i < ancount; i++) {
		pc = mc_open();
		p = read_labels(p, pc, data, size);
		printf("NAME: '%s'\n", mc_close(pc));
		printf("TYPE: %d\n", get_short(p));
		printf("CLAS: %d\n", get_short(p + 2));
		printf("TTL:  %u\n", get_long(p + 4));
		rdlength = get_short(p + 8);
		printf("RDLN: %d\n", rdlength);
		p += 10 + rdlength;
	}
}

static void dns_got_message(struct dns_act *da, char *src, char *data, int size){
	int len;
	char *msg;

	printf("GOT A DNS MESSAGE (src = '%s')\n", src == 0 ? "<unknown>" : src);
	dns_print_message(da, data, size);
	dns_make_message(da, data, &msg, &len);
	dns_print_message(da, msg, len);
	dns_udp_send(da, da->output, src, msg, len);
}

static void dns_handler(void *env, ev_t ev){
	struct dns_act *da = env;
	struct chunk *chunk;
	char *src;

	if (ev->type == evt_data) {
		chunk = ev_get_ptr(ev, evt_data_data);
		dns_got_message(da, 0, chunk->data, chunk->size);
	}
	else if (ev->type == evt_data_src) {
		chunk = ev_get_ptr(ev, evt_data_data);
		src = ev_get_ptr(ev, evt_data_src_addr);
		dns_got_message(da, src, chunk->data, chunk->size);
	}
	else if (ev->type == evt_close) {
		ev_channel_release(da->in);
		act_release(da->activity);
		free((char *) da);
	}

	ev_handled(ev);
}

static void dns_event_release(void *env, ev_t ev){
	ev_release(ev);
}

struct ev_channel *dns_create(struct ev_channel *output){
	struct dns_act *da = (struct dns_act *) calloc(1, sizeof(*da));

	da->activity = act_alloc(da);
	da->in = ev_channel_alloc(da->activity, dns_handler, da);
	da->release = ev_channel_alloc(da->activity, dns_event_release, da);
	da->output = output;
	return da->in;
}

/* Initialize this module.
 */
void dns_init(void){
	evt_data_data = evt_index(evt_data, "data");
	evt_data_src_addr = evt_index(evt_data_src, "addr");
	evt_error_descr = evt_index(evt_error, "descr");
}

void dns_done(void){
}
