/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include <stdlib.h>
#include "oslib.h"

/* An event is a set of typed properties.  The properties have names and values.  Creation
 * of these events, as well as access to the properties, should be as efficient as possible.
 * In many cases, a string lookup is not efficient enough.
 *
 * Our event architecture applies an event type hierarchy allowing us to prototype events
 * and provide efficient access.  Event types are created and managed with the following
 * interface.
 *
 *		evt_t evt_create(evt_t parent, char *name)
 *			Create a new event type.  It inherits all the properties of the given parent
 *			event type.  The type "name" is relative to the parent type.
 *		void evt_int_property(evt_t evt, char *name)
 *			Add a new integer property to the given type.
 *		void evt_ptr_property(evt_t evt, char *name, void (*release)(void *))
 *			Add a new pointer property to the given type.  The given release function
 *			is used for releasing events of the given type.
 *		void evt_set_print(evt_t evt, char *name, void (*print)(void *env, void *value, int how))
 *			Associate a print function with the given property.
 *		void evt_finalize(evt_t evt)
 *			Allow no more properties to be added to given event type.
 *		int evt_index(evt_t evt, char *name)
 *			Return the property index of the named property within the given, finalized,
 *			event type.
 *
 * Next, you can create events and access them:
 *
 *		ev_t ev_create(evt_t evt)
 *			Create a new event of the given, finalized, event type.  All the properties
 *			are initialized to null.
 *		void ev_set_int(ev_t ev, int index, int value)
 *			Set the integer value of an integer property.
 *		void ev_set_ptr(ev_t ev, int index, void *ptr)
 *			Set the pointer value of a pointer property.
 *		int ev_get_int(ev_t ev, int index)
 *			Get the integer value of an integer property.
 *		void *ev_get_ptr(ev_t ev, int index)
 *			Get the pointer value of a pointer property.
 *		void ev_print(ev_t ev, int how)
 *			Print the given event.  The amount of detail can be specified using
 *			the "how" value.
 *		void ev_release(ev_t ev)
 *			Release the event and its properties.
 */

struct property {
	char *name;
	enum { PROP_INT, PROP_PTR } type;
	void (*print)(void *env, struct event *, int how);
	void *env;
	union {
		struct {
			void (*release)(void *);
		} ptr;
	} u;
};

char *evt_name(evt_t evt){
	return evt->name;
}

evt_t evt_create(evt_t parent, char *name,
			ev_t (*create)(evt_t), void (*release)(evt_t, ev_t)){
	evt_t evt;

	evt = (evt_t) calloc(1, sizeof(*evt));
	evt->parent = parent;
	evt->name = mem_string_copy(name, strlen(name));
	evt->create = create;
	evt->release = release;
	return evt;
}

void evt_release(evt_t evt){
	ev_t ev;

	while ((ev = evt->free_list) != 0) {
		evt->free_list = ev->next;
		free((char *) ev);
	}
	free(evt->name);
	free((char *) evt);
}

/* Create an empty event.
 */
ev_t ev_empty(evt_t evt, int size){
	ev_t ev = (ev_t) calloc(1, size);

#ifdef debug
	printf("allocate empty '%s' event\n", evt->name);
#endif
	ev->type = evt;
	return ev;
}

/* Allocate a new event.
 */
ev_t ev_create(evt_t evt){
	ev_t ev;

	/* Try to get one of the free list for this event.
	 */
	if ((ev = evt->free_list) != 0) {
#ifdef debug
		printf("allocate '%s' event from free list --> %x\n", evt->name,
							(unsigned int) ev);
#endif
		evt->free_list = ev->next;
		ev->next = 0;
		return ev;
	}

	/* No luck.  Allocate one for real.
	 */
	ev = (*evt->create)(evt);
#ifdef debug
	printf("allocate new '%s' event --> %x\n", evt->name, (unsigned int)ev);
#endif
	return ev;
}

void ev_release(ev_t ev){
	evt_t evt = ev->type;

#ifdef debug
	printf("release '%s' event %x\n", evt->name, (unsigned int)ev);
#endif

	/* Call the type-specific release function.
	 */
	if (evt->release != 0) {
		(*evt->release)(evt, ev);
	}

	/* Tell the scheduler. 
	 */
	ev_released(ev);

	/* Place it on the free list.
	 */
	ev->next = evt->free_list;
	evt->free_list = ev;
}
