/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include <stdlib.h>
#include <assert.h>
#include "graph.h"

#define GET2D(m,n,x,y) (m)[(x) * (n) + (y)]

struct graph {
  char *name;
  int nnodes;
  char **node_names;
  char **node_locations;
  int *adj_matrix;
};

graph new_graph(int nnodes, char *name) {
/*   int i, j; */
  graph g;
  assert(nnodes >= 0);
  g = malloc(sizeof(*g));
  g->name = name;
  g->nnodes = nnodes;
  if (nnodes > 0) {
    g->adj_matrix = (int *) calloc(nnodes * nnodes,sizeof(int));
    assert(g->adj_matrix != NULL);
/*   for (i = 0; i<nnodes; i++) */
/*     for (j = 0; j<nnodes; j++) */
/*       GET2D(g->adj_matrix,nnodes,i,j) = -1; */
    g->node_names = (char **) calloc(nnodes,sizeof(char *));
    assert(g->node_names != NULL);
    g->node_locations = (char **) calloc(nnodes,sizeof(char *));
    assert(g->node_locations != NULL);
  }
  else {
    g->node_names = NULL;
    g->node_locations = NULL;
    g->adj_matrix = NULL;
  }
    
  return g;
}

void free_graph(graph g) {
  free(g->adj_matrix);
  free(g->node_names);
  free(g->node_locations);
  free(g);
}

void add_edge(graph g, int from, int to) {
  assert(from < g->nnodes && from >= 0);
  assert(to < g->nnodes && to >= 0);
/*   printf("adding edge in graph %s from %s to %s\n", */
/* 	 g->name, g->nodes[from], g->nodes[to]); */
  GET2D(g->adj_matrix,g->nnodes,from,to) = 1;
}

void add_node(graph g, char *name, char *location, int id) {
  assert(id < g->nnodes && id >= 0);
  g->node_names[id] = name;
  g->node_locations[id] = location;  
}

int points_to(graph g, int from, int to) {
  assert(from < g->nnodes && from >= 0);
  assert(to < g->nnodes && to >= 0);
  return (GET2D(g->adj_matrix,g->nnodes,from,to)) == 1;
}

/* struct list { */
/*   int node; */
/*   struct list *tl; */
/* }; */
/* typedef struct list *list; */

/* void free_list(list l) { */
/*   while (l != NULL) { */
/*     list tmp = l->tl; */
/*     free(l); */
/*     l = tmp; */
/*   } */
/* } */

/* list rev(list x) { */
/*   if (x == NULL) return x; */
/*   else { */
/*     list first  = x; */
/*     list second = x->tl; */
/*     x->tl = NULL; */
/*     while (second != NULL) { */
/*       list temp = second->tl; */
/*       second->tl = first; */
/*       first = second; */
/*       second = temp; */
/*     }  */
/*     return first; */
/*   } */
/* } */

static void dfsvisit(graph g, int cc, int node, int *visited, int *ccs) {
  int i;
  visited[node] = 1;
  ccs[node] = cc;
  for (i=0; i<g->nnodes; i++)
    if ((points_to(g,node,i) || points_to(g,i,node)) && !visited[i])
      dfsvisit(g,cc,i,visited,ccs);
}

/* calculates the connected components for the graph.  Returns
   the rest as [comps], which is an array of length [num]. */
void get_connected_comps(graph g, int **comps, int *num) {
  int i, cc, *visited, *ccs;

  if (g->nnodes == 0) {
    *num = 0;
    return;
  }

  /* calculate which components are connected */
  visited = calloc(g->nnodes,sizeof(int));
  ccs = calloc(g->nnodes,sizeof(int));
  for (i=0, cc=0; i<g->nnodes; i++) {
    if (!visited[i]) {
      dfsvisit(g,cc,i,visited,ccs);
      cc++;
    }
  }
  *num = cc;
  *comps = ccs;
  free(visited);
}

/* list topological_sort(graph g) */
/* { */
/*   int i, *visited; */
/*   list l = NULL; */
/*   visited = calloc(g->nnodes,sizeof(int)); */
/*   for (i=0; i<g->nnodes; i++) */
/*     dfsvisit(g,i,visited,&l); */
/*   return l; */
/* /\*   return rev(l); *\/ */
/* } */

/* void print_topsort_graph(graph g, struct print_channel *pc) { */
/*   list l = topological_sort(g); */
/*   list x; */
  
/*   pc_print(pc,"%s: ",g->name); */
/*   for (x=l; x!=NULL; x=x->tl) { */
/*     pc_print(pc, "%s ",g->node_names[x->node]); */
/*     if (x->tl != NULL && points_to(g,x->node,x->tl->node)) */
/*       pc_print(pc, "-> "); */
/*     else */
/*       pc_print(pc, "   "); */
/*   } */
/*   pc_print(pc,"\n"); */
/*   free_list(l); */
/* } */

void print_adj_graph(graph g, struct print_channel *pc) {
  int i, j;
  pc_print(pc,"%s:\n",g->name);

  for (i=0; i<g->nnodes; i++) {
    pc_print(pc,"  (%s: ",g->node_names[i]);
    for (j=0; j<g->nnodes; j++) {
      if (GET2D(g->adj_matrix,g->nnodes,i,j))
	pc_print(pc,"%s ",g->node_names[j]);
    }
    pc_print(pc,")\n");
  }
}

/* Prints the output in "dot" graph format; see
   http://www.research.att.com/sw/tools/graphviz/ */
void print_dot_graph(graph g, struct print_channel *pc) {
  int i,j;
  int *node_map;
  pc_print(pc,"digraph %s {\n",g->name);
  pc_print(pc,"  rankdir=\"LR\";\n");
  pc_print(pc,"  ranksep=0.1;\n");

  /* first print the edges */
  for (i=0; i<g->nnodes; i++) {
    int did_print = 0;
    for (j=0; j<g->nnodes; j++) {
      if (GET2D(g->adj_matrix,g->nnodes,i,j)) {
	if (!did_print) {
	  pc_print(pc,"  ");
	  did_print = 1;
	}
	pc_print(pc,"%s->%s; ",g->node_names[i],g->node_names[j]);
      }
    }
    if (did_print)
      pc_print(pc,"\n");
  }

  /* now do the clustering on particular nodes */
  node_map = calloc(g->nnodes,sizeof(int));
  for (i=0; i<g->nnodes; i++) {
    if (!node_map[i]) {
      char *location;
      node_map[i] = 1;
      location = g->node_locations[i];
      pc_print(pc,"  subgraph \"cluster_%s\" { label = \"%s\"; %s",
	       location, location, g->node_names[i]);
      for (j=i+1; j<g->nnodes; j++) {
	if (!node_map[j] && 
	    !strcmp(g->node_locations[j],
		    g->node_locations[i])) {
	  node_map[j] = 1;
	  pc_print(pc,"; %s",g->node_names[j]);
	}
      }
      pc_print(pc,"; }\n\n");
    }
  }
  pc_print(pc,"}\n");
  free(node_map);
}
