/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* This event handler processes data events coming from an HTTP request or reply
 * stream.  The headers on these streams are parsed, and output using a evt_http
 * event that has the following three fields:
 *		start:	the first line of the request
 *		map:	a ternary search tree with the fields
 *		size:	the size of the body
 *
 * If "size" is non-zero, one or more evt_data events totalling that size will follow
 * the evt_http event containing (fragments of) the body.
 */
#include <stdlib.h>
#include <stdio.h>
#include "oslib.h"
#include "http.h"

/* Flow control window.
 */
#define LO_WATER			2
#define HI_WATER			4

/* This module creates a new event type.
 */
evt_t evt_http;

struct http_act {
	struct activity *activity;
	struct ev_channel *in;			/* input event channel */
	struct ev_channel *release;		/* event release channel */
	struct ev_channel *dst;			/* output event channel */
	enum {
		HR_START,					/* awaiting the first line of the HTTP header */
		HR_FIELD_NAME,				/* awaiting the name of a field */
		HR_FIELD_VALUE,				/* awaiting the value of a field */
		HR_FIELD_EOLN,				/* just parsed LF character in a field value */
		HR_BODY,					/* awaiting the end of the body of an HTTP POST request */
		HR_ERROR					/* a parse error occurred */
	} state;
	bool_t lws;						/* linear white space */
	ev_t ev;						/* next output event */
	char *field_name;				/* name of field we're parsing */
	map_t map;						/* map of current set of fields */
	struct print_channel *pc;		/* print channel in which chars are temporarily stored */
	bool_t post;					/* parsing a post request (with a body) */
	int content_length;				/* remaining content_length in post request body */
};

/* Clean up a request parser activity.  Sends a close event on the output channel.
 */
static void http_release(struct http_act *ha){
	if (ha->field_name != 0) {
		free(ha->field_name);
	}
	tst_release(&ha->map, mem_release, 0);
	if (ha->pc != 0) {
		mc_release(ha->pc);
	}
	ev_channel_release(ha->release);
	act_release(ha->activity);
	free((char *) ha);
}

/* Add the next field to the map.
 */
static void http_add(struct http_act *ha){
	char *value, *tmp;
	void **p;

	value = mc_close(ha->pc);
	ha->pc = 0;

	/* Find or create the field.  If it was already there, concatenate the values.
	 */
	p = tst_find(&ha->map, ha->field_name, 1);
	if (*p != 0) {
		tmp = mem_print("%s,%s", *p, value);
		free(*p);
		*p = tmp;
	}
	else {
		*p = value;
	}
	free(ha->field_name);
	ha->field_name = 0;
}

/* Invoked when a parse error occurs.  Outputs an error event and puts the parser in an
 * error state in which nothing more can be parsed.
 */
static void http_error(struct http_act *ha, char *descr){
	ev_t ev;

	ev = ev_create(evt_error);
	ev_cast(ev, evt_error)->descr = "HTTP error";
	ev_send(ha->release, ha->dst, ev);
	ha->state = HR_ERROR;
}

/* Return this parser to the initial state.
 */
static void http_reset(struct http_act *ha){
	ha->state = HR_START;
	ha->pc = mc_open();
}

static void http_ev_send(struct http_act *ha, ev_t ev){
	/* Send the event.
	 */
	ev_send(ha->release, ha->dst, ev);

	/* Flow control:  if too many outstanding events, block input.  This
	 * flag gets unset in the event release handler.
	 */
	if (act_outstanding(ha->activity) > HI_WATER) {
		ev_set_blocked(ha->in, true);
	}
}

/* An HTTP message has been fully parsed.  Send an event on the output event channel.
 */
static void http_send(struct http_act *ha){
	/* Add the map to the output event.
	 */
	ev_cast(ha->ev, evt_http)->map = ha->map;
	ha->map = 0;

	/* Also add how many bytes are following this event.
	 */
	ev_cast(ha->ev, evt_http)->size = ha->content_length;

	http_ev_send(ha, ha->ev);
	ha->ev = 0;
}

/* If this character is an uppercase character, convert it to lower case.
 */
static char to_lower(char c){
	if ('A' <= c && c <= 'Z') {
		return 'a' + (c - 'A');
	}
	return c;
}

/* Another character of header arrived.
 */
static void http_input(struct http_act *ha, char c){
	char *tmp;
	int len;
	void **p;

	switch (ha->state) {
	case HR_START:
		/* We're in the process of parsing the first line.  Wait for the final carriage return.
		 */
		switch (c) {
		case '\r':
			/* ignore */
			break;
		case '\n':
			/* See what line we got.
			 */
			len = mc_size(ha->pc);
			tmp = mc_close(ha->pc);
			ha->pc = 0;

			/* If there was nothing there, HTTP/1.1 requires us to wait for more.
			 */
			if (len == 0) {
				free(tmp);
				ha->pc = mc_open();
				break;
			}

			/* Sanity check.
			 */
			if (len >= 2100) {
				http_error(ha, "huge start line");
				free(tmp);
				break;
			}

			/* Allocate an event and save this line in it.
			 */
			ha->ev = ev_create(evt_http);
			ev_cast(ha->ev, evt_http)->start = tmp;

			/* Now start awaiting fields.
			 */
			ha->state = HR_FIELD_NAME;
			ha->pc = mc_open();
			break;
		default:
			/* Default action: add character to print channel.
			 */
			pc_putchar(ha->pc, c);
		}
		break;
	case HR_FIELD_NAME:
		/* We're awaiting the next field name.
		 */
		switch (c) {
		case '\r': case '\n':
			/* Can't have CR or LF in field names.
			 */
			http_error(ha, "CRLF in field name");
			break;
		case ':':
			/* End of field name.
			 */
			ha->field_name = mc_close(ha->pc);
			ha->state = HR_FIELD_VALUE;
			ha->lws = false;
			ha->pc = mc_open();
			break;
		default:
			/* Field names are case-insensitive.
			 */
			pc_putchar(ha->pc, to_lower(c));
		}
		break;
	case HR_FIELD_VALUE:
		/* We're awaiting the end of the value, which is kind of tricky because it
		 * may continue over multiple lines.
		 */
		switch (c) {
		case '\r':
			/* ignore */
			break;
		case ' ': case '\t':
			/* Just remember this.  Multiple white space characters can be combined into a single space.
			 */
			ha->lws = true;
			break;
		case '\n':
			/* This may or may not be the end of the field.  We'll see...
			 */
			ha->state = HR_FIELD_EOLN;
			break;
		default:
			/* Linear white space is combined into a single space, unless it's leading.
			 */
			if (mc_size(ha->pc) != 0 && ha->lws) {
				pc_putchar(ha->pc, ' ');
			}
			pc_putchar(ha->pc, c);
			ha->lws = false;
		}
		break;
	case HR_FIELD_EOLN:
		switch (c) {
		case '\r':
			/* ignore */
			break;
		case '\n':
			/* This is the end of the field value for sure, and also the end of the header.
			 */
			http_add(ha);

			/* See if it has a body.
			 */
			p = tst_find(&ha->map, "content-length", 0);
			if (p != 0 && *p != 0) {
				ha->state = HR_BODY;
				if ((ha->content_length = atoi(*p)) < 0) {
					ha->content_length = 0;
				}
			}
			else {
				ha->content_length = 0;
			}

			/* Send the header.
			 */
			http_send(ha);

			/* Reset the state if there's no body.
			 */
			if (ha->content_length == 0) {
				http_reset(ha);
			}
			break;
		case ' ': case '\t':
			/* This must be a continuation of the field value.
			 */
			ha->lws = true;
			ha->state = HR_FIELD_VALUE;
			break;
		default:
			/* This is the end of the field value, and the first character of the next
			 * field name.
			 */
			http_add(ha);
			ha->state = HR_FIELD_NAME;
			ha->pc = mc_open();
			pc_putchar(ha->pc, to_lower(c));
		}
		break;
	default:
		err_fatal("http_input: bad state");
	}	
}

/* This function deals with HTTP input.
 */
void http_handler(void *env, struct event *ev){
	struct http_act *ha = env;
	struct chunk *chunk, *out_chunk;
	char *data;
	int i;
	ev_t out_ev;

	/* If it's data, parse it.
	 */
	if (ev->type == evt_data) {
		chunk = ev_cast(ev, evt_data)->chunk;
		data = chunk->data;
#ifdef debug
		printf("GOT '%.*s'\n", chunk->size, chunk->data);
#endif
		for (i = 0; i < chunk->size;) {
			switch (ha->state) {
			case HR_ERROR:
				/* Ignore input.
				 */
				i = chunk->size;
				break;
			case HR_BODY:
				/* Get the largest chunk possible and send it on the
				 * output channel.  First see if we can steal the
				 * entire thing.
				 */
				out_ev = ev_create(evt_data);
				if (i == 0 && chunk->size <= ha->content_length) {
					ev_cast(ev, evt_data)->chunk = 0;
					out_chunk = chunk;
				}

				/* Otherwise we'll have to copy as much as possible.
				 */
				else {
					out_chunk = (struct chunk *) calloc(1, sizeof(*out_chunk));
					out_chunk->size = chunk->size - i;
					if (out_chunk->size > ha->content_length) {
						out_chunk->size = ha->content_length;
					}
					out_chunk->data = malloc(out_chunk->size);
					memcpy(out_chunk->data, &chunk->data[i], out_chunk->size);
				}
				ev_cast(out_ev, evt_data)->chunk = out_chunk;

				/* Adjust the state.
				 */
				ha->content_length -= out_chunk->size;
				if (ha->content_length == 0) {
					http_reset(ha);
				}
				i += out_chunk->size;

				/* Send the event.
				 */
				http_ev_send(ha, out_ev);
				break;
			default:
				/* Otherwise it's part of the header, and we'll just parse
				 * one character at a time.
				 */
				http_input(ha, data[i]);
				i++;
			}
		}
	}

	/* If it's a close event, clean up everything.
	 */
	else if (ev->type == evt_close) {
		ev_send(ha->release, ha->dst, ev_create(evt_close));
		ev_channel_release(ha->in);
		ha->in = 0;
	}

	/* If we get a bad input event, we output an error event.
	 */
	else {
		http_error(ha, "bad input event");
	}

	/* Done with the input event.
	 */
	ev_handled(ev);
}

/* The event release handler.
 */
void http_event_release(void *env, struct event *ev){
	struct http_act *ha = env;

	ev_release(ev);

	/* If the input channel is closed, clean up.
	 */
	if (ha->in == 0) {
		if (act_outstanding(ha->activity) == 0) {
			http_release(ha);
		}
	}

	/* Otherwise see if we can open up our input channel.
	 */
	else if (act_outstanding(ha->activity) < LO_WATER) {
		ev_set_blocked(ha->in, false);
	}
}

/* Create a new HTTP parser activity.
 */
struct ev_channel *http_create(struct ev_channel *dst){
	struct http_act *ha = (struct http_act *) calloc(1, sizeof(*ha));

	ha->activity = act_alloc(ha);
	ha->in = ev_channel_alloc(ha->activity, http_handler, ha);
	ha->release = ev_channel_alloc(ha->activity, http_event_release, ha);
	ha->dst = dst;
	http_reset(ha);
	return ha->in;
}

ev_t evt_http_create(evt_t evt){
	return ev_empty(evt, sizeof(struct evt_http));
}

void evt_http_release(evt_t evt, ev_t ev){
	struct evt_http *evh = ev_cast(ev, evt_http);

	if (evh->start != 0) {
		free(evh->start);
		evh->start = 0;
	}
	if (evh->map != 0) {
		tst_release(&evh->map, mem_release, 0);
	}
	evh->size = 0;
}

/* Initialize this module.
 */
void http_init(void){
	extern evt_t evt_root;

	evt_http = evt_create(evt_root, "http", evt_http_create, evt_http_release);
}

void http_done(void){
	evt_release(evt_http);
}
