/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include "oslib.h"

int rmdir(const char *), chdir(const char *);

#ifdef WIN32

#include <io.h>

int mkdir(const char *);

struct dir_handle {
	long handle;
	char *saved;
};

dir_t dir_open(char *name){
	dir_t dir = (dir_t) calloc(1, sizeof(*dir));
	struct _finddata_t fi;

	name = mem_print("%s\\*.*", name);
	dir->handle = _findfirst(name, &fi);
	if (dir->handle == -1) {
		err_warning("dir_open: _find_first '%s' failed", name);
		free(name);
		free((char *) dir);
		return 0;
	}
	dir->saved = mem_string_copy(fi.name, strlen(fi.name));
	free(name);
	return dir;
}

char *dir_next(dir_t dir){
	char *name;
	struct _finddata_t fi;

	if ((name = dir->saved) != 0) {
		dir->saved = 0;
		return name;
	}
	if (_findnext(dir->handle, &fi) != 0)
		return 0;
	return mem_string_copy(fi.name, strlen(fi.name));
}

void dir_close(dir_t dir){
	(void) _findclose(dir->handle);
	if (dir->saved != 0)
		free(dir->saved);
	free((char *) dir);
}

#else /* WIN32 */

#include <unistd.h>
#include <dirent.h>

int mkdir(const char *path, mode_t mode);

struct dir_handle {
	DIR *handle;
};

dir_t dir_open(char *name){
	dir_t dir = (dir_t) calloc(1, sizeof(*dir));

	if ((dir->handle = opendir(name)) == 0) {
		err_warning("dir_open: can't open '%s'", name);
		free((char *) dir);
		return 0;
	}
	return dir;
}

char *dir_next(dir_t dir){
	struct dirent *d;

	if ((d = readdir(dir->handle)) == 0)
		return 0;
	return mem_string_copy(d->d_name, strlen(d->d_name));
}

void dir_close(dir_t dir){
	closedir(dir->handle);
	free((char *) dir);
}

#endif /* WIN32 */

/* To become obsolete.
 */
char *fp_read(FILE *fp){
	char *mem = 0;
	int datasize, realsize = 0, n;

	for (;;) {
		datasize = realsize + 513;
		mem = realloc(mem, datasize);
		n = fread(&mem[realsize], 1, 512, fp);
		if (n == 0)
			break;
		realsize += n;
	}
	mem[realsize] = 0;
	return mem;
}

void pc_read(struct print_channel *pc, FILE *fp){
	char buf[512];
	int n;

	for (;;) {
		n = fread(buf, 1, 512, fp);
		if (n == 0) {
			return;
		}
		pc_puts(pc, buf, n);
	}
}

/* Read everything available from the given file pointer, and return a (null-terminated)
 * string, as well as the length of the string (excluding the null-byte) in *len.
 */
char *fp_raw_read(FILE *fp, int *len){
	char *mem = 0;
	int datasize, realsize = 0, n;

	for (;;) {
		datasize = realsize + 513;
		mem = realloc(mem, datasize);
		n = fread(&mem[realsize], 1, 512, fp);
		if (n == 0) {
			break;
		}
		realsize += n;
	}
	mem[realsize] = 0;
	if (len != 0) {
		*len = realsize;
	}
	return mem;
}

/* Write the given string to the given file pointer.
 */
void fp_write(FILE *fp, char *data){
	unsigned int size = strlen(data);

	if (fwrite(data, 1, size, fp) != size)
		err_warning("fp_write: bad size written");
}

bool_t file_exists(char *file){
	FILE *fp;

	if ((fp = fopen(file, "r")) == 0)
		return false;
	fclose(fp);
	return true;
}

/* To become obsolete.
 */
char *file_read(char *file){
	FILE *fp;
	char *data;

#ifdef WIN32
	fp = fopen(file, "rb");
#else
	fp = fopen(file, "r");
#endif
	if (fp == 0) {
		perror(file);
		err_warning("file_read: can't open '%s'", file);
		return 0;
	}
	data = fp_read(fp);
	fclose(fp);
	return data;
}

bool_t pc_read_file(struct print_channel *pc, char *file){
	FILE *fp;

#ifdef WIN32
	fp = fopen(file, "rb");
#else
	fp = fopen(file, "r");
#endif
	if (fp == 0) {
		err_warning("pc_read_file: can't open %s", file);
		return false;
	}
	pc_read(pc, fp);
	fclose(fp);
	return true;
}

/* Read the given file.  Return the data (to be released with free(), and the length in *len.
 */
char *file_raw_read(char *file, int *len){
	FILE *fp;
	char *data;

#ifdef WIN32
	fp = fopen(file, "rb");
#else
	fp = fopen(file, "r");
#endif
	if (fp == 0) {
		perror(file);
		err_warning("file_raw_read: can't open '%s'", file);
		return 0;
	}
	data = fp_raw_read(fp, len);
	fclose(fp);
	return data;
}

bool_t file_write(char *file, char *data){
	FILE *fp;

#ifdef WIN32
	fp = fopen(file, "wb");
#else
	fp = fopen(file, "w");
#endif
	if (fp == 0) {
		err_warning("file_write: can't create %s", file);
		return false;
	}
	fp_write(fp, data);
	fclose(fp);
	return true;
}

bool_t file_remove(char *file){
	return unlink(file) == 0;
}

bool_t dir_create(char *dir){
#ifdef WIN32
	return mkdir(dir) == 0;
#else
	return mkdir(dir, 0750) == 0;
#endif
}

bool_t dir_remove(char *dir){
	return rmdir(dir) == 0;
}

bool_t dir_change(char *dir){
	return chdir(dir) == 0;
}
