/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include "oslib.h"

struct log_sub {
	struct log_sub *next;
	int level;
	void (*callback)(void *env, char *);
	void *env;
};

static struct log_sub *subscribers;
static struct print_channel *outlog;
static bool_t flushing;

void log_init(void){
}

/* Flush whatever is logged by invoking all subscribers and
 * truncating the log.
 */
void log_flush(int level){
	char *s;
	struct log_sub *ls;

	/* If there's no log, we're done.
	 */
	if (outlog == 0)
		return;

	/* Don't allow recursion here.  Likely to be infinite...
	 */
	if (flushing)
		return;
	flushing = true;

	s = mc_close(outlog);
	outlog = 0;
	for (ls = subscribers; ls != 0; ls = ls->next) {
		if (level <= ls->level)
			(*ls->callback)(ls->env, s);
	}
	free(s);

	flushing = false;
}

void log_vprintf(char *fmt, va_list ap){
	if (outlog == 0)
		outlog = mc_open();
	pc_vprint(outlog, fmt, ap);
}

void log_prnt(char *format, ...){
	va_list ap;

	va_start(ap, format);
	log_vprintf(format, ap);
	va_end(ap);
}

void log_printf(char *format, ...){
	va_list ap;

	va_start(ap, format);
	log_vprintf(format, ap);
	va_end(ap);
	log_flush(0);
}

struct log_sub *log_subscribe(int level, void (*callback)(void *, char *), void *env){
	struct log_sub *ls;

	ls = (struct log_sub *) calloc(1, sizeof(*ls));
	ls->level = level;
	ls->callback = callback;
	ls->env = env;
	ls->next = subscribers;
	subscribers = ls;
	return ls;
}

static void log_free(struct log_sub *ls){
	(*ls->callback)(ls->env, 0);
	free((char *) ls);
}

void log_unsubscribe(struct log_sub *ls){
	struct log_sub **pls;

	for (pls = &subscribers; *pls != 0; pls = &(*pls)->next)
		if (*pls == ls) {
			*pls = ls->next;
			log_free(ls);
			return;
		}
	err_warning("log_unsubscribe: no such subscription???");
}

void log_release(void){
	struct log_sub *ls;

	log_flush(0);
	while ((ls = subscribers) != 0) {
		subscribers = ls->next;
		log_free(ls);
	}
}
