/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include "oslib.h"
#include <string.h>

#define MAX_SIZE	(1024 * 1024)
#define ALLOC_MAGIC	0xA55AA55A
#define FREE_MAGIC	0x5AA55AA5
#define MAX_FREE	128

#undef malloc
#undef calloc
#undef realloc
#undef free

/*
char *malloc(int), *realloc(char *, int), *calloc(int, int);
*/

struct header {
	unsigned int size;
	unsigned int magic;
	char *file;
	int line;
	unsigned int trace;
	struct header *next;
	struct header **prev;
	int unused;
};

int malloc_debug = 1;

static char *free_list[MAX_FREE];
static int free_next;
static int alloced, freed;
static struct header *alloc_list;

char *Mem_Malloc(int size, char *file, int line){
	struct header *hdr;
	char *mem;
	int i;

#ifdef notdef
	if (size == 0)
		err_fatal("Malloc: size == 0");
#endif
	if (size < 0)
		err_fatal("Malloc: size < 0");
	if (size > MAX_SIZE)
		err_fatal("Malloc: size > MAX");

	if (!malloc_debug)
		return malloc(size);

	hdr = (struct header *) malloc(sizeof(*hdr) + size + 1);
	if (hdr == 0)
		err_fatal("Malloc: malloc --> 0");
	hdr->size = size;
	hdr->magic = ALLOC_MAGIC;
	hdr->file = file;
	hdr->line = line;
	if ((hdr->next = alloc_list) != 0) {
		hdr->next->prev = &hdr->next;
	}
	hdr->prev = &alloc_list;
	alloc_list = hdr;
	hdr->trace = 0;
	for (i = 0, mem = (char *) &hdr[1]; i < size; i++)
		mem[i] = -1;
	mem[size] = (char)0x99;
	alloced += size;
	return mem;
}

static void Mem_Dump(struct header *hdr, char *where){
	fprintf(stderr, "Dump from %s\n", where);
}

void Mem_Free(char *mem, char *file, int line){
	struct header *hdr;
	int i, size;

	if (mem == 0)
#ifdef notdef
		err_fatal("Free: null pointer");
#else
		return;
#endif
	if ((int) mem & 3)
		err_fatal("Free: unaligned memory");

	if (!malloc_debug) {
		free(mem);
		return;
	}

	hdr = (struct header *) mem - 1;
	if (hdr->size > MAX_SIZE)
		err_fatal("Free: hdr->size > MAX");
	if (hdr->magic != ALLOC_MAGIC)
		err_fatal("Free: bad magic");
	if ((mem[hdr->size] & 0xFF) != 0x99)
		err_fatal("Free: bad end magic");
	if ((*hdr->prev = hdr->next) != 0) {
		hdr->next->prev = hdr->prev;
	}
	if (hdr->trace)
		Mem_Dump(hdr, "Free");
	freed += hdr->size;
	size = sizeof(*hdr) + hdr->size;
	for (i = 0, mem = (char *) hdr; i < size; i++)
		mem[i] = -1;
	hdr->magic = FREE_MAGIC;
	if (free_list[free_next] != 0) {
		free(free_list[free_next]);
	}
	free_list[free_next] = mem;
	if (++free_next == MAX_FREE)
		free_next = 0;
}

char *Mem_Realloc(char *mem, int size, char *file, int line){
	struct header *hdr, *hdr2;
	int i;

	if (mem == 0)
#ifdef notdef
		err_fatal("Realloc: null pointer");
#else
		return Mem_Malloc(size, file, line);
#endif
	if ((int) mem & 3)
		err_fatal("Realloc: unaligned memory");
#ifdef notdef
	if (size == 0)
		err_fatal("Realloc: size == 0");
#endif
	if (size < 0)
		err_fatal("Realloc: size < 0");
	if (size > MAX_SIZE)
		err_fatal("Realloc: size > MAX");

	if (!malloc_debug)
		return realloc(mem, size);

	hdr = (struct header *) mem - 1;
	if (hdr->size > MAX_SIZE)
		err_fatal("Realloc: hdr->size > MAX");
	if (hdr->magic != ALLOC_MAGIC)
		err_fatal("Realloc: bad magic");
	if ((mem[hdr->size] & 0xFF) != 0x99)
		err_fatal("Realloc: bad end magic");
	if ((*hdr->prev = hdr->next) != 0) {
		hdr->next->prev = hdr->prev;
	}
	if (hdr->trace)
		Mem_Dump(hdr, "Realloc");
	freed += hdr->size;
	alloced += size;
	hdr2 = (struct header *) malloc(sizeof(*hdr2) + size + 1);
	hdr2->size = size;
	hdr2->magic = ALLOC_MAGIC;
	hdr2->file = file;
	hdr2->line = line;
	if ((hdr2->next = alloc_list) != 0) {
		hdr2->next->prev = &hdr2->next;
	}
	hdr2->prev = &alloc_list;
	alloc_list = hdr2;
	hdr2->trace = 0;
	if (size > (int)hdr->size)
		size = hdr->size;
	memcpy(&hdr2[1], &hdr[1], size);
	for (i = size, mem = (char *) &hdr2[1]; i < (int)hdr2->size; i++)
		mem[i] = -1;
	mem[hdr2->size] = (char)0x99;
	size = sizeof(*hdr) + hdr->size;
	for (i = 0, mem = (char *) hdr; i < size; i++)
		mem[i] = -1;
	hdr->magic = FREE_MAGIC;
	if (free_list[free_next] != 0) {
		free(free_list[free_next]);
	}
	free_list[free_next] = mem;
	if (++free_next == MAX_FREE)
		free_next = 0;
	return (char *) &hdr2[1];
}

char *Mem_Calloc(int size, int n, char *file, int line){
	struct header *hdr;

#ifdef notdef
	if (size == 0 || n == 0)
		err_fatal("Calloc: size == 0");
#endif
	if (size < 0 || n < 0)
		err_fatal("Calloc: size < 0");

	if (!malloc_debug)
		return calloc(size, n);

	size *= n;
	if (size > MAX_SIZE)
		err_fatal("Calloc: size > MAX");
	hdr = (struct header *) malloc(sizeof(*hdr) + size + 1);
	if (hdr == 0)
		err_fatal("Calloc: malloc --> 0");
	hdr->size = size;
	hdr->magic = ALLOC_MAGIC;
	hdr->file = file;
	hdr->line = line;
	if ((hdr->next = alloc_list) != 0) {
		hdr->next->prev = &hdr->next;
	}
	hdr->prev = &alloc_list;
	alloc_list = hdr;
	hdr->trace = 0;
	memset(&hdr[1], 0, size);
	((char *) &hdr[1])[size] = (char)0x99;
	alloced += size;
	return (char *) &hdr[1];
}

void Mem_Trace(char *mem, int on){
	struct header *hdr;

	if (mem == 0)
		err_fatal("Trace: null pointer");
	if ((int) mem & 3)
		err_fatal("Trace: unaligned memory");

	if (!malloc_debug)
		return;

	hdr = (struct header *) mem - 1;
	if (hdr->size > MAX_SIZE)
		err_fatal("Trace: hdr->size > MAX");
	if (hdr->magic != ALLOC_MAGIC)
		err_fatal("Trace: bad magic");
	if ((mem[hdr->size] & 0xFF) != 0x99)
		err_fatal("Trace: bad end magic");
	if (hdr->trace)
		Mem_Dump(hdr, "Trace");
	if (on >= 0)
		hdr->trace = on;
}

void Mem_Done(void){
	int i;
	struct header *hdr;

	if (!malloc_debug)
		return;

	for (i = 0; i < MAX_FREE; i++) {
		if (free_list[i] != 0) {
			free(free_list[i]);
		}
	}

	printf("alloced = %d\n", alloced);
	printf("freed = %d\n", freed);

	for (hdr = alloc_list; hdr != 0; hdr = hdr->next) {
		printf("%s %d '%s'\n", hdr->file, hdr->line, (char *) &hdr[1]);
	}
}
