/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#include "oslib.h"

#define MEM_BUF_SIZE	(256 * 1024)

static char mem_buf[MEM_BUF_SIZE];

struct print_channel *mc_open(void){
	struct print_channel *pc;

	pc = (struct print_channel *) calloc(sizeof(*pc), 1);
	pc->real_length = 32;
	pc->buf = malloc(pc->real_length);
	pc->buf[0] = 0;
	pc->data_length = 1;
	return pc;
}

char *mc_close(struct print_channel *pc){
	char *r = pc->buf;

	free((char *) pc);
	return r;
}

void mc_release(struct print_channel *pc){
	free(mc_close(pc));
}

void pc_puts(struct print_channel *pc, const char *s, int len){
	/* Make sure the buffer is large enough.
	 */
	if (len > pc->real_length - pc->data_length) {
		pc->real_length = pc->data_length + len + 256;
		pc->buf = realloc(pc->buf, pc->real_length);
	}

	/* Copy the string into it and null-terminate it.
	 */
	memcpy(&pc->buf[pc->data_length - 1], s, len);
	pc->buf[pc->data_length + len - 1] = 0;
	pc->data_length += len;
}

void pc_vprint(struct print_channel *pc, const char *format, va_list ap){
	(void) vsprintf(mem_buf, format, ap);
	pc_puts(pc, mem_buf, strlen(mem_buf));
}

void pc_print(struct print_channel *pc, const char *format, ...){
	va_list ap;

	va_start(ap, format);
	pc_vprint(pc, format, ap);
	va_end(ap);
}

void pc_putc(struct print_channel *pc, int c){
	/* Make sure the buffer is large enough.
	 */
	if (pc->real_length == pc->data_length) {
		pc->real_length = pc->data_length + 256;
		pc->buf = realloc(pc->buf, pc->real_length);
	}

	/* Copy the character into it.
	 */
	pc->buf[pc->data_length - 1] = c;
	pc->buf[pc->data_length++] = 0;
}

char *mem_string_copy(const char *s, int size){
	char *copy = malloc(size + 1);

	memcpy(copy, s, size);
	copy[size] = 0;
	return copy;
}

char *mem_print(const char *format, ...){
	va_list ap;

	/* First do the conversion.
	 */
	va_start(ap, format);
	(void) vsprintf(mem_buf, format, ap);
	va_end(ap);

	return mem_string_copy(mem_buf, strlen(mem_buf));
}

/* I couldn't find a good way of printing double values in a normal way.
 * None of %e, %f, or %g does quite what I'd like.  I'd like to see the
 * decimals when they're there, but I don't want to see a long list of
 * trailing zeroes...  This function makes a printable version of a
 * double the way I happen to like it.
 */
char *mem_double(double val){
	static char output[64];
	char buf[64], decimals[64], *p, *q;
	
	int a, e;

	/* Convert into A.BeC format.
	 */
	sprintf(buf, "%.10e", val);

	/* Skip initial - if any.
	 */
	p = buf;
	if (*p == '-')
		p++;

	/* Parse result.
	 */
	sscanf(p, "%d.%[0-9]e%d", &a, decimals, &e);

	/* Remove trailing zeroes.
	 */
	for (p = &decimals[strlen(decimals)]; --p >= decimals;)
		if (*p != '0')
			break;
	*++p = 0;

	/* Start output with optional -
	 */
	p = output;
	if (buf[0] == '-')
		*p++ = '-';

	/* If exponent out of bounds, return exponented form.
	 */
	if (e < -4 || e > 5) {
		if (decimals[0] != 0)
			sprintf(p, "%d.%se%d", a, decimals, e);
		else
			sprintf(p, "%de%d", a, e);
		return output;
	}

	/* Concatenate decimals.
	 */
	sprintf(buf, "%d%s", a, decimals);

	/* Stick the decimal point in the right place.  Add 0's if necessary.
	 */
	if (e >= 0) {
		for (q = buf; *q != 0 && e >= 0; p++, q++, e--)
			*p = *q;
		for (; e >= 0; p++, e--)
			*p = '0';
		if (*q != 0)
			*p++ = '.';
		for (; *q != 0; p++, q++)
			*p = *q;
	}
	else {
		*p++ = '0'; *p++ = '.';	
		while (++e < 0)
			*p++ = '0';
		for (q = buf; *q != 0; p++, q++)
			*p = *q;
	}

	*p = 0;
	return output;
}

char *mem_alloc_double(double v){
	char *p = mem_double(v);

	return mem_string_copy(p, strlen(p));
}

void mem_release(void *env, void *value){
	free(value);
}
