/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* A multimedia scheduler.
 *
 * Input: a network of compute nodes, connected by networks, both of
 * limited capacity, and a set of user specifications
 *
 * Output: for each node, what computations they should run, and how
 * often.
 *
 * Each operator has a set of inputs, and a single output, each of a
 * specific frame frame type.  The compute cost of the operator
 * depends on other attributes of the frames, such as their sizes.
 *
 * In the output, each computation is always performed on frames that
 * only differ in their contents, not their attributes.
 */

#include <string.h>
#include <assert.h>
#ifdef __linux__
#include <stdint.h>
#endif

#include "mmsched.h"
#include "graph.h"
#include "timer.h"

/* #define debug */

#define MAX_PATH 100

struct topology *topology = NULL;	/* for dynamic configs */
int quiet = 0;

/*** various configurations ***/

struct node nodes_config1[] = {
	{ "uav1", 1e8 },
	{ "uav2", 1e8 },
	{ "distributor", 1e9 },
	{ "pc1", 1e8 },
	{ "pc2", 1e8 },
	{ "pc3", 1e8 },
	{ 0, 0 }
};
struct node nodes_config2[] = {
	{ "pc1", 1e8, "longhorn.cs.cornell.edu", 7771 },
	{ "pc2", 1e8, "longhorn.cs.cornell.edu", 7772 },
	{ 0, 0 }
};
struct node nodes_config3[] = {
	{ "pc1", 1e8, "robbert.cs.cornell.edu", 7771 },
	{ "pc2", 1e8, "robbert.cs.cornell.edu", 7772 },
	{ 0, 0 }
};
struct node nodes_config4[] = {
	{ "pc1", 1e8, "localhost", 7771 },
	{ "pc2", 1e8, "localhost", 7772 },
	{ 0, 0 }
};
struct node nodes_config5[] = {
	{ "pc1", 1e8, "localhost", 7771 },
	{ "pc2", 1e8, "localhost", 7772 },
	{ "pc3", 1e8, "localhost", 7773 },
	{ "pc4", 1e8, "localhost", 7774 },
	{ 0, 0 }
};
struct node nodes_config6[] = {
	{ "pc1", 1e8, "localhost", 7771 },
	{ "pc2", 1e8, "localhost", 7772 },
	{ "pc3", 1e8, "localhost", 7773 },
	{ 0, 0 }
};
struct node nodes_config7[] = {
	{ "pc1", 1e8, "localhost", 7771 },
	{ "pc2", 1e8, "localhost", 7772 },
	{ "pc3", 1e8, "localhost", 7773 },
	{ "pc4", 1e8, "localhost", 7774 },
	{ 0, 0 }
};

/*** emulab configs ***/

//diamond
struct node nodes_config8[] = {
        { "pc3", 1e8, "pc3.bowtie.medianet",7773 },
        { "pc4", 1e8, "pc4.bowtie.medianet",7774 },
        { "pc5", 1e8, "pc5.bowtie.medianet",7775 },
        { "pc6", 1e8, "pc6.bowtie.medianet",7776 },
        { 0, 0 }
};

//line
struct node nodes_config9[] = {
        { "pc3", 1e8, "pc3.bowtie.medianet",7773 },
        { "pc4", 1e8, "pc4.bowtie.medianet",7774 },
        { "pc5", 1e8, "pc5.bowtie.medianet",7775 },
        { 0, 0 }
};

//bowtie
struct node nodes_config10[] = {
        { "pc1", 1e8, "pc1.bowtie.medianet",7771 },
        { "pc2", 1e8, "pc2.bowtie.medianet",7772 },
        { "pc3", 1e8, "pc3.bowtie.medianet",7773 },
        { "pc4", 1e8, "pc4.bowtie.medianet",7774 },
        { "pc5", 1e8, "pc5.bowtie.medianet",7775 },
        { "pc6", 1e8, "pc6.bowtie.medianet",7776 },
        { "pc7", 1e8, "pc7.bowtie.medianet",7777 },
        { "pc8", 1e8, "pc8.bowtie.medianet",7778 },
        { 0, 0 }
};

#define NUM_CONFIGS 10

struct node *node_configs[NUM_CONFIGS] = {
  nodes_config1,
  nodes_config2,
  nodes_config3,
  nodes_config4,
  nodes_config5,
  nodes_config6,
  nodes_config7,
  nodes_config8,
  nodes_config9,
  nodes_config10
};

struct network networks_config1[] = {
	{ "link1", 1000000, 1000000, .01, "uav1 distributor" },
	{ "link2", 1000000, 1000000, .01, "uav2 distributor" },
	{ "lan", 10000000, 10000000, .001, "distributor pc1 pc2 pc3" },
	{ 0, 0 }
};
struct network networks_config2[] = {
	{ "link", 1000000, 1000000, .01, "pc1 pc2" },
	{ 0, 0 }
};
struct network networks_config3[] = {
	{ "link", 1000000, 1000000, .01, "pc1 pc2" },
	{ 0, 0 }
};
struct network networks_config4[] = {
	{ "link", 300000, 300000, .01, "pc1 pc2" },
	{ 0, 0 }
};
struct network networks_config5[] = {
	{ "link1", 300000, 300000, .01, "pc1 pc3" },
	{ "link2", 300000, 300000, .01, "pc3 pc2" },
	{ "link3", 300000, 300000, .01, "pc1 pc4" },
	{ "link4", 300000, 300000, .01, "pc4 pc2" },
	{ 0, 0 }
};
struct network networks_config6[] = {
	{ "link1", 300000, 300000, .01, "pc1 pc3" },
	{ "link2", 300000, 300000, .01, "pc3 pc2" },
	{ 0, 0 }
};
struct network networks_config7[] = {
	{ "link1", 300000, 300000, .01, "pc1 pc3" },
	{ "link2", 300000, 300000, .01, "pc3 pc4" },
	{ "link3", 300000, 300000, .01, "pc3 pc2" },
	{ 0, 0 }
};

/*** emulab configs ***/

//diamond
struct network networks_config8[] = {
  { "l3", 300000, 300000, .01, "pc3 pc5", { "192.168.8.2", "192.168.8.3" } },
  { "l4", 300000, 300000, .01, "pc3 pc6", { "192.168.2.2", "192.168.2.3" } },
  { "l5", 300000, 300000, .01, "pc5 pc4", { "192.168.3.2", "192.168.3.3" } },
  { "l6", 300000, 300000, .01, "pc6 pc4", { "192.168.5.2", "192.168.5.3" } },
  { 0, 0 }
};

//line
struct network networks_config9[] = {
  { "l3", 300000, 300000, .01, "pc3 pc5", { "192.168.8.2", "192.168.8.3" } },
  { "l5", 300000, 300000, .01, "pc5 pc4", { "192.168.3.2", "192.168.3.3" } },
  { 0, 0 }
};

//bowtie
struct network networks_config10[] = {
  { "l1", 300000, 300000, .01, "pc1 pc3", { "192.168.4.2", "192.168.4.3" } },
  { "l2", 300000, 300000, .01, "pc2 pc3", { "192.168.6.2", "192.168.6.3" } },
  { "l3", 100000, 300000, .01, "pc3 pc5", { "192.168.8.2", "192.168.8.3" } },
  { "l4", 200000, 300000, .01, "pc3 pc6", { "192.168.2.2", "192.168.2.3" } },
  { "l5", 300000, 300000, .01, "pc5 pc4", { "192.168.3.2", "192.168.3.3" } },
  { "l6", 300000, 300000, .01, "pc6 pc4", { "192.168.5.2", "192.168.5.3" } },
  { "l7", 100000, 300000, .01, "pc4 pc7", { "192.168.7.2", "192.168.7.3" } },
  { "l8", 300000, 300000, .01, "pc4 pc8", { "192.168.1.2", "192.168.1.3" } },
  { 0, 0 }
};

struct network *network_configs[NUM_CONFIGS] = {
  networks_config1,
  networks_config2,
  networks_config3,
  networks_config4,
  networks_config5,
  networks_config6,
  networks_config7,
  networks_config8,
  networks_config9,
  networks_config10
};

/************************/
/***** GLOBAL STATE *****/
/************************/

struct node *nodes;			/* current nodes description */
int nnodes;				/* size of nodes[] */
struct network *networks;		/* current net description */
int nlinks;				/* size of networks[] */
struct computation computations[MAX_COMPS]; /* current computations */
int nspec_comps;		/* # user-specified computations */
int ntotal_comps;		/* # computations including those
				   inserted by the global scheduler,
				   like monitor, send, recv, etc. */
int nconnected_comps = 0;	/* number of connected components */
int *connected_comps;		/* connected components (cc) map: comp
				   -> cc # */

struct computation opt_computations[MAX_COMPS]; /* current opt computations */
int nspec_opt_comps = 0;		/* # specified opt computations */

int comps_needing_ports[MAX_COMPS];	/* indicates the two send/recv
					   computations inserted by
					   the scheduler that need to
					   have their ports filled
					   in. */
int ncomps_needing_ports = 0;

int version;				/* current config version */
struct timeval last_reconfig_time;	/* last time we configured */

/* users and specifications */

struct spec {
	map_t comps;		/* set of computations */
	double utility;		/* utility value */
	struct computation **icomps;
	int num_icomps;
};

struct user {
	struct spec *alts;
	int nalts;
};

struct user_utility {
  struct user *u;
  double utility;
};

static map_t users;
static int num_users;

/* the utility vector for the current computations array */
struct user_utility *current_utility = NULL;
int current_users = 0;

/* user-alterable parameters */
struct sched_parameters parameters = {
  7777,        /* schedsvr_port */
  "localhost", /* schedsvr_host */
  LOW_PORT,    /* nextport */
  0,           /* emulab_link */
  5,           /* min reconfig delta */
  1,           /* min creep delta */
  0.03         /* creep rate */
};

/* routine for building/accessing our own 2D arrays */
#define GET2D(m,n,x,y) (m)[(x) * (n) + (y)]

#define NETWORK(x, y)		GET2D(connection_matrix,nnodes,(x)-1,(y)-1)
#define CONNECTED(x, y)		(NETWORK((x), (y)) != 0)

/* TODO.  The connection_matrix describe connections between hosts.
 * It might be better to describe connections between * interfaces, as
 * there may be more than one connection * between any two hosts, and
 * they may have different * capacities.  */
int *connection_matrix;		/* nnodes x nnodes */

/**************************/
/* MISCELLANEOUS ROUTINES */
/**************************/

static double network_score();

/* See if a matches the first len bytes of b.
 */
int string_match(char *a, char *b, unsigned int len){
	if (strlen(a) != len) {
		return 0;
	}
	return strncmp(a, b, len) == 0;
}

char *string_copy(char *s){
	return mem_string_copy(s, strlen(s));
}

/* Given a pointer to a string s, returns the address of the next
   token starting in s, along with its length in len.  If no more
   tokens exist, s is moved to the NULL terminator, and NULL is
   returned. */
static char *next_string(char **s, int *len) {
  char *q, *p = *s;
  while (*p == ' ') {
    p++;
  }
  /* found the beginning of the string */
  if (*p == 0) {
    *s = p;
    return NULL;
  }
  /* find the end */
  for (q = p; *q != 0 && *q != ' '; q++);
  /* Move up the string to the next token, set the length,
     and return the pointer to the token. */
  *s = q;
  *len = q-p;
  return p;
}

/* Returns 1 iff computations are structurally equivalent, and 0 otherwise */
int comp_equals(struct computation *comp1, struct computation *comp2) {
  if (comp1 == comp2)
    return 1;
  else {
    if (!strcmp(comp1->name,comp2->name)) {
      if ((comp1->interval == comp2->interval) &&
	  (comp1->enforce == comp2->enforce) &&
	  (comp1->location == comp2->location) &&
	  (comp1->output_size == comp2->output_size) &&
	  (!strcmp(comp1->input, comp2->input)))
	/* TODO: need to check that the maps are equal as well */
	return 1;
      else {
	err_warning("Components share the name %s but have different parameters\n",comp1->name);
	return 0;
      }
    }
    return 0;
  }
}

/*******************/
/* GRAPH FUNCTIONS */
/*******************/

/* Given a list of nodes (attached to some network link), set the bits
 * in the connection_matrix to (net + 1) to show that they are
 * connected by the given network.  (0 means not connected.)  */
void connect_in_matrix(int net, char *members){
	char *p;
	int node, x, y;

	/* First clear membership marks.
	 */
	for (node = 0; node < nnodes; node++) {
		nodes[node].member = 0;
	}

	/* Next mark the nodes that are on this network.
	 */
	for (;;) {
	  int len;

	  /* Find the next input */
	  p = next_string(&members, &len);
	  if (p == NULL)
	    break;

	  for (node = 0; node < nnodes; node++) {
	    if (string_match(nodes[node].name, p, len)) {
	      break;
	    }
	  }
	  if (node == nnodes) {
	    fprintf(stderr, "no node '%.*s'\n", len, p);
	    exit(1);
	  }
	  nodes[node].member = 1;
	}

	/* Finally connect them up.
	 */
	for (x = 1; x <= nnodes; x++) {
	  if (!nodes[x-1].member) {
	    continue;
	  }
	  for (y = 1; y <= nnodes; y++) {
	    if (!nodes[y-1].member) {
	      continue;
	    }
	    NETWORK(x,y) = net + 1;
/*  	    connection_matrix[x * nnodes + y] = net + 1; */
	  }
	}
}

/* Allocate the connection matrix.  As a side-effect, also calculate
 * nnodes.  The matrix has nnodes x nnodes entries.  In entry (x, y),
 * put the network that connects node x to node y.  The network is the
 * index in the networks array (+ 1).  */
void create_connection_matrix(void){
	int node, net;

	for (node = 0; nodes[node].name != 0; node++)
		;
	nnodes = node;
	connection_matrix = (int *) calloc(nnodes * nnodes, sizeof(int));

	for (net = 0; networks[net].name != 0; net++) {
		connect_in_matrix(net, networks[net].nodes);
	}
	nlinks = net;
}

/* Used for dataflow trees in the CMN */
struct pathtree {
  enum { NODE, LEAF } type;
  union {
    int dst_comp;
    struct {
      int id;
      struct pathtree *children;
    } node;
  } u;
  struct pathtree *next;  /* next sibling */
};

/* a dataflow tree originating at a given source */
struct srctree {
  int src_comp;
  struct pathtree *dsts;
};

/* For recursively freeing all of the memory associated with a
   path tree */
static void free_path(struct pathtree *p);
static void free_path(struct pathtree *p) {
  struct pathtree *q;
  while (p != NULL) {
    q = p->next;
    if (p->type == NODE)
      free_path(p->u.node.children);
    free (p);
    p = q;
  }
}

/* For converting a path to a path-tree */
static struct pathtree *path_to_tree(int *path, int len, int comp);
static struct pathtree *path_to_tree(int *path, int len, int comp) {
  struct pathtree *p;
  p = (struct pathtree *)malloc(sizeof(struct pathtree));
  p->next = NULL;
  if (len == 0) {
    p->type = LEAF;
    p->u.dst_comp = comp;
  }
  else {
    p->type = NODE;
    p->u.node.id = *path;
    p->u.node.children = path_to_tree(path+1,len-1,comp);
  }
  return p;
}

/* for merging a path into a path tree */
static struct pathtree *merge_path(struct pathtree *trees,
				   int *path, int len, int comp)
{
  struct pathtree *p;
  if (len != 0) {
    int finished = 0;
    p = trees;
    while (p != NULL) {
      if ((p->type == NODE) && (p->u.node.id == *path)) {
	p->u.node.children = 
	  merge_path(p->u.node.children, path+1, len-1, comp);
	finished = 1;
	break;
      }
      else
	p = p->next;
    }
    if (finished)
      return trees;
    else {
      p = path_to_tree(path,len,comp);
      p->next = trees;
      return p;
    }
  }
  else {
    p = path_to_tree(NULL,0,comp);
    p->next = trees;
    return p;
  }
}

/* prints a path tree, for debugging purposes */
static void print_pathtree(struct pathtree *p);
static void print_pathtree(struct pathtree *p) {
  while (p != NULL) {
    if (p->type == NODE) {
      printf("(%d, [ ",p->u.node.id);
      print_pathtree(p->u.node.children);
      printf(" ])");
    }
    else {
      printf("%s",computations[p->u.dst_comp].name);
    }
    p = p->next;
    if (p != NULL)
      printf(", ");
  }  
}
			   

/* computes two matrices performing the all-pairs shortest path
   computation, where "shortest path" means maximum bottleneck
   bandwidth.  The results are returned in two allocated
   arrays (which the caller must free).  The first, bestpaths(x,y), is
   the best matrix (i.e. the bottleneck bandwdith) for the best
   path from x to y, and the second, predecessors(x,y), is the
   immediately preceding node on the best bath from x to y. */

static void compute_all_paths(double **bestpaths, int **predecessors) {
  int i, j, k;
  /* allocate the output arrays */
  int *pred = (int *) malloc(nnodes * nnodes * sizeof(int));
  double *summaries = (double *)calloc(nnodes * nnodes, sizeof(double));

  /* initialize the output arrays */
  for (i=0; i<nnodes; i++) {
    for (j=0; j<nnodes; j++) {
      GET2D(pred,nnodes,i,j) = -1;
      if (i != j) {
	int net = NETWORK(i+1,j+1);
	if (net != 0) {
	  double capacity = networks[net-1].capacity - networks[net-1].load;
	  if (capacity <= 0.0) capacity = 1.0;
/* 	  double capacity = networks[net-1].capacity; */
	  GET2D(summaries,nnodes,i,j) = capacity;
	  if (capacity != 0.0)
	    GET2D(pred,nnodes,i,j) = i+1;
	}
      }
    }
  }
  
  /* calculate the max bottleneck b/w paths */
  for (k=0; k<nnodes; k++) {
    for (i=0; i<nnodes; i++) {
      for (j=0; j<nnodes; j++) {
	double old = GET2D(summaries,nnodes,i,j);
	double oldik = GET2D(summaries,nnodes,i,k);
	double oldkj = GET2D(summaries,nnodes,k,j);
	double alt = MIN(oldik,oldkj);
	if (old < alt) {
	  GET2D(summaries,nnodes,i,j) = alt;
	  GET2D(pred,nnodes,i,j) = GET2D(pred,nnodes,k,j);
	}
      }
    }
  }

  /* return the results */
  *bestpaths = summaries;
  *predecessors = pred;
}

/* returns the path from src to dst as computed by the predecessor
   matrix.  Returns it (by side-effect) in the variable path.  Returns
   also the length of that path.  Note that the path includes both
   the source and the destination (not just the nodes in between) */
int get_path(int *predecessors, int src, int dst, int *path, int sz) {
  int i,j;
  int len;

  /* calculate the path length */
  i = dst;
  len = 1;
  while (i != src) {
    i = GET2D(predecessors,nnodes,src-1,i-1);
    len++;
  }
  if (len > sz) {
    fprintf(stderr,"get_path failed: path argument too small (%d < %d)\n",
	    sz, len);
    exit(1);
  }

  /* fill in the path */
  i = dst;
  j = len-1;
  path[j] = i;
  while (i != src) {
    j--;
    i = GET2D(predecessors,nnodes,src-1,i-1);
    path[j] = i;
  }

  return len;
}

/******************/
/* COST FUNCTIONS */
/******************/

double send_cost(map_t args, int *frame_size){
	char *size;
	void **p;

	*frame_size = 0;
	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in send");
		return 100000;
	}
	return 10000 + atoi(size) * 10;
}

double recv_cost(map_t args, int *frame_size){
	char *size;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in recv");
		*frame_size = 10000;
	}
	else {
		*frame_size = atoi(size);
	}
	return *frame_size * 10;
}

double mpegsource_cost(map_t args, int *frame_size){
	char *width, *height;
	int w, h;
	void **p;

	p = tst_find(&args, "width", 0);
	if (p == 0 || (width = *p) == 0) {
		err_warning("no width specified in mpegsource");
		w = 100;
	}
	else {
		w = atoi(width);
	}
	p = tst_find(&args, "height", 0);
	if (p == 0 || (height = *p) == 0) {
		err_warning("no height specified in mpegsource");
		h = 100;
	}
	else {
		h = atoi(height);
	}
	*frame_size = w * h / 8 + 100;

	return w * h * 3 + 1000;
}

double mpegclip_cost(map_t args, int *frame_size){
	char *width, *height;
	int w, h;
	void **p;

	p = tst_find(&args, "ow", 0);
	if (p == 0 || (width = *p) == 0) {
		err_warning("no width specified in mpegsource");
		w = 100;
	}
	else {
		w = atoi(width);
	}
	p = tst_find(&args, "oh", 0);
	if (p == 0 || (height = *p) == 0) {
		err_warning("no height specified in mpegsource");
		h = 100;
	}
	else {
		h = atoi(height);
	}
	*frame_size = w * h / 8 + 100;

	return w * h * 5 + 300;
}

double mpegdisplay_cost(map_t args, int *frame_size){
	char *width, *height;
	int w, h;
	void **p;

	p = tst_find(&args, "width", 0);
	if (p == 0 || (width = *p) == 0) {
		err_warning("no width specified in mpegsource");
		w = 100;
	}
	else {
		w = atoi(width);
	}
	p = tst_find(&args, "height", 0);
	if (p == 0 || (height = *p) == 0) {
		err_warning("no height specified in mpegsource");
		h = 100;
	}
	else {
		h = atoi(height);
	}
	*frame_size = w * h / 8 + 100;

	return w * h * 10 + 500;
}

double Seq_add_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in Seq_add");
		s = 1000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

double Seq_MPEG_prio_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in Seq_MPEG_prio");
		s = 1000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

double Seq_restore_MPEG_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in Seq_restore_MPEG");
		s = 1000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

double Seq_MPEG_drop_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in Seq_MPEG_droP");
		s = 1000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

double Monitor_cost(map_t args, int *frame_size){
  *frame_size = 1000; // will send one monitoring message
  return 1e3;
}

double PacketpairSend_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in PacketpairSend");
		s = 3000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

double PacketpairRecv_cost(map_t args, int *frame_size){
	char *size;
	int s;
	void **p;

	p = tst_find(&args, "size", 0);
	if (p == 0 || (size = *p) == 0) {
		err_warning("no size specified in PacketpairRecv");
		s = 3000;
	}
	else {
		s = atoi(size);
	}
	*frame_size = s;

	return s * 10 + 500;
}

struct operation {
	char *name;
	double (*cost)(map_t args, int *frame_size);
};
map_t operations;
struct operation op_table[] = {
	{ "send", send_cost },
	{ "recv", recv_cost },
	{ "mpegsource", mpegsource_cost },
	{ "mpegclip", mpegclip_cost },
	{ "mpegdisplay", mpegdisplay_cost },
	{ "Seq_add", Seq_add_cost },
	{ "Seq_MPEG_drop", Seq_MPEG_drop_cost },
	{ "MPEG_drop", Seq_MPEG_drop_cost },
	{ "Seq_MPEG_prio", Seq_MPEG_prio_cost },
	{ "MPEG_prio", Seq_MPEG_prio_cost },
	{ "Seq_restore_MPEG", Seq_restore_MPEG_cost },
	{ "Monitor", Monitor_cost },
	{ "PacketpairSend", PacketpairSend_cost },
	{ "PacketpairRecv", PacketpairRecv_cost },
	{ 0, 0 }
};

void init_operations(void){
	void **p;
	int i;

	for (i = 0; op_table[i].name != 0; i++) {
		p = tst_find(&operations, op_table[i].name, 1);
		*p = &op_table[i];
	}
}

/* Return the cost of the given operation per frame.
 */
double cost_lookup(map_t op, int *output_size){
	void **p = tst_find(&op, "cmd", 0);
	char *cmd;
	struct operation *operation;

	if (p == 0 || (cmd = *p) == 0) {
		err_warning("no cmd specified in operation");
		*output_size = 1000;
		return 1e3;
	}
	p = tst_find(&operations, cmd, 0);
	if (p == 0 || (operation = *p) == 0) {
#ifdef debug
		err_warning("unknown operation '%s'", cmd);
#endif
		*output_size = 1000;
		return 1e3;
	}
	return (*operation->cost)(op, output_size);
}

/* Installs a monitor operation */
static int install_monitor(int index, int version, int src) {
  struct computation *comp;
  void **p;

  /* Put in the Monitor operation if so desired. */
  comp = &computations[index];
  comp->name = mem_print("mon_%s", nodes[src-1].name);
  comp->operation = 0;
  p = tst_find(&comp->operation, "cmd", 1);
  *p = mem_print("Monitor");
  p = tst_find(&comp->operation, "host", 1);
  *p = string_copy(parameters.schedsvr_host);
  p = tst_find(&comp->operation, "port", 1);
  *p = mem_print("%d", parameters.schedsvr_port);
  p = tst_find(&comp->operation, "uri", 1);
  *p = mem_print("report.cgi");
  p = tst_find(&comp->operation, "version", 1);
  *p = mem_print("%d", version);
  comp->assigned_loc = src;
  comp->location = src; /* pin it down here */
  comp->cost = cost_lookup(comp->operation, &comp->output_size);
  comp->input = mem_print("");
  comp->interval = 1;
  comp->enforce = 0;
  comp->type = COMP_MONITOR;
  comp->delay = 0;
  comp->in_links = 0;
  comp->ninputs = 0;

  return index+1;
}

void fill_in_ports(int for_real) {
  void **p;
  struct computation *comp;
  int i;

  if (for_real) {
    /* fill in the UDP ports for the send/recv computations */
    for (i=0; i<ncomps_needing_ports; i++) {
      comp = &computations[comps_needing_ports[i]];
      p = tst_find(&comp->operation, "port", 1);
      *p = mem_print("%d", parameters.nextport);
      comp = &computations[comps_needing_ports[i]+1];
      p = tst_find(&comp->operation, "port", 1);
      *p = mem_print("%d", parameters.nextport);
      parameters.nextport++;
      if (parameters.nextport >= HIGH_PORT)
	parameters.nextport = LOW_PORT;
    }
  }

  /* clear the state */
  ncomps_needing_ports = 0;
}

/* Called when we are about to insert a send/recv pair. */

static int install_opt_comp(int src_node, int *src_comp, int index) {
  struct computation *comp;

  /* Put in any optional computations specified by user */
  if (computations[*src_comp].insert != 0) {
    char *inserts = computations[*src_comp].insert;
    while (1) {
      int len, src;
      char *p;

      /* Find the next input */
      p = next_string(&inserts, &len);
      if (p == NULL)
	break;

      /* Find the given name in the list of optional computations.
       */
      for (src = 0; src < nspec_opt_comps; src++) {
	if (string_match(opt_computations[src].name, p, len)) {
	  break;
	}
      }
      assert(opt_computations[src].name != 0);

      /* Add this computation to the list and connect it in */
      memcpy(comp = &computations[index++], &opt_computations[src],
	     sizeof(struct computation));
      comp->assigned_loc = src_node;
      comp->input = string_copy(computations[*src_comp].name);
      comp->in_links = (int *) malloc(sizeof(int));
      comp->in_links[0] = *src_comp;
      comp->ninputs = 1;
      comp->name = mem_print("%s%d",computations[index-1].name, index-1);
      
      /* Iterate */
      *src_comp = index-1;      
    }
  }
  return index;
}

/* Installs a pair of send/recv components between src and dst. */
   
static int install_send_recv(int src_node, int src_comp, 
			     int dst_node, double interval, int output_size,
			     int index) {
  int link;
  void **p;
  struct computation *comp;
  struct network *net;
  int reverse = 0;

  /* Make a note of the send and recv computations we are
     about to insert that will need ports filled in when finishing. */
  assert (ncomps_needing_ports < MAX_COMPS);
  comps_needing_ports[ncomps_needing_ports++] = index;

  /* Put in the send operation.	 */
  link = NETWORK(src_node,dst_node);
  net = &networks[link-1];

  comp = &computations[index++];
/*   comp->name = mem_print("s%d_%s", */
/* 			 index-1,computations[src_comp].name); */
  comp->name = mem_print("s%d", index-1);
  comp->operation = 0;
  p = tst_find(&comp->operation, "cmd", 1);
  *p = mem_print("send");
  if (reverse) {
    p = tst_find(&comp->operation, "connection-action", 1);
    *p = string_copy("accept");
  }
  else {
    p = tst_find(&comp->operation, "host", 1);
    if (parameters.emulab_link) {
      // When running on emulab, we have to choose the correct
      //   link address based on who we're sending to
      if (!strncmp(nodes[dst_node-1].host, net->nodes,
		   strlen(nodes[dst_node-1].name))) {
	*p = string_copy(net->emulab_hops[0]);
      }
      else {
	*p = string_copy(net->emulab_hops[1]);
      }	
    }
    else
      *p = string_copy(nodes[dst_node-1].host);
  }

  p = tst_find(&comp->operation, "size", 1);
  *p = mem_print("%d", output_size);
  p = tst_find(&comp->operation, "link", 1);
  *p = mem_print(net->name);
  comp->assigned_loc = src_node;
  comp->cost = cost_lookup(comp->operation, &comp->output_size);
  comp->input = string_copy(computations[src_comp].name);
  comp->interval = interval;
  comp->enforce = 0;
  comp->type = COMP_SEND;
  comp->delay = 0;
  comp->in_links = (int *) malloc(sizeof(int));
  comp->in_links[0] = src_comp;
  comp->ninputs = 1;

  /* Put in the receive operation. */
  comp = &computations[index];
/*   comp->name = mem_print("r%d_%s", */
/* 			 index,computations[index-1].name); */
  comp->name = mem_print("r%d",index);
  comp->operation = 0;
  p = tst_find(&comp->operation, "cmd", 1);
  *p = mem_print("recv");
  if (reverse) {
    p = tst_find(&comp->operation, "connection-action", 1);
    *p = string_copy("connect");
    p = tst_find(&comp->operation, "host", 1);
    if (parameters.emulab_link) {
      // When running on emulab, we have to pick the right address
      if (!strncmp(nodes[dst_node-1].host, net->nodes,
		   strlen(nodes[dst_node-1].name))) {
	*p = string_copy(net->emulab_hops[0]);
      }
      else {
	*p = string_copy(net->emulab_hops[1]);
      }	
    }
    else
      *p = string_copy(nodes[src_node-1].host);
  }
  p = tst_find(&comp->operation, "size", 1);
  *p = mem_print("%d", output_size);
  p = tst_find(&comp->operation, "link", 1);
  *p = mem_print(net->name);
  comp->assigned_loc = dst_node;
  comp->cost = cost_lookup(comp->operation, &comp->output_size);
  comp->input = "";
  comp->interval = interval;
  comp->enforce = 0;
  comp->type = COMP_RECV;
  comp->delay = 0;
  comp->in_links = (int *) malloc(sizeof(int));
  comp->in_links[0] = index - 1;
  comp->ninputs = 1;

  return index+1;
}

#ifdef DO_PACKETPAIR
static int install_packetpair(int src_node, int dst_node, 
			      double interval, int output_size,
			      int index) {
  
  void **p;
  struct computation *comp;
  int send_comp;

  /* Put in the packetpair sender. */
  send_comp = index++;
  comp = &computations[send_comp];
  comp->name = mem_print("packetpair_src_%s",nodes[src_node-1].name);
  comp->operation = 0;
  p = tst_find(&comp->operation, "cmd", 1);
  *p = mem_print("PacketpairSend");
  p = tst_find(&comp->operation, "size", 1);
  *p = mem_print("%d", output_size * 2);
  comp->assigned_loc = src_node;
  comp->cost = cost_lookup(comp->operation, &comp->output_size);
  comp->input = mem_print("");
  comp->interval = interval;
  comp->enforce = 1; /* ensure it gets scheduled then */
  comp->type = COMP_USER;
  comp->delay = 0;
  comp->in_links = 0;
  comp->ninputs = 0;

  /* Put in the packetpair receive operation. */
  comp = &computations[index++];
  comp->name = mem_print("packetpair_dst_%s",nodes[dst_node-1].name);
  comp->operation = 0;
  p = tst_find(&comp->operation, "cmd", 1);
  *p = mem_print("PacketpairRecv");
  p = tst_find(&comp->operation, "size", 1);
  *p = mem_print("%d", output_size * 2);
  comp->assigned_loc = dst_node;
  comp->cost = cost_lookup(comp->operation, &comp->output_size);
  comp->interval = interval;
  comp->enforce = 0;
  comp->type = COMP_USER;
  comp->delay = 0;

  /* Put in the send and receive operations */
  index = install_send_recv(src_node, send_comp, dst_node,
			    interval, output_size * 2,
			    index);

  /* Tie in the packetpair receive operation to the recv comp. */
  comp->input = string_copy(computations[index-1].name);
  comp->in_links = (int *) malloc(sizeof(int));
  comp->in_links[0] = index - 1;
  comp->ninputs = 1;

  return index;
}
#endif

static int install_all_monitors(int index) {
  int i;
  for (i=1; i<=nnodes; i++)
    ntotal_comps = install_monitor(ntotal_comps, version, i);
  return ntotal_comps;
}

#ifdef notdef
/* Flows the component execution interval backwards until a non
   send/recv computation is discovered. */
static void flow_interval_backwards(int dst_comp, double interval);

static void flow_interval_backwards(int dst_comp, double interval) {
  int i;
  for (i=0; i<computations[dst_comp].ninputs; i++) {
    int src_comp = computations[dst_comp].in_links[i];
    if (computations[src_comp].type == COMP_SEND ||
	computations[src_comp].type == COMP_RECV) {
      /* make sure that the downstream components have the same
	 interval; if not, pick the smaller one (meaning more
	 computation). */
      if (computations[src_comp].interval != 0)
	computations[src_comp].interval = 
	  MIN(interval, computations[src_comp].interval);
      else
	computations[src_comp].interval = interval;
      flow_interval_backwards(src_comp, interval);
    }
  }
}
#endif
      
/* Connect together the src computation with the computations
   described by the given path tree. */
static void make_path(struct pathtree *p, int src_comp, 
		      double interval, int output_size);

static void make_path(struct pathtree *p, int src_comp, 
		      double interval, int output_size)
{
  int src = computations[src_comp].assigned_loc;
  while (p != NULL) {
    /* For nodes, we might have to insert send/recv's, or do splits */
    if (p->type == NODE) {
      int dst = p->u.node.id;
      int old_src_comp = src_comp;
      int old_src = src;
      /* If the computations are on two different nodes, insert
	 a send/recv pair */
      if (src != dst) {
	ntotal_comps = install_opt_comp(src, &src_comp, ntotal_comps);
	ntotal_comps =
	  install_send_recv(src, src_comp, dst,
			    interval, output_size,
			    ntotal_comps);

	/* reset the source and destination, and recurse */
	src = dst;
	src_comp = ntotal_comps - 1;
	computations[src_comp].insert = computations[old_src_comp].insert;
      }
      /* Connect to the children */
      make_path(p->u.node.children, src_comp,
		interval, output_size);

      /* Restore the source for the adjacent children */
      src_comp = old_src_comp;
      src = old_src;
    }

    /* For leaves, we can connect up the computations directly */
    else {
      int dst_comp = p->u.dst_comp;
      int ninputs = computations[dst_comp].ninputs;
      if (ninputs == 0)
	computations[dst_comp].in_links = (int *) malloc(sizeof(int));
      else
	computations[dst_comp].in_links = 
	  (int *) realloc((char *) computations[dst_comp].in_links,
			  (ninputs + 1) * sizeof(int));
      computations[dst_comp].in_links[ninputs] = src_comp;
      computations[dst_comp].ninputs++;
      /* now flow the interval backwards */
      /* XXX This really appears to be a bug in how we do things, so I'm
	 commenting it out. */
/*        flow_interval_backwards(dst_comp, computations[dst_comp].interval); */
    }
    p = p->next;
  }  
}

void create_paths(struct srctree *trees, int numtrees) {
  int i;
  
  /* for each source node in the path, insert the necessary
     connection to its downstream children */
  for (i=0; i<numtrees; i++) {
    make_path(trees[i].dsts, trees[i].src_comp, 
	      computations[trees[i].src_comp].interval,		  
	      computations[trees[i].src_comp].output_size);
  }
}

/* Set up the all-pairs shortest paths for the current network. 
   Returns the number of computations */
static NEW_TIMER("recompute_all_paths",t1);
int recompute_all_paths(int cc, int index, int **predecessors) {
  double *summaries;
  int i, n;
  START_TIME(t1);

  /* Clear any old inputs from the user comps */
  assert(index >= nspec_comps);
  ntotal_comps = index;
  for (i=0, n=0;i<nspec_comps;i++) {
    if (connected_comps[i] >= cc) {
      n++;
      if (computations[i].in_links != NULL) {
	free(computations[i].in_links);
	computations[i].in_links = NULL;
      }
      computations[i].ninputs = 0;
    }
  }

  /* Now calculate the all-pairs shortest path matrix for
     the current state of the network.  Need to set up the
     load on the network to include prior configs */
  network_score();
  compute_all_paths(&summaries,predecessors);
  free(summaries);

  END_TIME(t1);

  /* Return the number of computations we're considering. */
  return n;
}

/* In order to make sure that the computations on different nodes are
 * connected, we need to add send and recv computations to the mix.
 * At the moment, we ignore the possibility of multiple available
 * paths.
 *
 * cc is the connected component we are currently considering.
 * n is the number of computations to consider in this component.
 */
static NEW_TIMER("insert_necessary_connections",t2);
int insert_necessary_connections(int cc, int index, int *predecessors){
  int i, dst, n;
  struct srctree *trees = NULL;
  int numtrees = 0;
  START_TIME(t2);

  /* set up the network to properly calculate the paths */
  ntotal_comps = index;
  for (i=0, n=0;i<nspec_comps;i++) {
    if (connected_comps[i] >= cc) {
      n++;
      if (computations[i].in_links != NULL) {
	free(computations[i].in_links);
	computations[i].in_links = NULL;
      }
      computations[i].ninputs = 0;
    }
  }

  /* Make space for the tree-paths we're about to set up */
  trees = malloc(n * sizeof(struct srctree));

  /* Now find the minimum paths between adjacent computations */
  for (dst = 0; dst < nspec_comps; dst++) {
    char *inputs = computations[dst].input;

    /* ignore user comps not from this connected component */
    if (connected_comps[dst] != cc) continue;

    /* Find all the minimum paths to this destination */
    while (1) {
      int len, src;
      char *p;
      int best_path[MAX_PATH], best_len;

      /* Find the next input */
      p = next_string(&inputs, &len);
      if (p == NULL)
	break;

      /* Find the given name in the list of computations.
       */
      for (src = 0; src < nspec_comps; src++) {
	if (string_match(computations[src].name, p, len)) {
	  break;
	}
      }
      assert(computations[src].name != 0);

      /* Now create the minimum path */
      best_len = get_path(predecessors,
			  computations[src].assigned_loc,
			  computations[dst].assigned_loc,
			  best_path, MAX_PATH);

      /* Add it to the path tree */
      for (i=0; i<numtrees; i++) {
	if (!strcmp(computations[src].name,
		    computations[trees[i].src_comp].name)) {
	  trees[i].dsts = 
	    merge_path(trees[i].dsts, best_path, best_len, dst);
	  break;
	}
      }
      if (i==numtrees) {
	trees[numtrees].src_comp = src;
	trees[numtrees].dsts = 
	  merge_path(NULL, best_path, best_len, dst);
	numtrees++;
      }
    }
  }

  /* print them out for debugging purposes */
#ifdef debug
  printf("insert_necessary results:\n");
  for (i=0; i<numtrees; i++) {
    printf("src=%s:\n  ",computations[trees[i].src_comp].name);
    print_pathtree(trees[i].dsts);
    printf("\n");
  }
#endif

  /* now insert the computations based on the path tree */
  create_paths(trees, numtrees);

  /* free up temporary memory */
  for (i=0; i<numtrees; i++)
    free_path(trees[i].dsts);
  free(trees);

  END_TIME(t2);

  return ntotal_comps;
}

/* The aggregate score is the harmonic mean of the leftover capacity;
   this biases smaller values.  If the schedule can't be met, it will
   return the most overprovisioned score. */
static double aggregate_score(double *load, double *capacity, int num) {
  double min = 1.0 /* must be > 0.0 */, score = 0.0, tmp;
  int i;

  for (i = 0; i < num; i++) {
    if (capacity[i] != 0)
      tmp = (capacity[i] - load[i]) / capacity[i];
    else
      tmp = -1;
    if (tmp > 0) {
      tmp = 1 / (num*tmp);
      score += tmp;
    }
    else if (min > tmp) {
      min = tmp;
    }
  }
  score = 1.0 / score;
  if (min <= 0.0)
    return min;
  else
    return score;
}

/* Calculate the node score, which is the aggregation (see above) of
 * the individual node scores.  An individual node score is the
 * difference between the capacity and the load, divided by the
 * capacity.  As a side effect, this function also calculates the
 * loads on each node, which is used by comp_score().  */
double node_score(void){
  int node, comp;
  double min;
  double *load, *capacity;

  /* Initialize everything first.
   */
  load = malloc(sizeof(double) * nnodes);
  capacity = malloc(sizeof(double) * nnodes);
  for (node = 0; node < nnodes; node++) {
    load[node] = nodes[node].load = nodes[node].max_step = 0;
    capacity[node] = nodes[node].capacity;
  }

  /* For each computation, add the cost per operation / interval to the
   * load.  Also, add the cost to the max_step in order to calculate
   * max delay.
   */
  for (comp = 0; comp < ntotal_comps; comp++) {
#ifdef debug
    printf("%s on %d: %lf %lf (%lf)\n",
	   computations[comp].name,
	   computations[comp].assigned_loc,
	   computations[comp].cost,
	   computations[comp].interval,
	   computations[comp].cost / computations[comp].interval);
#endif
    node = computations[comp].assigned_loc - 1;
    load[node] = 
      (nodes[node].load +=
       computations[comp].cost / computations[comp].interval);
    nodes[node].max_step += computations[comp].cost;
  }

  /* Finally, calculate the minimum left-over capacity.
   */
  min = aggregate_score(load, capacity, nnodes);
  free(load);
  free(capacity);
  return min;
}

/* Calculate the network score, which is the aggregation (see above)
 * of the individual network scores.  An individual network score is
 * the difference between the capacity and the load, divided by the
 * network's capacity.  As a side effect, this function also calculates
 * the loads on each network, which is used by comp_score();
 *
 * Will not count computations that do not have an in-links.
*/
double network_score(){
  int net, comp, src;
  double min, *load, *capacity;

  /* Initialize everything first.
   */
  for (net = 0; networks[net].name != 0; net++) {
    networks[net].load = networks[net].max_step = 0;
  }
  load = malloc(sizeof(double) * nlinks);
  capacity = malloc(sizeof(double) * nlinks);
  for (net = 0; net < nlinks; net++) {
    load[net] = 0;
    capacity[net] = networks[net].capacity;
  }

  /* For each recv computation, add the frame size / interval to the
   * network load.  Also, add the size to the max_step in order to
   * calculate max delay.
   */
  for (comp = 0; comp < ntotal_comps; comp++) {
    if (computations[comp].type == COMP_RECV &&
	computations[comp].in_links != 0) {
      src = computations[comp].in_links[0];
      net = NETWORK(computations[src].assigned_loc,
		    computations[comp].assigned_loc) - 1;
      networks[net].load += computations[comp].output_size /
	computations[comp].interval;
      load[net] = networks[net].load;
      networks[net].max_step += computations[comp].output_size;
    }
  }

  /* Finally, calculate the minimum left-over capacity.
   */
  min = aggregate_score(load, capacity, nlinks);
  free(load);
  free(capacity);
  return min;
}

/* Determine the delay for the given computations.
 */
void determine_delay(int comp){
  int src, net, node, i;
  double max;

  /* If determined already, we're done.
   */
  if (computations[comp].delay > 0) {
    return;
  }
  
  /* Scan through the list of input computations.
   */
  for (max = 0, i = 0; i < computations[comp].ninputs; i++) {
    src = computations[comp].in_links[i];
    
    /* Recursively determine delay to input computation.
     */
    determine_delay(src);
    
    /* Keep track of maximum.
     */
    if (computations[src].delay > max) {
      max = computations[src].delay;
    }
  }
  
  /* Add the maximum compute cost per operation..
   */
  node = computations[comp].assigned_loc - 1;
  computations[comp].delay = max + nodes[node].max_step / nodes[node].capacity;

  /* Also add the maximum network delay in case of a RECV operation.
   */
  if (computations[comp].type == COMP_RECV) {
    src = computations[comp].in_links[0];
    net = NETWORK(computations[src].assigned_loc, computations[comp].assigned_loc) - 1;
    computations[comp].delay += 
      (networks[net].latency +
       networks[net].capacity > 0 ? 
       (networks[net].max_step / networks[net].capacity) : -1);
  }
}

/* For each computation, figure out its maximum delay to the sources.
 * Aggregate over all computations that have a specified maximum delay
 * (see above the aggregation formula). */
double comp_score(void){
  int comp;
  double min, *delay, *max_delay, num = 0;

  /* Reset all delays.
   */
  for (comp = 0; comp < nspec_comps; comp++) {
    computations[comp].delay = 0;
  }
  /* Calculate the delays.
   */
  for (comp = 0; comp < nspec_comps; comp++) {
    determine_delay(comp);
    if (computations[comp].max_delay > 0)
      num++;
  }
  /* Aggregate the result.
   */
  if (num > 0) {
    int i = 0;
    delay = malloc(sizeof(double) * num);
    max_delay = malloc(sizeof(double) * num);
    for (comp = 0; comp < nspec_comps; comp++) {
      if (computations[comp].max_delay > 0) {
	max_delay[i] = computations[comp].max_delay;
	delay[i] = computations[comp].delay;
	i++;
      }
    }
    assert(i==num);
    min = aggregate_score(delay, max_delay, num);
    free(delay);
    free(max_delay);
  }
  else
    min = 1;

  return min;
}

/* The score has three components
 *
 * - for each compute node, score 0 at capacity, < 0 when capacity
 *    exceeded, > 0 when capacity left over.  1 means completely
 *    unloaded, so the score cannot * be higher than 1.  
 * - same for network links
 * - for each computation, score 0 when end-to-end
 *    delays exactly met, < 0 when delay exceeded, > 0 when better
 *    delay achieved.  Again, however, 1 is max.
 *
 * We take the minimum of the three scores as the total score.
 *
 * 
 */
double calculate_score(){
  double score1, score2, score3, score;

  /* Calculate the individual scores..
   */
  score1 = node_score();
  score2 = network_score();
  score3 = comp_score();
  /* Finally, do the actual score computation by taking the minimum..
   */
  score = score1 < score2 ? score1 : score2;
  score = score < score3 ? score : score3;
#ifdef debug
  printf("calculate score: node:%lf network:%lf comp:%lf; total=%lf\n",
	 score1, score2, score3, score);
#endif
  return score;
}

/* Find a working configuration.  Place computations on nodes so that
 * node's capacities and link's capacities are not exceeded.  The next
 * concern would be too minimize the maximum load on these components.
 *
 * The strategy is as follows: we 'score' a configuration using the
 * concerns listed above.  Then we try, for each computation, whether
 * moving them to a new node improves the score, and if so, move it.
 * We do this until we can no longer find such a computation.
 */
double find_good_config(int for_real){
  int comp, node, current_loc, best_loc, improvement;
  double best_score, score, current_score;
  int comp_index = nspec_comps;
  int cc;
  int *predecessors = NULL;

  /* For each connected component, get the best schedule, adding
     the schedule created by the previous CC.
  */
  for (cc = 0; cc<nconnected_comps; cc++) {
    int saved_comps;

    /* Calculate the all-pairs shortest paths for the
       current state of the network. */
    if (predecessors != NULL)
      free(predecessors);
    recompute_all_paths(cc, comp_index, &predecessors);

    /* Insert the necessary connections and monitors */
    saved_comps = insert_necessary_connections(cc,comp_index,predecessors);
    install_all_monitors(ntotal_comps);
    current_score = calculate_score();
    ntotal_comps = saved_comps;

    do {
      improvement = 0;
      for (comp = 0; comp < nspec_comps; comp++) {

	/* Only consider moving computations from the current CC
	 * or those whose location is not fixed.
	 */
	if (connected_comps[comp] != cc ||
	    computations[comp].location != 0) {
	  continue;
	}

	/* Save current location.
	 */
	current_loc = computations[comp].assigned_loc;
	
	/* Try to find the best location.
	 */
	best_score = current_score;
	best_loc = current_loc;
	for (node = 0; nodes[node].name != 0; node++) {

	  /* If this is the original configuration, skip.  We already know.
	   */
	  if (node + 1 == current_loc) {
	    continue;
	  }
	  computations[comp].assigned_loc = node + 1;
	
	  /* Calculate a new score.
	   */
	  saved_comps =
	    insert_necessary_connections(cc,comp_index,predecessors);
	  install_all_monitors(ntotal_comps);
	  score = calculate_score();
	  ntotal_comps = saved_comps;
	  if (score > best_score) {
	    best_score = score;
	    best_loc = node + 1;
	  }
	}
	
	/* If a better location is available, update the assigned location.
	 * Otherwise restore to original state.
	 */
	if (best_score > current_score) {
	  computations[comp].assigned_loc = best_loc;
	  current_score = best_score;
	  improvement = 1;
	}
	else {
	  computations[comp].assigned_loc = current_loc;
	}
      }
    } while (improvement);

    /* No point in continuing if we have a negative score */
    if (current_score < 0.0) {
#ifdef debug
      printf("failed to get non-negative score for CC's <= %d; aborting\n",
	     cc);
#endif
      break;
    }

    /* Insert the connections for the best configuration.
     */
    insert_necessary_connections(cc,comp_index,predecessors);
    fill_in_ports(for_real);

    comp_index = ntotal_comps;

#ifdef debug
    { int i;
    printf("Settled on configuration for CC:\n");
    for (i=0; i<comp_index; i++) {
      if (i < nspec_comps) {
	if (connected_comps[i] == cc)
	  printf("comp %s at location %s\n",
		  computations[i].name,
		  nodes[computations[i].assigned_loc-1].name );
      }
      else 
	printf("comp %s at location %s\n",
		computations[i].name,
		nodes[computations[i].assigned_loc-1].name );
    }
    }
#endif
  }

  /* Finally, insert the monitors when finished.
   */
  install_all_monitors(ntotal_comps);
  return calculate_score();
}

char *xml_escape(char *s){
	struct print_channel *pc = mc_open();

	for (; *s != 0; s++)
		switch (*s) {
		case '<': pc_print(pc, "&lt;"); break;
		case '>': pc_print(pc, "&gt;"); break;
		case '&': pc_print(pc, "&amp;"); break;
		case '"': pc_print(pc, "&quot;"); break;
		/* case '\'': pc_print(pc, "&apos;"); break; */
		case '\r': pc_print(pc, "&#xD;"); break;
		case '\n': case '\t':
			pc_putchar(pc, *s);
			break;
		default:
			if ((*s & 0xFF) < 0x20) {
				pc_print(pc, "&#x%x;", *s & 0xFF);
			}
			else {
				pc_putchar(pc, *s);
			}
		}
	return mc_close(pc);
}

/* Print the current configuration.
 */
char *mmsched_dump(int *len){
	struct print_channel *pc = mc_open();
	int comp, node, net;
	char *op;

	pc_print(pc, "<html><body>\n");

	pc_print(pc, "<h2>Scores:</h2>\n");
	pc_print(pc, "<table border=\"1\"><tr><th>Which</th><th>Score</th></tr>");
	pc_print(pc, "<tr><td>Compute Nodes</td><td>%f</td></tr>\n", node_score());
	pc_print(pc, "<tr><td>Network Links</td><td>%f</td></tr>\n", network_score());
	pc_print(pc, "<tr><td>Computations</td><td>%f</td></tr>\n", comp_score());
	pc_print(pc, "</table>");

	pc_print(pc, "<h2>Computations:</h2>\n");
	pc_print(pc, "<table border=\"1\">");
	pc_print(pc, "<tr><th>Name</th><th>Location</th><th>Operation</th><th>Cost</th><th>Interval</th><th>Delay</th><th>Inputs</th><th>Output</th></tr>\n");
	for (comp = 0; comp < ntotal_comps; comp++) {
		op = tst_unparse(computations[comp].operation);
		pc_print(pc, "<tr><td>%s</td><td>%s</td><td>%s</td><td>%f</td><td>%f</td><td>%f</td><td>%s</td><td>%d</td></tr>\n",
			computations[comp].name,
			nodes[computations[comp].assigned_loc - 1].name, op,
			computations[comp].cost, computations[comp].interval,
			computations[comp].delay, computations[comp].input,
			computations[comp].output_size);
		free(op);
	}
	pc_print(pc, "</table>\n");

	pc_print(pc, "<h2>Nodes:</h2>\n");
	pc_print(pc, "<table border=\"1\">\n");
	pc_print(pc, "<tr><th>Name</th><th>Capacity</th><th>Load</th><th>Max Step</th></tr>\n");
	for (node = 0; node < nnodes; node++) {
		pc_print(pc, "<tr><td>%s</td><td>%f</td><td>%f</td><td>%f</td></tr>\n",
			nodes[node].name, nodes[node].capacity,
			nodes[node].load / nodes[node].capacity,
			nodes[node].max_step / nodes[node].capacity);
	}
	pc_print(pc, "</table>\n");

	pc_print(pc, "<h2>Networks:</h2>\n");
	pc_print(pc, "<table border=\"1\">\n");
	pc_print(pc, "<tr><th>Name</th><th>Latency</th><th>Capacity</th><th>Load</th><th>Max Step</th><th>Members</th></tr>\n");
	for (net = 0; networks[net].name != 0; net++) {
		pc_print(pc,
			 "<tr><td>%s</td><td>%f</td><td>%f</td><td>%f (%f)</td><td>%f</td><td>%s</td></tr>\n",
			networks[net].name, networks[net].latency,
			networks[net].capacity,
			networks[net].load,
			networks[net].load / networks[net].capacity,
			networks[net].max_step / networks[net].capacity,
			networks[net].nodes);
	}
	pc_print(pc, "</table>\n");
	pc_print(pc, "</body></html>\n");
	*len = mc_size(pc);
	return mc_close(pc);
}

int aa_visit(void *env, char *attr, void **v){
	char *val = xml_escape(*v);

	pc_print(env, " %s=\"%s\"", attr, val);
	free(val);
	return 1;
}

void add_assignment(struct print_channel *pc, int node){
  int comp, i, input;

  pc_print(pc, "<computations host=\"%s\" version=\"%d\">\n",
	   nodes[node].name, version);
  for (comp = 0; comp < ntotal_comps; comp++) {
    if (computations[comp].assigned_loc - 1 != node) {
      continue;
    }
    pc_print(pc, "  <comp name=\"%s\">\n", computations[comp].name);
    pc_print(pc, "    <op");
    tst_travel(&computations[comp].operation, aa_visit, pc);
    pc_print(pc, "/>\n");
    if (computations[comp].enforce)
      pc_print(pc,"    <interval enforce=\"true\">%f</interval>\n",
	       computations[comp].interval);
    else
      pc_print(pc,"    <interval>%f</interval>\n",
	       computations[comp].interval);
    if (computations[comp].type != COMP_RECV &&
	computations[comp].ninputs > 0) {
      pc_print(pc, "    <inputs>\n");
      for (i = 0; i < computations[comp].ninputs; i++) {
	input = computations[comp].in_links[i];
	pc_print(pc, "      <input name=\"%s\"/>\n",
		 computations[input].name);
      }
      pc_print(pc, "    </inputs>\n");
    }
    pc_print(pc, "  </comp>\n");
  }
  pc_print(pc, "</computations>\n");
}

#ifdef notdef
void add_assignment_graph(struct print_channel *pc, int node){
  int comp, i, input, n;
  graph g;
  int *node_map;

  /* count up the computations for this node
   */
  for (n = 0, comp = 0; comp < ntotal_comps; comp++) {
    if (computations[comp].assigned_loc - 1 != node) {
      continue;
    }
    else n++;
  }
  
  /* create a graph.
   */
  g = new_graph(n,nodes[node].name);
  node_map = malloc(ntotal_comps * sizeof(int));
  for (comp = 0, i = 0; comp < ntotal_comps && i < n; comp++) {
    if (computations[comp].assigned_loc - 1 != node) {
      continue;
    }
    add_node(g, computations[comp].name, i);
    node_map[comp] = i++;
  }
  for (comp = 0; comp < ntotal_comps; comp++) {
    if (computations[comp].assigned_loc - 1 != node) {
      continue;
    }
    for (i=0; i < computations[comp].ninputs; i++) {
      input = computations[comp].in_links[i];
      if (computations[input].assigned_loc - 1 == node)
	add_edge(g, node_map[input], node_map[comp]);
    }
  }
  free(node_map);

  /* print the graph.
   */
  print_adj_graph(g,pc);

  /* clean up.
   */
  free_graph(g);
}
#endif

void mmsched_send_assignments(void *env,
			      void (*dump_assign)(void *env,
						  char *addr,
						  char *assignment)){
  struct print_channel *pc;
  int node;
  char *assignment, *addr;

  for (node = 0; node < nnodes; node++) {
    pc = mc_open();
    add_assignment(pc, node);
    assignment = mc_close(pc);
    addr = mem_print("%s:%d", nodes[node].host, nodes[node].port);
    dump_assign(env, addr, assignment);
    free(addr);
    free(assignment);
  }
}

void graph_assignments(struct print_channel *pc){
  int comp, i, input;
  graph g;

  /* create a graph.
   */
  g = new_graph(ntotal_comps,"Schedule");
  for (comp = 0; comp < ntotal_comps; comp++) {
    add_node(g, computations[comp].name,
	     nodes[computations[comp].assigned_loc-1].name, comp);
    for (i=0; i < computations[comp].ninputs; i++) {
      input = computations[comp].in_links[i];
      add_edge(g, input, comp);
    }
  }

  /* print the graph.
   */
  print_dot_graph(g,pc);

  /* clean up.
   */
  free_graph(g);
}


/* Return a document with the assignments to each node.  If do_graph
   is false, it is in XML, else it is ASCII graphics.
 */
char *mmsched_assignments(int *len, int do_graph){
  struct print_channel *pc = mc_open();
  int node;

  if (do_graph) {
    graph_assignments(pc);
  }
  else {
    pc_print(pc, "<assignments>\n");
    for (node = 0; node < nnodes; node++) {
      add_assignment(pc, node);
    }
    pc_print(pc, "</assignments>\n");
  }
  *len = mc_size(pc);
  return mc_close(pc);
}

/****************************************************************************************/

/* returns 1 iff the two utility vectors are equal, and 0 otherwise */
int utility_eq(struct user_utility *u1, int u1sz,
	       struct user_utility *u2, int u2sz) {
  if (u1sz == u2sz) {
    int i;
    for (i=0; i<u1sz; i++) {
      /* assumes they are in the same order */
      assert(u1[i].u == u2[i].u);
      if (u1[i].utility != u2[i].utility)
	return 0;
    }
    return 1;
  }
  else
    return 0;
}

int util_init_visit(void *env, char *user, void **v){
  struct user_utility *users = * (struct user_utility **) env;
  struct user *u = *v;

  /* Initialize this element */
  users->u = u;
  users->utility = 0.0;

  /* Move to the next element in the array */
  *(struct user_utility **)env = users + 1;

  return 1;
}

struct user_utility *init_utility() {
  struct user_utility *x, *start;

  start = x = malloc(num_users * sizeof(struct user_utility));
  tst_travel(&users, util_init_visit, &x);

  return start;
}

int add_comp_visit(void *env, char *name, void **v) {
  struct computation *comp = *v;
  int i;

  /* only add the computation to the list if it's not
     already there, from another user */
  for (i=0; i<nspec_comps; i++) {
    if (comp_equals(comp,&computations[i])) {
      /* They are structurally equivalent.  If this is a mobile
	 computation, we need to make sure that all intermediate
	 computations are the same XXX. */
#ifdef debug
      printf("Found computation %s already, merging\n",
	     comp->name);
#endif
      return 1;
    }
  }
  computations[nspec_comps++] = *comp;
  return 1;
}

int ccl_visit(void *env, char *user, void **v){
  struct user_utility *user_utility = * (struct user_utility **) env;
  struct user *u = *v;
  int i, j;
  double utility = -1.0; 

  /* find the user and his relevant utility */
  for (i = 0; i< num_users; i++) {
    if (user_utility[i].u == u) {
      utility = user_utility[i].utility;
      break;
    }
  }
  assert(utility >= 0.0);

  /* get the computations */
  /* XXX assumes utilities are sorted in user spec?!? */
  for (i = 0; i < u->nalts; i++) {
    if (u->alts[i].utility <= utility) {
#ifdef debug
      printf("ccl_visit: using alt=%d:%lf for utility %lf\n",
	     i, u->alts[i].utility, utility);
#endif
      break;
    }
  }
  if (i == u->nalts) {
#ifdef debug
    int j;
    printf("ccl_visit: nalts=%d, none for utility %lf\n",
	   u->nalts, utility);
    for (j = 0; j < u->nalts; j++)
      printf("alt[%d].utility = %lf\n",j,u->alts[j].utility);
#endif
    return 1;
  }
  tst_travel(&u->alts[i].comps, add_comp_visit, 0);
  
  /* now get the optional insert computations */
  for (j = 0; j<u->alts[i].num_icomps; j++)
    memcpy(&opt_computations[nspec_opt_comps+j],u->alts[i].icomps[j],
	   sizeof(struct computation));
  nspec_opt_comps += u->alts[i].num_icomps;

  return 1;
}

/* create a graph of computations 0 to index-1 */
static graph create_comp_graph(int index, char *name) {
  int comp;
  graph g;

  /* create the graph */
  g = new_graph(index,name);
  for (comp = 0; comp < index; comp++) {
    char *inputs = computations[comp].input;

    /* add the node */
    add_node(g, computations[comp].name, NULL, comp);

    /* add all of the inputs as edges */
    while (1) {
      int len, src;
      char *p;

      /* Find the next input */
      p = next_string(&inputs, &len);
      if (p == NULL)
	break;

      /* Find the given name in the list of computations.
       */
      for (src = 0; src < index; src++) {
	if (string_match(computations[src].name, p, len)) {
	  break;
	}
      }
      assert(computations[src].name != 0);

      /* Add the edge.
       */
      add_edge(g, src, comp);
    }
  }

  return g;
}

/* Initializes the computations array with all user computations
   at the given utilities.  Also creates a graph of these
   computations. */
void create_comp_list(struct user_utility *utility){
  graph g;

  /* initialize the computations array */
  nspec_comps = 0;
  nspec_opt_comps = 0;
  tst_travel(&users, ccl_visit, &utility);
  ntotal_comps = nspec_comps;

#ifdef debug
  {
    int i;
    printf("create_comp_list: utility=%lf\n",utility->utility);
    for (i=0; i<ntotal_comps; i++)
      printf("  computation %d=%s\n",i,computations[i].name);
    for (i=0; i<nspec_opt_comps; i++)
      printf("  opt computation %d=%s\n",i,opt_computations[i].name);
  }
#endif

  /* calculate the connected components */
  g = create_comp_graph(nspec_comps,"user comps");
  if (connected_comps != NULL)
    free(connected_comps);
  get_connected_comps(g, &connected_comps, &nconnected_comps);
  free_graph(g);
 }

struct {
	xps_t xps;
	map_t op;
	double interval;
	double maxdelay;
	char *location;
	struct print_channel *inputs;
	map_t comps;
	struct spec *alts;
	int nalts;
} nodes_parser;

struct {
	xps_t xps;
	double try;
	double succ;
	double estimate;
} report_parser;

struct {
	xps_t xps;
	map_t op;
	double interval;
	double maxdelay;
	char *location;
	struct print_channel *inputs;
	struct print_channel *inserts;
	map_t comps;
	struct computation *icomps[MAX_I_COMPS];
	int num_icomps;
	struct spec *alts;
	int nalts;
} spec_parser;

void got_send_try(void *env, struct stack_entry *se, int depth){
	double rate;

	if (sscanf(se[depth - 1].data, "%lf", &rate) != 1) {
		err_warning("send/try: bad rate");
		return;
	}
	report_parser.try = rate;
}

void got_send_succ(void *env, struct stack_entry *se, int depth){
	double rate;

	if (sscanf(se[depth - 1].data, "%lf", &rate) != 1) {
		err_warning("send/succ: bad rate");
		return;
	}
	report_parser.succ = rate;
}

void got_send_estimate(void *env, struct stack_entry *se, int depth){
	double rate;
	/* Get the estimate */
	if (sscanf(se[depth - 1].data, "%lf", 
		   &rate) != 1) {
		err_warning("estimate: bad rate");
		return;
	}
	report_parser.estimate = rate;
}

static int get_net(char *report, struct stack_entry *se, 
		   int depth, char **name, int *vers){
	void **p;
	int net;

	p = tst_find(&se[depth - 1].attrs, "name", 0);
	if (p == 0 || (*name = *p) == 0) {
		err_warning("%s: no name",report);
		return -1;
	}
	p = tst_find(&se[depth - 1].attrs, "version", 0);
	if (p == 0 || *p == 0) {
		err_warning("%s: no version",report);
#ifdef notdef
		return -1;
#else
		*vers = version;
#endif
	}
	else {
		*vers = atoi(*p);
	}

	/* Ignore if not for the current version.
	 */
	if (*vers != version) {
		err_warning("%s report version %d != curr %d",
			    report,*vers,version);
		return -1;
	}

	/* Look up the link.
	 */
	for (net = 0; networks[net].name != 0; net++) {
		if (strcmp(networks[net].name, *name) == 0) {
			break;
		}
	}
	if (networks[net].name == 0) {
		err_warning("%s report for unknown link '%s'", report,*name);
		return -1;
	}

	return net;
}

static struct timeval starttime, scratch;

static void print_event(struct timeval *tm, char *event_name, 
			char *format, ...){
  va_list ap;

  if (starttime.tv_sec == 0)
    starttime = *tm;
  scratch.tv_sec = tm->tv_sec - starttime.tv_sec;
  scratch.tv_usec = tm->tv_usec - starttime.tv_usec;
  if (scratch.tv_usec < 0) {
    scratch.tv_usec += 1000000;
    scratch.tv_sec--;
  }

  if (!quiet) {
    printf("%d.%06d %s ",(int)scratch.tv_sec,(int)scratch.tv_usec,event_name);
    va_start(ap, format);
    vprintf(format, ap);
    va_end(ap);
  }
}

#ifndef DO_PACKETPAIR
/* When not using packetpair estimates, we periodically increase the
 * capacity of each link, with the intention that once we "creep"
 * beyond a certain threshhold, then a reconfiguration will be tried.
 * This will in effect test if enough bandwidth is available; if not
 * the configuration will fall back to the original.
 */
void creep_net_capacity(struct timeval *tm) {
  int net, usecs;

  for (net = 0; networks[net].name != 0; net++) {

    /* Don't creep a link for which we got a definitive b/w
       estimate recently */
    if ((tm->tv_sec - networks[net].last_max_update.tv_sec) <=
	parameters.min_reconfig_delta) {
      networks[net].last_creep = *tm; /* update the time */
      continue;
    }

    /* Figure out the last time we did a creep on this network. */
    usecs = 
      (tm->tv_sec - networks[net].last_creep.tv_sec) * 1000000 +
      (tm->tv_usec - networks[net].last_creep.tv_usec);

    /* Now creep the bandwidth some for each second that has elapsed */
    if ((usecs / 1000000) > parameters.min_reconfig_creep_delta) {
      while (usecs > 1000000) {
	networks[net].capacity = 
	  networks[net].capacity * (1 + parameters.creep_rate);
	usecs -= 1000000;
	if (networks[net].capacity >= networks[net].link_capacity) {
	  networks[net].capacity = networks[net].link_capacity;
	}
	else {
	  print_event(tm,"BWCREEP","%s %.2lf\n",
		      networks[net].name, networks[net].capacity);
	}
      }
      networks[net].last_creep = *tm; /* update the time */
    }
  }
}
#endif


/* After receiving a b/w monitoring report, recalculate the capacity
 * of the link in question.  Takes into account "real" measurements of
 * actual perceived throughput, and estimates using packet pairs (if
 * enabled).  */
void got_send(void *env, struct stack_entry *se, int depth){
  int net, version;
  char *name;
  double old_capacity;
  struct timeval tm;
#ifdef DO_PACKETPAIR
  int secs, usecs;
  double scale;
#endif

  /* Get the current time */
  gettimeofday(&tm,NULL);

  /* Figure out what link this report is for */
  net = get_net("send",se,depth,&name,&version);
  if (net == -1) goto cleanup;
  
  if (networks[net].last_creep.tv_sec == 0) {
    networks[net].last_creep = tm;
    networks[net].last_max_update = tm;
  }
  old_capacity = networks[net].capacity;

  /* See if we got a b/w report */
  if (report_parser.try != -1.0) {
    assert(report_parser.succ != -1.0);

    /* Note this report value as a lower bound */
    networks[net].min_capacity = report_parser.succ;
/*     networks[net].last_min_update = tm; */

    /* Adjust the upper bound on the capacity if either
       it's greater than the current upper bound, or we couldn't
       send as much as we wanted. */
    if (((1+EPS_BW) * report_parser.succ) < report_parser.try ||
	report_parser.succ > networks[net].capacity) {
      
      networks[net].max_capacity = report_parser.succ;
      networks[net].last_max_update = tm;
      
      networks[net].capacity = report_parser.succ;

      print_event(&tm,"BWUPDATE","%s %.2lf %.2lf %.2lf\n",
		  name, networks[net].capacity, report_parser.succ,
		  report_parser.try);

      /* This is a definitive update setting a known upper bound
         on the capacity. */
      if (((1+EPS_BW) * report_parser.succ) < report_parser.try) {
	networks[net].last_max_update = tm;
	networks[net].last_creep = tm;
      }
    }
    else {
      print_event(&tm,"BWREPORT","%s %.2lf %.2lf\n",
		  name, networks[net].capacity, report_parser.try);
    }
  }

  /* See if we got a b/w estimate */
  if (report_parser.estimate != -1.0) {
#ifdef DO_PACKETPAIR    
    /* Figure out the last time we got a real update */
    secs = tm.tv_sec - networks[net].last_max_update.tv_sec;
    usecs = tm.tv_usec - networks[net].last_max_update.tv_usec;
    if (usecs < 0) {
      secs--; usecs += 1000000;
    }
    
    /* Adjust our capacity based on the last direct value
       and the current estimate, scaled based on the elapsed time */
    scale = secs * ESTIMATE_SCALE;
    if (scale > 1.0)
      scale = 1.0;
    
    /* TODO: need to take into account the staleness of the
       minimum measurement */
    networks[net].capacity = 
      networks[net].max_capacity * (1.0 - scale) +
      (networks[net].min_capacity * (1.0 - scale) +
       report_parser.estimate * scale) * scale;
    
    print_event(&tm,"ESTIMATE", "%s %.2lf %.2lf\n",
		name, version, report_parser.estimate, networks[net].capacity);
#else
    printf("***ignoring estimate report\n");
#endif
  }

#ifndef DO_PACKETPAIR
  creep_net_capacity(&tm);
#endif

 cleanup:       
	report_parser.try = -1.0;
	report_parser.succ = -1.0;
	report_parser.estimate = -1.0;
}

void got_nodes_node_capacity(void *env, struct stack_entry *se, int depth){
}

void got_nodes_node_host(void *env, struct stack_entry *se, int depth){
}

void got_nodes_node_port(void *env, struct stack_entry *se, int depth){
}

void got_nodes_node(void *env, struct stack_entry *se, int depth){
}

void got_nodes(void *env, struct stack_entry *se, int depth){
}

/* Got an operation.
 */
void got_spec_alt_comp_op(void *env, struct stack_entry *se, int depth){
	if (spec_parser.op != 0) {
		err_warning("spec/alt/comp/op: duplicate");
		tst_release(&spec_parser.op, mem_release, 0);
	}
	spec_parser.op = se[depth - 1].attrs;
	se[depth - 1].attrs = 0;
}

void got_spec_alt_comp_interval(void *env, struct stack_entry *se, int depth){
	double interval;

	if (spec_parser.interval != 0) {
		err_warning("spec/alt/comp/interval: duplicate");
	}
	if (sscanf(se[depth - 1].data, "%lf", &interval) != 1) {
		err_warning("spec/alt/comp/rate: bad interval");
		return;
	}
	spec_parser.interval = interval;
}

void got_spec_alt_comp_maxdelay(void *env, struct stack_entry *se, int depth){
	double maxdelay;

	if (spec_parser.maxdelay != 0) {
		err_warning("spec/alt/comp/maxdelay: duplicate");
	}
	if (sscanf(se[depth - 1].data, "%lf", &maxdelay) != 1) {
		err_warning("spec/alt/comp/maxdelay: bad maxdelay");
		return;
	}
	spec_parser.maxdelay = maxdelay;
}

void got_spec_alt_comp_location(void *env, struct stack_entry *se, int depth){
  if (spec_parser.location != 0) {
    err_warning("spec/alt/comp/location: duplicate");
    free(spec_parser.location);
  }
  spec_parser.location = string_copy(se[depth - 1].data);
}

void got_spec_alt_comp_inputs_input(void *env,
				    struct stack_entry *se, int depth){
  void **p;

  p = tst_find(&se[depth - 1].attrs, "name", 0);
  if (p == 0 || *p == 0) {
    err_warning("input: no name");
    return;
  }
  if (spec_parser.inputs == 0) {
    spec_parser.inputs = mc_open();
    pc_print(spec_parser.inputs, "%s", *p);
  }
  else {
    pc_print(spec_parser.inputs, " %s", *p);
  }
}

void got_spec_alt_comp_inserts_insert(void *env,
				      struct stack_entry *se, int depth){
  void **p;
  
  p = tst_find(&se[depth - 1].attrs, "name", 0);
  if (p == 0 || *p == 0) {
    err_warning("input: no name");
    return;
  }
  if (spec_parser.inserts == 0) {
    spec_parser.inserts = mc_open();
    pc_print(spec_parser.inserts, "%s", *p);
  }
  else {
    pc_print(spec_parser.inserts, " %s", *p);
  }
}

void got_spec_alt_comp(void *env, struct stack_entry *se, int depth){
  struct computation *comp;
  void **p;
  int node;

  comp = (struct computation *) calloc(1, sizeof(*comp));
  p = tst_find(&se[depth - 1].attrs, "name", 0);
  if (p == 0 || *p == 0) {
    comp->name = mem_print("unknown");
  }
  else {
    comp->name = string_copy(*p);
  }
  comp->operation = spec_parser.op;
  spec_parser.op = 0;

  /* if this is a send or recv, add an attribute saying so
   */
  p = tst_find(&comp->operation,"cmd",0);
  if (p == 0 || (*p) == 0) {
    err_warning("no cmd specified in operation !");
  }
  else {
    if (!strcmp(*p,"send") || !strcmp(*p,"recv")) {
      p = tst_find(&comp->operation, "origin", 1);
      *p = string_copy("user");
    }
  }

  comp->interval = spec_parser.interval;
  spec_parser.interval = 0;
  comp->max_delay = spec_parser.maxdelay;
  spec_parser.maxdelay = 0;

  /* Find the location.
   */
  if (spec_parser.location != 0) {
    /* If it's a wild-card, then we insert it on all intervening
       nodes.  Note it with a negative assignment. */
    if (!strcmp(spec_parser.location,"*")) {
      comp->location = -1;
    }
    else {
      for (node = 0; node < nnodes; node++) {
	if (strcmp(nodes[node].name, spec_parser.location) == 0) {
	  break;
	}
      }
      if (node == nnodes) {
	err_warning("bad location %s!", spec_parser.location);
      }
      else {
	comp->location = comp->assigned_loc = node + 1;
      }
    }
    free(spec_parser.location);
    spec_parser.location = 0;
  }

  /* If no location has been assigned yet, temporarily stick it on
   * the first node.
   */
  if (comp->assigned_loc == 0) {
    comp->assigned_loc = 1;
  }
  
  /* Look up the computation's cost and output size.
   */
  comp->cost = cost_lookup(comp->operation, &comp->output_size);

  /* Get the inputs and insertions
   */
  if (spec_parser.inputs != 0) {
    comp->input = mc_close(spec_parser.inputs);
    spec_parser.inputs = 0;
  }
  else {
    comp->input = mem_print("");
  }

  if (spec_parser.inserts != 0) {
    comp->insert = mc_close(spec_parser.inserts);
    spec_parser.inserts = 0;
  }
  else {
    comp->insert = mem_print("");
  }

  /* Add to set of computations.
   */
  if (comp->location != -1) {
    p = tst_find(&spec_parser.comps, comp->name, 1);
    if (*p != 0) {
      err_warning("duplication comp");
    }
    *p = comp;
  }
  else {
    assert(spec_parser.num_icomps < MAX_I_COMPS);
    comp->location = 0; /* probably not necessary */
    spec_parser.icomps[spec_parser.num_icomps++] = comp;
  }
}

void got_spec_alt(void *env, struct stack_entry *se, int depth){
  void **p;
  double utility;
  int i, j;

  /* Get the utility value.
   */
  p = tst_find(&se[depth - 1].attrs, "utility", 0);
  if (p == 0 || *p == 0) {
    err_warning("no utility specified");
    utility = 1;
  }
  else {
    if (sscanf(*p, "%lf", &utility) != 1) {
      err_warning("bad utility specified");
      utility = 1;
    }
    else {
      if (utility < 0) {
	err_warning("utility < 0");
	utility = 0;
      }
      else if (utility > 1) {
	err_warning("utility > 1");
	utility = 1;
      }
    }
  }

  /* Find place to insert.
   */
  for (i = 0; i < spec_parser.nalts; i++) {
    if (utility == spec_parser.alts[i].utility) {
      err_warning("duplicate utility");
      break;
    }
    if (utility > spec_parser.alts[i].utility) {
      break;
    }
  }

  /* Grow list.
   */
  if (spec_parser.nalts == 0) {
    spec_parser.alts = (struct spec *) malloc(sizeof(*spec_parser.alts));
  }
  else {
    spec_parser.alts = (struct spec *) realloc((char *) spec_parser.alts,
					       (spec_parser.nalts + 1) * 
					       sizeof(*spec_parser.alts));
  }
  
  /* Insert.
   */
  for (j = spec_parser.nalts; j > i; j--) {
    spec_parser.alts[j] = spec_parser.alts[j - 1];
  }
  spec_parser.alts[i].comps = spec_parser.comps;
  spec_parser.comps = 0;
  spec_parser.alts[i].utility = utility;
  spec_parser.alts[i].icomps = malloc(spec_parser.num_icomps *
				      sizeof(struct computation *));
  memcpy(spec_parser.alts[i].icomps,spec_parser.icomps,
	 spec_parser.num_icomps * sizeof(struct computation *));
  spec_parser.alts[i].num_icomps = spec_parser.num_icomps;
  spec_parser.num_icomps = 0;
  spec_parser.nalts++;
#ifdef debug
  printf("got another spec alt at utility %lf; nalts=%d\n",
	 utility,spec_parser.nalts); 
#endif
}

/* figures out which nodes we have assigned components to, not
   counting nodes that just have monitors */
static char *nodes_in_assignment() {
  int comp, first;
  int *which_nodes = calloc(nnodes,sizeof(int));
  int node, sz;
  char *buf;

  /* which nodes are we using? */
  for (comp = 0; comp < ntotal_comps; comp++) {
    if (computations[comp].type != COMP_MONITOR) {
#ifdef debug
      if (which_nodes[computations[comp].assigned_loc-1] != 1)
	printf("found node |%s|\n",
	       nodes[computations[comp].assigned_loc-1].name);
#endif
      which_nodes[computations[comp].assigned_loc-1] = 1;
    }
  }

  /* how big must our output string be? */
  for (sz=0, node=0; node<nnodes; node++) {
    if (which_nodes[node])
      sz += (strlen(nodes[node].name) + 1);
  }
  sz++;
  buf = malloc(sz);
  buf[0] = '\0';

  /* construct the output string */
  for (first=1,node=0; node<nnodes; node++)
    if (which_nodes[node]) {
      if (!first)
	strcat(buf,".");
      else
	first=0;
      strcat(buf,nodes[node].name);
    }
  free(which_nodes);

  /* will be freed by caller */
  return buf;
}

static void print_config(struct computation *cached_comps, int num_comps,
			 struct computation *new_comps, int new_num_comps) {
  int i;
  printf("cached comps:\n");
  for (i=0;i<num_comps;i++) {
    printf("  %s\n",cached_comps[i].name);
  }
  printf("new comps:\n");
  for (i=0;i<new_num_comps;i++) {
    printf("  %s\n",new_comps[i].name);
  }
}

/* #define debug */
static int is_new_config(struct computation *cached_comps, int num_comps,
			 struct computation *new_comps, int new_num_comps,
			 char *reason, int sz) {
  int i, j;
#ifdef debug
  printf("is_new_config:\n");
#endif
  if (new_num_comps != num_comps) {
    if (reason != NULL)
      snprintf(reason,sz,"# new comps = %d != %d",new_num_comps,num_comps);
#ifdef debug
    print_config(cached_comps,num_comps,new_comps,new_num_comps);
#endif
    return 1;
  }
  else {
    for (i=0;i<new_num_comps;i++) {
      for (j=0;j<new_num_comps;j++) {
	if (!strcmp(cached_comps[j].name,new_comps[i].name)) {
	  if (cached_comps[j].type == new_comps[i].type &&
	      cached_comps[j].assigned_loc == new_comps[i].assigned_loc)
	    break; /* found it */
#ifdef debug
	  else
	    printf("**comp %s: cacheloc=%d, cachetyp=%d, loc=%d, typ=%d\n",
		   cached_comps[j].name, cached_comps[j].type,
		   cached_comps[j].assigned_loc, new_comps[i].type,
		   new_comps[i].assigned_loc);
#endif
	}
      }
      if (j == new_num_comps) { /* didn't find it */
	if (reason != NULL)
	  snprintf(reason,sz,"new comp %s has no match in old comps",
		   new_comps[i].name);
	return 1;
      }
    }
    return 0; /* no changes; found all the computations */
  }
}

/* Takes a workable schedule and tries to improve it by adjusting
   individual user utilities. */

static void optimize_sched(struct user_utility *user_utility,
			   struct computation *best_computations,
			   int *best_num_comps){
  double score;
  int i;
  int got_improvement;

  /* If there is only one user, we know this is optimal,
     so we just return. */
  if (num_users == 1)
    return;

  /* Try increasing each user's utility value, and keeping it
     as long as the score is nonzero.
     
     TODO. We give the first user priority over the second; we could
     add explicit priorities to adjust htis problem. */
  do {
    got_improvement = 0;
    for (i=0; i<num_users; i++) {

      /* Nothing to do if we're at the maximum */
      if (user_utility[i].utility == 1.0)
	continue;

      else {
	double best_utility = user_utility[i].utility;
	struct user *u = user_utility[i].u;
	double try_utility = 2.0; /* some # > 1.0 */
	int j;

	/* Find the next alternative configuration whose utility is
	   just greater than the current best; if there is none, we
	   move on to the next user. */
	for (j=0; j<u->nalts; j++) {
	  if ((u->alts[j].utility > best_utility) &&
	      (u->alts[j].utility < try_utility)) {
	    try_utility = u->alts[j].utility;
	  }
	}
	
	/* Nothing better, keeping going */
	if (try_utility > 1.0)
	  continue;
	
	/* Otherwise, try to adjust this utility to this amount */
	else {
#ifdef debug
	  printf("trying to optimize user %d to util %.2lf from %.2lf\n",
		 i,try_utility,best_utility);
#endif
	  user_utility[i].utility = try_utility;
	  create_comp_list(user_utility);
	  score = find_good_config(0);
	  /* Didn't work; restore the old value */
	  if (score < 0) {
	    user_utility[i].utility = best_utility;
	    continue;
	  }
	  /* Did work; keep track of this and continue. */
	  else {
#ifdef debug
	    printf("Got some improvement---adjusting schedule\n");
#endif
	    memcpy(best_computations,computations,ntotal_comps *
		   sizeof(struct computation));
	    *best_num_comps = ntotal_comps;
	    got_improvement = 1;
	  }
	}
      }
    }
  } while (got_improvement);
}

/* Searches for the best configuration linearly among the users'
   utility arrays.  Basically boils an n-dimensional problem down
   to a one-dimensional problem.  */
static void linear_search_for_sched(double hi, double lo,
				    struct user_utility *user_utility,
				    struct computation *best_computations,
				    int *best_num_comps){
  double best_utility = lo, utility, score;
  int i;

  struct computation last_computations[MAX_COMPS];
  int last_nspec_comps = 0;
  double last_score = -1.0;

  /* Do a binary search on utility for a first approximation.
   */
    
  for (utility = hi; lo < (hi-EPS); utility = (lo + hi) / 2) {

    /* Initialize the user utility vector */
    for (i=0; i<num_users; i++) {
      user_utility[i].utility = utility;
    }

    /* Create the list of all computations.
     */
    create_comp_list(user_utility);

    /*
     * If this is the same set of computations as the last 
     * utility, reuse its score.  Otherwise, find a good configuration.
     */
/* #define OPT */
#ifdef OPT
    if (is_new_config(computations, nspec_comps,
		      last_computations, last_nspec_comps,
		      NULL, 0)) {
      memcpy(last_computations,computations,
	     nspec_comps * sizeof(struct computation));
      last_nspec_comps = nspec_comps;
      score = find_good_config(0);
      last_score = score;
    }
    else {
      score = last_score;
#ifdef debug
      printf("User comps did not change, using last score\n");
#endif
    }
#else
#endif
    score = find_good_config(0);

#ifdef debug
    printf("Score for %lf: %lf (lo=%lf,hi=%lf)\n", utility, score, lo, hi);
#endif
    /* See if this configuration meets the demands.  If not,
     * try lower.  If so, try higher.
     */
    if (score < 0) {
      hi = utility;
      /* copy back the last good config, so that the algorithm
	 works properly in find_good_config() */
      memcpy(computations,best_computations,*best_num_comps *
	     sizeof(struct computation));
      ntotal_comps = *best_num_comps;
    }
    else {
      lo = utility;
      /* Cache best setup
       */
#ifdef debug
      printf("best num comps=%d utiliy=%lf\n",*best_num_comps,lo);
#endif
      
      memcpy(best_computations,computations,ntotal_comps *
	     sizeof(struct computation));
      *best_num_comps = ntotal_comps;
      best_utility = lo;
    }
  }
  /* set the utility vector properly */
  for (i=0; i<num_users; i++) {
    user_utility[i].utility = best_utility;
  }
}

static int improves_utility(struct user_utility *new_utility,
			    double current_config_score) {
  /* if the old configuration score is < 0, we know to just
     use the new one. */
  if (current_config_score < 0.0)
    return 1;
  else 
    return !utility_eq(new_utility,num_users,
		       current_utility, current_users);
}			    

/* Calculates a new global schedule.  The two arguments implement a
   closure that does something with the resulting schedule.  For
   example, calling dump_assignments might print the schedule to the
   screen, or it might send the schedule to some other node, etc.
   This function is (obviously) chosen by the caller. */
static void update_sched(void *env,
			 void (*dump_assignments)(void *env, 
						  char *addr,
						  char *assignment)){
  struct computation cached_computations[MAX_COMPS];
  struct computation best_computations[MAX_COMPS];
  int cached_num_comps = ntotal_comps;
  int best_num_comps = 0;
  struct user_utility *user_utility;
  struct timeval start, end;
  char reason[200];
  double old_score;

  gettimeofday(&start,NULL);
	
  /* Initialize user utility array */
  user_utility = init_utility();

  /* Cache current setup and get its score under current conditions
   */
  memcpy(cached_computations,computations,ntotal_comps *
	 sizeof(struct computation));
  old_score = calculate_score();

  /* Find a linear-based configuration
   */
  linear_search_for_sched(1,0,user_utility,
			  best_computations,&best_num_comps);

  /* Attempt to optimize it
   */
  optimize_sched(user_utility,
		 best_computations,&best_num_comps);

  gettimeofday(&end,NULL);
  start.tv_sec = end.tv_sec - start.tv_sec;
  start.tv_usec = end.tv_usec - start.tv_usec;
  if (start.tv_usec < 0) {
    start.tv_usec += 1000000;
    start.tv_sec--;
  }
  if (!quiet)
    printf("ALG TIME = %d.%06d\n", (int)start.tv_sec, (int)start.tv_usec);
    PRINT_TIME(t1);
    PRINT_TIME(t2);

  /* Update the local schedulers if the configuration is different
   * and the utility vector has changed.  If the config changes but
   * the util stays the same, it's better to leave things as they are
   * to avoid instability during reconfig.
   */
  if (is_new_config(cached_computations,cached_num_comps,
		    best_computations,best_num_comps,
		    reason,sizeof(reason)) &&
      improves_utility(user_utility, old_score)) {
    int i;
    char buf[1000];
    char buf2[100];
    struct timeval tm;
    char *which_nodes;
    struct print_channel *pc = mc_open();
    char *graph;

    version++;
  
    /* Create the final computation list and its configuration.
     */
    create_comp_list(user_utility);
    (void) find_good_config(1);
  
    /* Send the new assignments to the hosts
     */
    gettimeofday(&tm,NULL);
    buf[0] = '\0';
    for (i=0; i<num_users; i++) {
      snprintf(buf2,sizeof(buf2),"%.2lf",user_utility[i].utility);
      strcat(buf,buf2);
      if (i != (num_users-1))
	strcat (buf,"-");
    }
    which_nodes = nodes_in_assignment();
/*     print_event(&tm,"RECONFIG","%s %s\n", buf, which_nodes); */
/*     print_event(&tm,"  reason:","%s\n", reason); */
/*     graph_assignments(pc); */
    printf("%s\n",graph = mc_close(pc));
    free(which_nodes);
    free(graph);

    assert(is_new_config(cached_computations,cached_num_comps,
			 computations,ntotal_comps,reason,
			 sizeof(reason)));
    mmsched_send_assignments(env, dump_assignments);
    last_reconfig_time = tm;

    /* update the current utility vector.
     */
    free(current_utility);
    current_utility = user_utility;
    current_users = num_users;
  }

  /* Restore the old configuration if no benefit or change
   */
  else {
    free(user_utility);
    memcpy(computations,cached_computations,cached_num_comps *
	   sizeof(struct computation));
    ntotal_comps = cached_num_comps;
  }
}

/* Received a new user spec.
 */
char *mmsched_enter(void *env,
		    void (*dump_assignments)(void *env, char *addr,
					     char *assignment),
		    char *user, char *spec){
  xp_t xp = xp_create(spec_parser.xps);
  void **p;
  struct user *u;

  /* Parse the spec.
   */
  xp_data(xp, spec, strlen(spec));
  xp_free(xp);
  p = tst_find(&users, user, 1);
  if ((u = *p) == 0) {
    *p = u = (struct user *) calloc(1, sizeof(*u));
    num_users++;
  }
  u->alts = spec_parser.alts;
  spec_parser.alts = 0;
  u->nalts = spec_parser.nalts;
  spec_parser.nalts = 0;

  /* Update the schedule.
   */
  if (dump_assignments != NULL)
    update_sched(env,dump_assignments);

  return mem_print("Accepted!");
}

/* Received a new report of the form
 *	<send name="name of operation" version="current version">
 *	  <try># bytes attempting to send per sec</try>
 *    <succ># bytes succeeded to send per sec</succ>
 *  </send>
 */
char *mmsched_report(void *env,
		     void (*dump_assignments)(void *env,
					      char *addr,
					      char *assignment),
		     char *user, char *spec){
  xp_t xp = xp_create(report_parser.xps);
  struct timeval tm;

  /* Parse the spec.
   */
  xp_data(xp, spec, strlen(spec));
  xp_free(xp);

  /* Update the schedule.
   */
  gettimeofday(&tm,NULL);
  if ((tm.tv_sec - last_reconfig_time.tv_sec) > parameters.min_reconfig_delta)
    update_sched(env,dump_assignments);

  return mem_print("Accepted!");
}

static void spec_parser_init(void){
  xps_t xps = xps_create();
  void *env = &spec_parser;

  /* Set up the XML parser to parse user specifications.
   */
  spec_parser.xps = xps;
  xps_upcall(xps, "/spec/alt/comp/op", got_spec_alt_comp_op, env);
  xps_upcall(xps, "/spec/alt/comp/interval", got_spec_alt_comp_interval, env);
  xps_upcall(xps, "/spec/alt/comp/maxdelay", got_spec_alt_comp_maxdelay, env);
  xps_upcall(xps, "/spec/alt/comp/location", got_spec_alt_comp_location, env);
  xps_upcall(xps, "/spec/alt/comp/inputs/input",
	     got_spec_alt_comp_inputs_input, env);
  xps_upcall(xps, "/spec/alt/comp/inserts/insert",
	     got_spec_alt_comp_inserts_insert, env);
  xps_upcall(xps, "/spec/alt/comp", got_spec_alt_comp, env);
  xps_upcall(xps, "/spec/alt", got_spec_alt, env);
}

static void report_parser_init(void){
  xps_t xps = xps_create();
  void *env = &report_parser;

  /* Set up the XML parser to parse report specifications.
   */
  report_parser.xps = xps;
  report_parser.try = -1.0;
  report_parser.succ = -1.0;
  report_parser.estimate = -1.0;
	
  xps_upcall(xps, "/report/send/try", got_send_try, env);
  xps_upcall(xps, "/report/send/succ", got_send_succ, env);
  xps_upcall(xps, "/report/send/estimate", got_send_estimate, env);
  xps_upcall(xps, "/report/send", got_send, env);
}

static void nodes_parser_init(void){
  xps_t xps = xps_create();
  void *env = &nodes_parser;

  /* Set up the XML parser to parse nodes specifications.
   */
  nodes_parser.xps = xps;
  xps_upcall(xps, "/nodes/node/capacity", got_nodes_node_capacity, env);
  xps_upcall(xps, "/nodes/node/host", got_nodes_node_host, env);
  xps_upcall(xps, "/nodes/node/port", got_nodes_node_port, env);
  xps_upcall(xps, "/nodes/node", got_nodes_node, env);
  xps_upcall(xps, "/nodes", got_nodes, env);
}

void mmsched_init(int confignum){
  if (confignum == -1) {
    assert(topology != NULL);

    nodes = topology->nodes;
    nnodes = topology->num_nodes;
    networks = topology->nets;
    nlinks = topology->num_nets;
  }
  else {
    if (confignum >= NUM_CONFIGS) {
      fprintf(stderr,"bad configuration number %d\n",confignum);
      exit(1);
    }
    nodes = node_configs[confignum];
    networks = network_configs[confignum];
  }

  /* Initialize the operation cost table.
   */
  init_operations();

  /* Initialize the XML parsers.
   */
  spec_parser_init();
  report_parser_init();
  nodes_parser_init();

  /* Create a connection matrix of the system based on the
   * network and node specifications.
   */
  create_connection_matrix();

#ifdef notdef

  /* Print it.
   */
  printf("Utility: %f\n", actual_utility);
  print_config();
  
  print_xml(2);
#endif
}
