/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

#ifndef MMSCHED_H_
#define MMSCHED_H_

#include <sys/time.h>
#include "oslib.h"

#ifndef MAX
#define MAX(x,y) ((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x,y) ((x) < (y) ? (x) : (y))
#endif

#define SCHEDSVR_HOST	"localhost"
#define SCHEDSVR_PORT	7777
#define MAX_BW			1000000
#define EPS_BW			0.01
#define ESTIMATE_SCALE		0.2

#define LOW_PORT		7000			/* lowest TCP port to allocate */
#define HIGH_PORT		SCHEDSVR_PORT		/* ceiling for TCP ports */
#define MAX_COMPS		1000				/* maximum # computations */
#define MAX_I_COMPS		10				/* maximum # of intermediate computations */
#define MAX_SPECS		10				/* maximum # alternatives */
#define EPS				.05				/* used in termination of utility determination */

struct sched_parameters {
  int schedsvr_port;      /* port listening on for HTTP configs/reports */
  char *schedsvr_host;    /* the host the server is running on */
  int nextport;           /* the starting port of inserted send/recv's */
  int emulab_link;        /* whether we send to the links on emulab */
  int min_reconfig_delta; /* minimum # secs between (any two) reconfig */
  int min_reconfig_creep_delta;  /* minimum # secs between a reconfig
				    and when we start creeping the link */
  double creep_rate;      /* percentage by which to increase B/W when
			     creeping. */
};
extern struct sched_parameters parameters;

/* Specification of compute nodes.  Specify using XML as follows:
 *
 *	<nodes>
 *	  <node name="pc1">
 *	    <capacity>1e8</capacity>
 *		<host>longhorn.cs.cornell.edu</host>
 *		<port>7771</port>
 *	  </node>
 *	  <node .../>
 *  </nodes>
 */
struct node {
	char *name;
	double capacity;	/* compute units per second currently available */
	char *host;			/* TCP address */
	int port;			/* TCP port */
	int nextport;		/* to generate TCP port numbers */
	int member;			/* used in connection matrix computation only */
	double load;		/* used in scoring: total #compute units per second necessary */
	double max_step;	/* maximum #units a particular comp may have to wait */
};

/* Specification of networks.
 */
struct network {
	char *name;
	double capacity;	/* bytes / second */
	double link_capacity;	/* bytes / second */
	double latency;		/* minimum latency in seconds */
	char *nodes;		/* members */
	char *emulab_hops[2];	/* addresses to use on emulab */
	double load;		/* used in scoring: total #bytes /
				   second necessary */
	double max_step;	/* maximum #bytes a particular frame
				   may have to wait */

	double max_capacity; /* value in last try != send report */
	struct timeval last_max_update; /* staleness of above value */
	double min_capacity; /* value in last try == send report */
	struct timeval last_creep;
};

/* Specification of work to be done.  Specified as DAG.  Each node specifies
 * one simple operation and its parameters, the minimum interval between
 * starts of the operation, and the output frame type and other attributes.
 * Optionally, the location can be fixed at a particular compute node.  For
 * each node, the (set of, but typically one) next node is specified, as well
 * as which of the inputs to that node. 
 */

struct computation {
	char *name;		/* name of node in DAG */
	map_t operation;	/* description of operation */
	double cost;		/* # compute cycles */
	double interval;	/* minimum time between starting operations */
	int enforce;		/* indicates whether interval should
				   be enforced or simply assumed by
				   local scheduler */
	int location;		/* where to run.  0 if anywhere */
	int output_size;	/* output frame size */
	char *input;		/* list of inputs from other nodes */
	char *insert;		/* list of comps scheduler may insert */
	int assigned_loc;	/* Assigned location */
	double max_delay;	/* required max delay in seconds.  0 is don't care */

	/* Rest are temporary fields.
	 */
	int *in_links;		/* like inputs, but updated for send/recv ops */
	int ninputs;		/* # inputs */
	enum { COMP_USER, COMP_SEND, COMP_RECV, COMP_MONITOR } type;
	double delay;		/* actual maximum delay to sources.  0 is uninitialized */
};

struct topology {
  int num_nodes;
  struct node *nodes;
  int num_nets;
  struct network *nets;
};


#endif
