/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* Derived from oslib.h, Copyright 2000 by Reliable Network Solutions,
   Inc. */
#ifndef __OSLIB_H__
#define __OSLIB_H__

#include <stdlib.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#ifdef notdef
#define srandom(seed)	USE RAND_SEED
#define random()		USE RAND_GET
#define srand(seed)		USE RAND_SEED
#define rand()			USE RAND_GET
#endif

#ifdef WIN32

#include <winsock.h>
#include <time.h>
#else /* WIN32 */

#include <sys/time.h>

#endif /* WIN32 */

#ifdef MALLOC_DEBUG
void *Mem_Malloc(int, char *, int);
void *Mem_Calloc(int, int, char *, int);
void *Mem_Realloc(void *, int, char *, int);
void Mem_Free(void *, char *, int);

#define malloc(s)			Mem_Malloc((s), __FILE__, __LINE__)
#define calloc(n, s)		Mem_Calloc((n), (s), __FILE__, __LINE__)
#define realloc(p, s)		Mem_Realloc((p), (s), __FILE__, __LINE__)
#define free(p)				Mem_Free((p), __FILE__, __LINE__)
#endif /* MALLOC_DEBUG */

#ifdef __cplusplus
extern "C"{
#endif 

/* Return whether the given character is considered a blank.
 */
#define is_blank(c) \
	((c) == ' ' || (c) == '\t' || (c) == '\n' || (c) == '\r')

struct print_channel {
	char *buf;
	int real_length;
	int data_length;
};

#define pc_putchar(pc, c) \
	do if ((pc)->real_length <= (pc)->data_length) \
		pc_putc((pc), (c)); \
	else { \
		(pc)->buf[(pc)->data_length - 1] = (c); \
		(pc)->buf[(pc)->data_length++] = 0; \
	} while (0)

#define mc_current(pc)	((pc)->buf)
#define mc_size(pc)	((pc)->data_length - 1)

struct token {
	char *str;
};

#ifdef __cplusplus
typedef enum { Xfalse, Xtrue } bool_t;
#else
typedef enum { false, true } bool_t;
#endif

typedef double timestamp_t;
typedef struct map_node *map_t;
typedef unsigned int unicode_t;
typedef struct xml *xml_t;

#ifdef SSLEAY

#include "tsec.h"

typedef EVP_PKEY *private_key_t;
typedef EVP_PKEY *public_key_t;

#else /* SSLEAY */

typedef char *private_key_t;
typedef char *public_key_t;

#endif /* SSLEAY */

struct print_channel *mc_open(void);
void pc_print(struct print_channel *pc, const char *format, ...);
void pc_putc(struct print_channel *pc, int c);
void pc_puts(struct print_channel *pc, const char *s, int len);
void pc_vprint(struct print_channel *pc, const char *format, va_list ap);
bool_t pc_read_file(struct print_channel *pc, char *file);
void mc_release(struct print_channel *mc);
char *mc_close(struct print_channel *mc);
char *mem_string_copy(const char *, int);
char *mem_print(const char *, ...);
char *mem_double(double);
char *mem_alloc_double(double);
void mem_release(void *, void *);

typedef struct net_chan *chan_t;

void net_init(void);
void net_send(char *addr, void *msg, int size);
void net_recv(void (*got_input)(void *, char *, int), void *env);
bool_t net_open_mcast(char *addr);
bool_t net_open_bcast(char *addr);
void net_block(unsigned int msec);
int net_start_server(unsigned short port,
		void *(*got_client)(void *env, chan_t chan), void *env,
		void (*got_input)(void *, char *, int));
chan_t net_start_client(const char *addr,
		void (*got_input)(void *, char *, int), void *env);
void net_send_chan(chan_t chan, char *msg, int size);
bool_t net_add_line(int socket,
		void (*got_input)(void *, char *, int), void *env);
void net_watch(void (*upcall)(void *, map_t), void *env);
unsigned short net_bind(unsigned short);
void net_free(void);

/* Ternary search tree indexed by string.
 */
void tst_init(void);
void **tst_insert(map_t *pn, const char *key);
void **tst_lookup(map_t map, const char *s);
void tst_travel(map_t *, int (*upcall)(void *, char *, void **), void *env);
bool_t tst_parse(map_t *, char *info);
bool_t tst_parse_append(map_t *, char *info);
void tst_print(struct print_channel *pc, map_t tst);
void tst_putstr(struct print_channel *pc, const char *s);
char *tst_getstr(char **sp, int *len);
bool_t tst_read(char *file, map_t *map);
void tst_write(char *file, map_t map);
char *tst_unparse(map_t);
void tst_remove(map_t *, const char *s, void (*release)(void *, void *), void *env);
void tst_release(map_t *pn, void (*release)(void *, void *), void *env);
bool_t tst_do_parse(char **info, void (*upcall)(void *, char *, char *, int),
								void *env);

#define tst_find(pn, key, create) \
	((create) ? tst_insert((pn), (key)) : tst_lookup(*(pn), (key)))

timestamp_t time_now();
char *time_to_string(timestamp_t t, bool_t round);
bool_t time_from_string(char *t, timestamp_t *result);

xml_t xml_create(struct print_channel *pc);
void xml_start(xml_t xml, char *fmt, ...);
void xml_start_tag(xml_t xml, char *fmt, ...);
void xml_text(xml_t xml, char *fmt, ...);
bool_t xml_include(xml_t xml, char *file);
bool_t xml_comment_include(xml_t xml, char *file);
void xml_help(xml_t xml, char *file);
void xml_end(xml_t xml, char *tag);
void xml_end_tag(xml_t xml, char *tag);
void xml_release(xml_t xml);
char *xml_convert(char *s);
char *xml_escape(char *s);
char *xml_unescape(char *s);

void token_init(struct token *t, char *s);
char *token_get(struct token *t, char *delim);
void token_final(struct token *t);

unsigned int rand_get();
void rand_seed(unsigned int);

struct stack_entry {
	char *name;					/* XML tag */
	map_t attrs;				/* attributes */
	struct print_channel *pc;	/* this is where we collect the data */
	char *data;					/* this is where we store it when done */
};

typedef struct xsyntax *xps_t;
typedef struct xparse *xp_t;

xps_t xps_create(void);
void xps_upcall(xps_t xps, char *path,
	void (*upcall)(void *env, struct stack_entry *, int), void *env);
void xps_release(xps_t xps);
xp_t xp_create(xps_t xps);
void xp_data(xp_t xp, char *data, int size);
void xp_free(xp_t xp);

unsigned char *uni_to_utf8(unicode_t c, int *size);
unicode_t utf8_to_uni(unsigned char *buf, int len, int *size);

char *file_raw_read(char *file, int *len);
char *fp_read(FILE *fp);
void fp_write(FILE *fp, char *data);
char *file_read(char *file);
char *fp_raw_read(FILE *fp, int *len);
bool_t file_write(char *file, char *data);
bool_t file_remove(char *file);
bool_t file_exists(char *file);
bool_t dir_remove(char *dir);
bool_t dir_create(char *dir);
bool_t dir_change(char *dir);

typedef struct dir_handle *dir_t;
dir_t dir_open(char *);
char *dir_next(dir_t);
void dir_close(dir_t);

void log_init(void);
void log_flush(int level);
void log_vprintf(char *fmt, va_list ap);
void log_prnt(char *format, ...);
void log_printf(char *format, ...);
struct log_sub *log_subscribe(int level, void (*callback)(void *, char *), void *env);
void log_unsubscribe(struct log_sub *ls);
void log_release(void);

void err_log(char *s);
void err_warning(char *fmt, ...);
void err_warning_socket(char *fmt, ...);
void err_fatal(char *fmt, ...);
void err_fatal_socket(char *fmt, ...);
int err_init(void);

void Mem_Done();

/* Every event should start with this header.
 */
typedef struct event {
	struct event *next;				/* useful for lots of stuff... */
	struct event_type *type;		/* type of the event */
	struct ev_channel *release;		/* release channel (identifies source) */
	struct ev_channel *dst;			/* destination channel */
} *ev_t;

/* Event type.  We don't make a distinction between event types and
 * event factories, by the way.
 */
typedef struct event_type {
	struct event_type *parent;
	char *name;
	ev_t free_list;

	/* Auxiliary info.
	 */
	void *env;

	/* Methods.
	 */
	ev_t (*create)(struct event_type *);
	void (*release)(struct event_type *, ev_t);
} *evt_t;

evt_t evt_create(evt_t parent, char *name,
	ev_t (*create)(evt_t), void (*release)(evt_t, ev_t));
void evt_release(evt_t);

#define ev_cast(ev, t)		((struct t *) (ev))

/* Some common event types/factories.
 */
extern evt_t evt_error, evt_close, evt_data, evt_data_src, evt_client;

struct evt_error {
	struct event parent;
	char *descr;
};

struct evt_close {
	struct event parent;
	int dummy;
};

struct evt_data {
	struct event parent;
	struct chunk *chunk;
};

struct evt_data_src {
	struct evt_data parent;
	char *addr;
};

struct evt_client {
	struct event root;
	struct socket_info *si;
};

void act_init(void);
struct activity *act_alloc(void *env);
void *act_env(struct activity *);
void act_release(struct activity *);
bool_t act_outstanding(struct activity *act);
void act_done(void);

bool_t run_block(void);

ev_t ev_empty(evt_t evt, int size);
ev_t ev_create(evt_t evt);
struct ev_channel *ev_channel_alloc(struct activity *act, void (*handler)(void *env, ev_t), void *env);
bool_t ev_blocked(struct ev_channel *chan);
void ev_set_blocked(struct ev_channel *chan, bool_t);
void ev_send(struct ev_channel *release, struct ev_channel *chan, ev_t ev);
void ev_handled(ev_t ev);
void ev_released(ev_t ev);
void ev_release(ev_t ev);
void ev_channel_release(struct ev_channel *chan);

struct chunk {
	int size;
	char *data;
};

struct socket_info *socket_info(void);

struct ev_channel *sink_create(void);
struct ev_channel *print_create(char *descr, bool_t binary);
struct ev_channel *tcp_output_create(struct socket_info *);
bool_t tcp_server_create(unsigned short port, struct ev_channel *dst);
struct socket_info *tcp_client_create(char *addr);
bool_t tcp_input_create(struct socket_info *, struct ev_channel *dst);
struct socket_info *udp_create(unsigned short port);
struct ev_channel *udp_output_create(struct socket_info *);
bool_t udp_input_create(struct socket_info *, struct ev_channel *dst);

void map_release(void *map);
void do_free(void *p);

#ifdef __cplusplus
}
#endif 

#endif /* __SHARED_H__ */
