/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* Off-line version of global scheduler to be used as a "black box". */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include "oslib.h"
#include "mmsched.h"

extern void mmsched_init(int);
extern char *mmsched_enter(void *env,
			   void (*dump_assignments)(void *env, char *addr,
						    char *assignment),
			   char *user, char *spec);
extern char *mmsched_assignments(int *len, int do_graph);

char *file_raw_read(char *file, int *len);

static void print_assignment(void *ign, char *addr, char *assignment){
/*    printf("%s\n", assignment); */
}

static void log_output(void *env, char *s){
  FILE *out = env;
  if (s != 0) {
    fprintf(out, "%s\n", s);
    fflush(out);
  }
}

/* for parsing network configurations */
extern FILE *yyin;
extern int yyparse();
#ifdef YYDEBUG
extern int yydebug;
#endif

extern int quiet;

int main(int argc, char **argv){
  int config = 3;
  char *configFile = "";
  char *progName = argv[0];
  char *output_format = "xml";
  struct log_sub *log;
  char ch;
  int idx;

  /*
   * Usage: sched [-c config | -f netfile] [-F format] userspec1 [userspec2 ..]
   */
  while ((ch = getopt(argc, argv, "p:u:f:F:c:?")) != EOF) 
    switch (ch) {
    case 'c':
      config = atoi(optarg);
      break;
    case 'f':
      configFile = optarg;
      break;
    case 'F':
      output_format = optarg;
      break;
    case 'p':
      parameters.nextport = atoi(optarg);
      if (parameters.nextport < LOW_PORT || parameters.nextport > HIGH_PORT) {
	fprintf(stderr,"-p must have arg in range %d-%d\n",
		LOW_PORT, HIGH_PORT);
	return 1;
      }
      break;
    case '?':
    default:
      fprintf(stderr, 
	      "usage: %s [-c config | -f BRITE-config-file] "
	      "       [-F xml | -F dot] userspec1 [userspec2] ... \n",
	      progName);
      fprintf(stderr,"  -c config	"
	             "network configuration number (def=%d)\n",config);
      fprintf(stderr,"  -f BRITE network description file (default=%s)\n",
	      configFile);
      fprintf(stderr,"  -F output format (either xml or dot, default=%s)\n",
	      output_format);
      return 1;
    }

  /* initialization */
  setbuf(stdout, 0);
  tst_init();
  log = log_subscribe(100, log_output, stderr);
  quiet = 1;

  /* read in user-specified network configuration */
  if (configFile[0] != '\0') {
    yyin = fopen(configFile,"r");
    if (yyin == NULL) {
      fprintf(stderr,"Could not open file %s for reading\n",configFile);
      exit(1);
    }
#ifdef YYDEBUG
    yydebug = 1;
#endif
    if (yyparse()) {
      fprintf(stderr,"Parsing failed!\n");
      exit(1);
    }
    mmsched_init(-1);
  }
  /* pre-defined configuration */
  else
    mmsched_init(config);

  /* read in user specifications */
  idx = optind;
  while (idx < argc) {
    char buf[100];
    char *data;
    int len;
    data = file_raw_read(argv[idx], &len);
    sprintf(buf,"user%d",idx);
    if (idx < (argc-1))
      mmsched_enter(NULL, NULL, buf, data);
    else {
/*       quiet = 0; */
      mmsched_enter(NULL, print_assignment, buf, data);
    }
    free(data);
    idx++;
  }

  log_unsubscribe(log);

  { char *assignments;
    assignments = mmsched_assignments(&idx, strcmp(output_format,"xml") != 0);
    printf("%s\n",assignments); 
    free(assignments); 
  }

  return 0;
}
