/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* Ternary string search tree implementation.  It maps a string key to a
 * void *.  It works basically as follows:  each node corresponds to a
 * prefix of some key (which is the entire key if we're at a leaf).  The
 * split character, if not 0, is the first character beyond this prefix.
 * In that case, suffixes that start with this character are in the middle
 * subtree, suffixes that start with smaller characters are in the left
 * subtree, while ones that start with a higher character in the right subtree.
 * In the case of a leaf node, the split character is 0.  The node then
 * contains a copy of the entire key, the offset into this key corresponding
 * to the current node, as well as the void * pointer.
 *
 * This file contains both a recursive and a non-recursive implementation.
 * The recursive implementation is easier to understand, but the non-recursive
 * one is faster.  The non-recursive implementation needs to maintain additional
 * information in each node.  In particular, it maintains a pointer to the
 * parent node, as well as, in the case of traveling through the tree, what
 * path it is currently following.
 *
 * The interface is basically as follows:
 *
 *	void **tst_find(map_t *map, char *key, int create)
 *		This function looks up an entry in the map, returning 0 if it can find it
 *		or otherwise a pointer to the void * pointer associated with the entry.
 *		If "create" is non-zero, a new entry is automatically created if it did
 *		not previously exist.  The void * pointer is null initialized in that case.
 *
 *	void tst_remove(map_t *map, char *key, void (*release)(void *env, void *p), void *env)
 *		Remove an entry from the map.  The corresponding release function is invoked if
 *		the entry existed and the pointer p was non-null.  The function is invoked with
 *		the specified env pointer and p.  Use "mem_release" if the data pointed to was
 *		allocated using malloc (use env = 0).
 *
 *	void tst_travel(map_t *map, int (*upcall)(void *env, char *key, void **pp), void *env)
 *		Travel to the entire tree, invoking upcall for each key.  The upcall is invoked with
 *		the specified env pointer, the key, and a pointer to the void * pointer.  Iff the
 *		upcall returns 0, the node is removed.
 *
 *	void tst_release(map_t *map, void (*release)(void *env, void *p), void *env)
 *		Release an entire tree.  Invokes "release" for every non-zero void * pointer.
 *
 *	char *tst_unparse(map_t map)
 *		Returns a string representation of trees of C string pointers.  The result can be
 *		released using free().
 *
 *	char *tst_parse(map_t *map, char *s)
 *		This function can be used to parse the output from tst_parse and transform it back
 *		into a map.  The resulting map can be released using tst_releave(&map, mem_release, 0);
 *
 * The parsing format deserves a few words here as well.  Basically, the format is
 * "a=x,b=y,c=x,...".  However, if the keys a, b, c, or the values, x, y, z have complicated
 * contents (for example, they contain an '=' sign or a ','), we need to do something special.
 * The simplest solution would have been to escape special characters, but that leads to a lot
 * of unnecessary copying.  For this reason, we have two different quoting mechanisms.  For
 * strings consisting of entirely non-funny characters, there is no quoting whatsoever.  The
 * funny characters are '=', ',', ' ', TAB, CR, LF, '{', and '}'.  There is one exception:
 * strings that start with the character '[' have to be quoted.
 */

#include "oslib.h"
#include <string.h>

/* See description in comments above.
 */
struct map_node {
	int split;
#ifndef RECURSION
	map_t parent, *pn;	/* only used for traversal */
	enum dir_t { LEFT, MIDDLE, RIGHT, DONE } dir;
#endif
	union {
		struct {	/* split != 0 */
			map_t left, middle, right;
		} internal;
		struct {	/* split == 0 */
			void *info;
			char *key;
			int offset;
		} tail;
	} u;
};

/* Remove an entry from the tree.
 */
void tst_remove(map_t *pn, const char *s, void (*release)(void *, void *), void *env){
	map_t map;

	/* Stop if we go beyond a leaf.
	 */
	if ((map = *pn) == 0) {
		return;
	}

	/* See if this is a leaf node.  If so, see if this is the key we were
	 * looking for (by comparing suffixes at this point).  If so, release
	 * the node.  And if the pointer is not null, invoke the release function.
	 */
	if (map->split == 0) {
		if (strcmp(&map->u.tail.key[map->u.tail.offset], s) == 0) {
			if (release != 0 && map->u.tail.info != 0)
				(*release)(env, map->u.tail.info);
			free(map->u.tail.key);
			free((char *) map);
			*pn = 0;
		}
		return;
	}

	/* This is an internal node.  Recursively travel to the three child trees.
	 */
	if ((*s & 0xFF) < map->split) {
		tst_remove(&map->u.internal.left, s, release, env);
	}
	else if ((*s & 0xFF) > map->split) {
		tst_remove(&map->u.internal.right, s, release, env);
	}
	else {
		tst_remove(&map->u.internal.middle, s + 1, release, env);
	}

	/* Clean up this node if it no longer has any children.
	 */
	if (map->u.internal.left == 0 && map->u.internal.middle == 0 &&
					map->u.internal.right == 0) {
		free((char *) map);
		*pn = 0;
	}
}

/* Find a node in the tree.  Returns 0 if it doesn't exist.
 */
void **tst_lookup(map_t map, const char *s){
	/* Find your way starting at the root node towards the leaf.
	 */
	while (map != 0) {
		/* If we hit a leaf, see if it's the right one.  If not,
		 * return 0.
		 */
		if (map->split == 0) {
			if (strcmp(&map->u.tail.key[map->u.tail.offset], s) == 0) {
				return &map->u.tail.info;
			}
			return 0;
		}

		/* Choose the correct branch.
		 */
		if ((*s & 0xFF) < map->split) {
			map = map->u.internal.left;
		}
		else if ((*s & 0xFF) > map->split) {
			map = map->u.internal.right;
		}
		else {
			map = map->u.internal.middle;
			s++;
		}
	}

	/* Didn't find it.
	 */
	return 0;
}

/* Look up a node, creating it if not existent.  A pointer to
 * the void * pointer is returned.
 */
void **tst_insert(map_t *pn, const char *key){
	map_t map, n;
	int offset = 0;		/* offset into the key */

	/* Create a node if it isn't there yet.
	 */
	for (;;) {
		/* If there is no node, create one.
		 */
		if ((map = *pn) == 0) {
			map = (map_t) calloc(sizeof(*map), 1);
			*pn = map;
			map->u.tail.key = mem_string_copy(key,
				strlen(key + offset) + offset);
			map->u.tail.offset = offset;
			return &map->u.tail.info;
		}

		/* If end is reached, test for equality.  If not the same,
		 * but they start with the same character (can't be 0),
		 * insert a new node so we're not at the end any more.
		 */
		if (map->split == 0) {
			if (strcmp(&map->u.tail.key[map->u.tail.offset],
							&key[offset]) == 0)
				return &map->u.tail.info;

			/* Insert a new node.
			 */
			n = (map_t) calloc(sizeof(*map), 1);
			if (map->u.tail.key[map->u.tail.offset] != 0) {
				n->split = map->u.tail.key[
						map->u.tail.offset++] & 0xFF;
				n->u.internal.middle = map;
			}
			else {
				n->split = key[offset] & 0xFF;
				n->u.internal.left = map;
			}
			map = *pn = n;
		}

		/* Travel to the correct branch.
		 */
		if ((key[offset] & 0xFF) < map->split)
			pn = &map->u.internal.left;
		else if ((key[offset] & 0xFF) > map->split)
			pn = &map->u.internal.right;
		else {
			pn = &map->u.internal.middle;
			offset++;
		}
	}
}

#ifdef RECURSION

/* Travel through the entire tree, invoking the given upcall everywhere.
 * If the upcall returns 0, remove the node.
 */
void tst_travel(map_t *pn, int (*upcall)(void *, char *, void **), void *env){
	map_t map;

	if ((map = *pn) == 0)
		return;

	if (map->split == 0) {
		if (!(*upcall)(env, map->u.tail.key, &map->u.tail.info)) {
			free(map->u.tail.key);
			free((char *) map);
			*pn = 0;
		}
	}
	else {
		tst_travel(&map->u.internal.left, upcall, env);
		tst_travel(&map->u.internal.middle, upcall, env);
		tst_travel(&map->u.internal.right, upcall, env);

		/* Clean up if there's nothing left.
		 */
		if (map->u.internal.left == 0 && map->u.internal.middle == 0
					&& map->u.internal.right == 0) {
			free((char *) map);
			*pn = 0;
		}
	}
}

#else /* RECURSION */

/* Travel through the entire tree, invoking the given upcall everywhere.
 * If the upcall returns 0, remove the node.
 */
void tst_travel(map_t *pn, int (*upcall)(void *, char *, void **), void *env){
	map_t map, parent = 0;

    start:
	if ((map = *pn) == 0)
		goto restore;

	if (map->split == 0) {
		if (!(*upcall)(env, map->u.tail.key, &map->u.tail.info)) {
			free(map->u.tail.key);
			free((char *) map);
			*pn = 0;
		}
		goto restore;
	}

	/* Save state in the map node.
	 */
	map->pn = pn;
	map->parent = parent;

	/* Visit all children.
	 */
	for (map->dir = LEFT;; map->dir = map->dir + 1) {
		switch (map->dir) {
		case LEFT:   pn = &map->u.internal.left;   parent = map;
			     goto start;
		case MIDDLE: pn = &map->u.internal.middle; parent = map;
			     goto start;
		case RIGHT:  pn = &map->u.internal.right;  parent = map;
			     goto start;
		case DONE:   pn = map->pn;		   parent = map->parent;
		}

		/* Clean up if there's nothing left.
		 */
		if (map->u.internal.left == 0 && map->u.internal.middle == 0
					&& map->u.internal.right == 0) {
			*map->pn = 0;
			free((char *) map);
		}

    restore:
		/* Restore state of parent level.
		 */
		if ((map = parent) == 0)
			return;
	}
}

#endif /* RECURSION */

static char funny[256];

static void funny_init(void){
	funny[0] = 1;
	funny['='] = funny[','] = 1;
	funny['{'] = funny['}'] = 1;
	funny[' '] = funny['\t'] = 1;
	funny['\n'] = funny['\r'] = 1;
}

static bool_t is_funny(const char *s){
	if (*s == '[')
		return true;
	while (*s != 0)
		if (funny[*s++ & 0xFF])
			return true;
	return false;
}

static bool_t well_nested(const char *s){
	int level = 0;

	while (*s != 0)
		switch (*s++) {
		case '{':
			level++;
			break;
		case '}':
			if (--level < 0)
				return false;
			break;
		}
	return level == 0;
}

/* Add the string to the given channel.  There are three formats.  If it's
 * a very simple string (no spaces and such), just output it.  If it has
 * spaces, but the brackets are well-nested, quote it with a pair of extra
 * brackets.
 */
void tst_putstr(struct print_channel *pc, const char *s){
	if (!is_funny(s))
		pc_puts(pc, s, strlen(s));
	else if (well_nested(s)) {
		pc_putchar(pc, '{');
		pc_puts(pc, s, strlen(s));
		pc_putchar(pc, '}');
	}
	else {
		pc_putchar(pc, '[');
		while (*s != 0) {
			switch (*s) {
			case '%':
				pc_putchar(pc, '%'); pc_putchar(pc, '%');
				break;
			case '{':
				pc_putchar(pc, '%'); pc_putchar(pc, '(');
				break;
			case '}':
				pc_putchar(pc, '%'); pc_putchar(pc, ')');
				break;
			case ']':
				pc_putchar(pc, '%'); pc_putchar(pc, ']');
				break;
			default:
				pc_putchar(pc, *s);
			}
			s++;
		}
		pc_putchar(pc, ']');
	}
}

/* Opposite of tst_putstr().
 */
char *tst_getstr(char **sp, int *len){
	char *s = *sp, *p;
	int level;
	struct print_channel *pc;

	switch (*s) {
	case 0:
		*len = 0;
		return s;
	case '{':
		for (p = ++s, level = 1; level != 0; p++)
			switch (*p) {
			case 0:
				err_warning("tst_getstr: { not well-nested");
				return 0;
			case '{':
				level++;
				break;
			case '}':
				if (--level == 0) {
					*sp = p + 1;
					*len = p - s;
					return s;
				}
			}
	case '[':
		pc = mc_open();
		while (*++s != ']')
			switch (*s) {
			case 0:
				err_warning("tst_getstr: [ not well-nested");
				return 0;
			case '%':
				switch (*++s) {
				case 0:   err_warning("tst_getstr: bad esc");
					  mc_release(pc); return 0;
				case '%': pc_putchar(pc, '%'); break;
				case '(': pc_putchar(pc, '{'); break;
				case ')': pc_putchar(pc, '}'); break;
				case ']': pc_putchar(pc, ']'); break;
				default:  pc_putchar(pc, *s); break;
				}
				break;
			default:
				pc_putchar(pc, *s);
			}
		*sp = s + 1;
		*len = -1;
		return mc_close(pc);
	default:
		for (p = s; !funny[*p & 0xFF]; p++)
			;
		*sp = p;
		*len = p - s;
		return s;
	}
}

/* Parse *info, invoke upcall with the given environment pointer, as well as
 * the category, the (not necessarily null-terminated) value, and the size of
 * the value.  *info is updated.  Returns whether well-formed or not.
 */
bool_t tst_do_parse(char **info, void (*upcall)(void *env, char *category,
					char *value, int size), void *env) {
	char *p, *category, *value, sv[16];
	int clen, vlen;

	if (funny[0] != 1)
		err_fatal("tst_do_parse: not initialized");
	for (p = *info;; p++) {
		/* Skip blanks.
		 */
		while (is_blank(*p))
			p++;
		if (*p == 0)
			break;

		/* Find the end of the category and copy the entire thing.  clen
		 * is set to -1 if category was allocated rather than pointing to
		 * actual data.
		 */
		category = tst_getstr(&p, &clen);
		if (category == 0 || clen == 0) {
			err_warning("tst_do_parse: bad category");
			return false;
		}
		if (clen > 0) {
			if (clen < 16) {	/* copy into local buf iff small enough. */
				memcpy(sv, category, clen);
				sv[clen] = 0;
				category = sv;
			}
			else
				category = mem_string_copy(category, clen);
		}

		/* Skip blanks and make sure it ends in an equals sign.
		 */
		while (*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r')
			p++;
		if (*p != '=') {
	fail:
			if (clen < 0 || clen >= 16)
				free(category);
			return false;
		}
		p++;

		/* Skip blanks.
		 */
		while (*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r')
			p++;

		/* Get the value.
		 */
		value = tst_getstr(&p, &vlen);
		if (value == 0) {
			err_warning("tst_do_parse: bad value");
			goto fail;
		}

		/* Invoke the upcall.
		 */
		if (upcall != 0)
			(*upcall)(env, category, value,
					vlen < 0 ? strlen(value) : vlen);
		if (clen < 0 || clen >= 16)
			free(category);
		if (vlen < 0)
			free(value);

		/* Break if there is no more stuff here.
		 */
		if (*p != ',')
			break;
	}

	/* Update info pointer and return.
	 */
	*info = p;
	return true;
}

/* Release the entire ternary tree.
 */
void tst_release(map_t *pn, void (*release)(void *, void *), void *env){
	map_t map;

	if ((map = *pn) != 0) {
		if (map->split == 0) {
			if (release != 0 && map->u.tail.info != 0)
				(*release)(env, map->u.tail.info);
			free((char *) map->u.tail.key);
		}
		else {
			tst_release((map_t *) &map->u.internal.left,
							release, env);
			tst_release((map_t *) &map->u.internal.middle,
							release, env);
			tst_release((map_t *) &map->u.internal.right,
							release, env);
		}
		free((char *) map);
		*pn = 0;
	}
}

static void tst_parse_upcall(void *env, char *cat, char *val, int size){
	map_t *map = env;
	void **q;

	/* See if it has changed, and if so, overwrite it.
	 */
	q = tst_insert(map, cat);
	if (*q == 0 || strlen(*q) != (unsigned int)size || strncmp(*q, val, size) != 0) {
		if (*q != 0)
			free(*q);
		*q = mem_string_copy(val, size);
	}
}

/* Info is of the form "a=b,c=d,...".  Return false if input's bad.
 * Values may be bracketed by "{" and "}" or [].  Unlike tst_parse,
 * the data is added to the given map.
 */
bool_t tst_parse_append(map_t *map, char *info){
	if (!tst_do_parse(&info, tst_parse_upcall, map)) {
		tst_release(map, mem_release, 0);
		return false;
	}
	return true;
}

/* Info is of the form "a=b,c=d,...".  Return false if input's bad.
 * Values may be bracketed by "{" and "}" or [].
 */
bool_t tst_parse(map_t *map, char *info){
	*map = 0;
	return tst_parse_append(map, info);
}

struct info {
	struct print_channel *pc;
	int first;
};

static int info_entry(void *env, char *category, void **value){
	struct info *info = env;

	if (!info->first)
		pc_putchar(info->pc, ',');
	info->first = 0;
	if (*category == 0)
		err_warning("info_entry: empty category");
	tst_putstr(info->pc, category);
	pc_putchar(info->pc, '=');
	if (*value != 0)
		tst_putstr(info->pc, *value);

	return 1;
}

void tst_print(struct print_channel *pc, map_t tst){
	struct info info;

	if (funny[0] != 1)
		err_fatal("tst_print: not initialized");
	info.pc = pc;
	info.first = 1;
	tst_travel(&tst, info_entry, &info);
}

char *tst_unparse(map_t tst){
	struct print_channel *pc = mc_open();

	tst_print(pc, tst);
	return mc_close(pc);
}

bool_t tst_read(char *file, map_t *map){
	char *contents;

	if ((contents = file_read(file)) == 0)
		return false;
	*map = 0;
	tst_parse(map, contents);
	free(contents);
	return true;
}

void tst_write(char *file, map_t map){
	char *contents = tst_unparse(map);

	file_write(file, contents);
	free(contents);
}

void tst_init(void){
	funny_init();
}
