/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* This module processes evt_client events that announce new TCP clients, and
 * starts new activities that read the TCP input and parse it using an HTTP parser,
 * as well as an activity to write the TCP output.  The parsed requests are processed
 * by this module, however, as a single activity.  This is because multiple requests
 * share the same state, and needs to be synchronized.
 */
#include "oslib.h"
#include "http.h"
#include <string.h>

extern evt_t evt_http;

static int evt_http_start;
static int evt_http_map;
static int evt_http_size;
static int evt_data_data;
static int evt_error_descr;
static int evt_client_socket;

/* One of these is allocated per registered user.
 */
struct websvr_user {
	char *passwd;
	char *redirect;		/* set if redirecting */
	enum {
		USER_IDLE,		/* no activity */
		USER_RECV,		/* user is awaiting messages */
		USER_SEND		/* a message has been buffered for this user */
	} tag;
	union {
		struct {	/* tag == USER_RECV */
			struct websvr_client *clients;		/* user's client records */
		} recv;
		struct {	/* tag == USER_SEND */
			char *type;							/* document type */
			char *body;							/* buffered data */
			int size;							/* size of this data */
		} send;
	} u;
};

/* This is the "global" data of the web server.
 */
struct websvr_act {
	int refcnt;
	struct activity *activity;
	struct ev_channel *in;
	struct ev_channel *release;
	char *db;
	map_t users;					/* of struct websvr_user records */
};

/* One of these is allocated per connected web client.
 */
struct websvr_client {
	struct websvr_client *next;		/* linked list pointer */
	struct websvr_act *activity;	/* the web server activity */
	char *cmd;						/* the command that we're handling */
	char *uri;						/* uri in command */
	char *version;					/* HTTP version */
	map_t map;						/* HTTP header fields */
	int expected;					/* how much more body data expected */
	char *body;						/* buffered body data */
	int size;						/* size of this data */
	struct ev_channel *in;			/* input channel for this client */
	struct ev_channel *output;		/* output channel for this client */
};

/* Invoked for every user record when the websvr activity is closed.
 * This can only happen if there's no activity on any user.
 */
static void websvr_user_release(void *env, void *v){
	struct websvr_user *wsu = v;

	if (wsu->tag != USER_IDLE) {
		err_warning("websvr_user_release: non-idle user");
	}
	free(wsu->passwd);
	free((char *) wsu);
}

/* A reference to the websvr activity disappeared.
 */
static void websvr_release(struct websvr_act *wsa){
	if (--wsa->refcnt > 0) {
		return;
	}
	tst_release(&wsa->users, websvr_user_release, wsa);
	ev_channel_release(wsa->release);
	act_release(wsa->activity);
	free((char *) wsa);
}

/* A normal 200 type response.
 *
 * TODO.  Should drop connection if output is blocked.
 */
static void websvr_respond(struct websvr_client *wsc,
			char *code, char *data, int size, char *type){
	struct websvr_act *wsa = wsc->activity;
	struct print_channel *pc;
	ev_t ev;
	struct chunk *chunk;
	void **p;
	char *conn;
	double vn;
	int hdrlen;
	char *hdr;

	/* If the size isn't specified, assume it's null-terminated data.
	 */
	if (size == -1) {
		size = strlen(data);
	}

	/* Create the response header.
	 */
	pc = mc_open();
	pc_print(pc, "%s %s\r\n", wsc->version, code);
	pc_print(pc, "Server: Robbert's Web Server 1.0\r\n");
	pc_print(pc, "Content-Type: %s\r\n", type == 0 ? "text/html" : type);
	pc_print(pc, "Content-Length: %d\r\n", size);
	pc_print(pc, "\r\n");
	hdrlen = mc_size(pc);
	hdr = mc_close(pc);

	/* Create an output event.
	 */
	chunk = (struct chunk *) calloc(1, sizeof(*chunk));
	chunk->size = hdrlen + size;
	chunk->data = malloc(chunk->size);
	memcpy(chunk->data, hdr, hdrlen);
	memcpy(&chunk->data[hdrlen], data, size);
	ev = ev_create(evt_data);
	ev_cast(ev, evt_data)->chunk = chunk;
	free(hdr);

	/* Send the output event.
	 */
	wsa->refcnt++;
	ev_send(wsa->release, wsc->output, ev);

	/* See if we need to close the connection.  If version HTTP/1.0 or before, close it
	 * unless "Connection: Keep-Alive" has been specified.  If version HTTP/1.1 or later,
	 * close it only if "Connection: close" has been specified.
	 */
	p = tst_find(&wsc->map, "connection", 0);
	conn = p == 0 ? 0 : *p;
	if (sscanf(wsc->version, "HTTP/%lf", &vn) != 1) {
		err_warning("websvr_respond: bad version spec '%s'", wsc->version);
	}
	else if ((vn <= 1.0 && (conn == 0 || (*conn != 'k' && *conn != 'K'))) ||
			(vn > 1.0 && conn != 0 && (*conn == 'c' || *conn == 'C'))) {
		err_warning("websvr_respond: close the connection (%f,%s)", vn, wsc->version);
		wsa->refcnt++;
		ev_send(wsa->release, wsc->output, ev_create(evt_close));
	}

	/* Clean up.
	 *
	 * TODO.  Check to see if no other request can come in and overwrite.
	 */
	if (wsc->size > 0) {
		free(wsc->body); wsc->body = 0;
		wsc->size = 0;
	}
	wsc->expected = 0;
	free(wsc->cmd);		wsc->cmd = 0;
	free(wsc->uri);		wsc->uri = 0;
	free(wsc->version);	wsc->version = 0;
	tst_release(&wsc->map, mem_release, 0);
}

/* A 301 redirect response
 *
 * TODO.  Should drop connection if output is blocked.
 */
static void websvr_redirect(struct websvr_client *wsc,
				struct websvr_user *wsu){
	struct websvr_act *wsa = wsc->activity;
	struct print_channel *pc = mc_open();
	ev_t ev;
	struct chunk *chunk;
	void **p;
	char *conn;
	double vn;
	int hdrlen, size;
	char *hdr, *data;
	char *location = mem_print("http://%s%s", wsu->redirect, wsc->uri);

printf("REDIRECT TO '%s'\n", location);

	/* Create the body of the message.
	 */
	pc_print(pc, "<html><body><p>\n");
	pc_print(pc, "This receiver moved to <a href=\"%s\">%s</a>\n",
						location, location);
	pc_print(pc, "</p></body></html>\n");
	size = mc_size(pc);
	data = mc_close(pc);

	/* Create the response header.
	 */
	pc = mc_open();
	pc_print(pc, "%s 301 Moved Permanently\r\n", wsc->version);
	pc_print(pc, "Server: Robbert's Web Server 1.0\r\n");
	pc_print(pc, "Location: %s\r\n", location);
	pc_print(pc, "Content-Type: %s\r\n", "text/html");
	pc_print(pc, "Content-Length: %d\r\n", size);
	pc_print(pc, "\r\n");
	hdrlen = mc_size(pc);
	hdr = mc_close(pc);

	free(location);

	/* Create an output event.
	 */
	chunk = (struct chunk *) calloc(1, sizeof(*chunk));
	chunk->size = hdrlen + size;
	chunk->data = malloc(chunk->size);
	memcpy(chunk->data, hdr, hdrlen);
	memcpy(&chunk->data[hdrlen], data, size);
	ev = ev_create(evt_data);
	ev_cast(ev, evt_data)->chunk = chunk;
	free(hdr);

	/* Send the output event.
	 */
	wsa->refcnt++;
	ev_send(wsa->release, wsc->output, ev);

	/* See if we need to close the connection.  If version HTTP/1.0 or before, close it
	 * unless "Connection: Keep-Alive" has been specified.  If version HTTP/1.1 or later,
	 * close it only if "Connection: close" has been specified.
	 */
	p = tst_find(&wsc->map, "connection", 0);
	conn = p == 0 ? 0 : *p;
	if (sscanf(wsc->version, "HTTP/%lf", &vn) != 1) {
		err_warning("websvr_respond: bad version spec '%s'", wsc->version);
	}
	else if ((vn <= 1.0 && (conn == 0 || (*conn != 'k' && *conn != 'K'))) ||
			(vn > 1.0 && conn != 0 && (*conn == 'c' || *conn == 'C'))) {
		err_warning("websvr_respond: close the connection (%f,%s)", vn, wsc->version);
		wsa->refcnt++;
		ev_send(wsa->release, wsc->output, ev_create(evt_close));
	}

	/* Clean up.
	 *
	 * TODO.  Check to see if no other request can come in and overwrite.
	 */
	if (wsc->size > 0) {
		free(wsc->body); wsc->body = 0;
		wsc->size = 0;
	}
	wsc->expected = 0;
	free(wsc->cmd);		wsc->cmd = 0;
	free(wsc->uri);		wsc->uri = 0;
	free(wsc->version);	wsc->version = 0;
	tst_release(&wsc->map, mem_release, 0);
}

/* Convert a hex digit to an integer.
 */
static bool_t get_hex(char c, int *result){
	if ('0' <= c && c <= '9') {
		*result =  c - '0';
		return true;
	}
	if ('a' <= c && c <= 'z') {
		*result = c - 'a' + 10;
		return true;
	}
	if ('A' <= c && c <= 'Z') {
		*result = c - 'A' + 10;
		return true;
	}
	return false;
}

/* Invoked for every entry in a url-encoded string.  Adds the
 * entry to the given map.
 */
static void **url_add(struct print_channel *pc, map_t *map){
	char *name = mc_close(pc);
	void **p;

	if (*name == 0) {
		err_warning("url_add: empty entry");
		free(name);
		return 0;
	}
	p = tst_find(map, name, 1);
	free(name);
	if (*p != 0) {
		err_warning("url_add: duplicate entry");
		free(*p);
		*p = 0;
	}
	return p;
}

/* This function decodes an HTTP url-encoded string and returns a map.
 */
static map_t url_decode(char *url, int size){
	struct print_channel *pc = 0;
	int i, c, d;
	map_t map = 0;
	void **p = 0;

	for (i = 0; i < size; i++) {
		c = url[i];
		if (pc == 0) {
			pc = mc_open();
		}
		switch (c) {
		case '+':
			pc_putchar(pc, ' ');
			break;
		case '%':
			if (i == size - 1) {
				err_warning("url_decode: no hex");
				break;
			}
			if (!get_hex(url[++i], &c)) {
				err_warning("url_decode: hex character expected");
				break;
			}
			if (i == size - 1) {
				err_warning("url_decode: short hex");
			}
			else if (!get_hex(url[++i], &d)) {
				err_warning("url_decode: another hex character expected");
			}
			else {
				c = (c << 4) | d;
			}
			pc_putchar(pc, c);
			break;
		case '=':
			if (p != 0) {
				err_warning("url_decode: unexpected =");
				pc_putchar(pc, '=');
			}
			else {
				p = url_add(pc, &map);
				pc = mc_open();
			}
			break;
		case '&':
			if (p == 0) {
				(void) url_add(pc, &map);
			}
			else {
				*p = mc_close(pc);
				p = 0;
			}
			pc = mc_open();
			break;
		default:
			pc_putchar(pc, c);
		}
	}

	if (pc != 0) {
		if (p == 0) {
			(void) url_add(pc, &map);
		}
		else {
			*p = mc_close(pc);
		}
	}

	return map;
}

/* Invoked for every user from websvr_store_users.  Adds one
 * user entry to the file.
 */
static int store_users_visit(void *env, char *id, void **v){
	struct print_channel *pc = env;
	struct websvr_user *wsu = *v;

	pc_print(pc, "%s^%s^%s\n", id, wsu->passwd, wsu->redirect);
	return 1;
}

/* Store all the users and their password on a file.
 */
static void websvr_store_users(struct websvr_act *wsa){
	struct print_channel *pc = mc_open();
	char *data;

	tst_travel(&wsa->users, store_users_visit, pc);
	data = mc_close(pc);
	if (!file_write(wsa->db, data)) {
		err_fatal("websvr_store_users: file_write failed");
	}
	free(data);
}

/* Read the users database.  Mostly parsing.
 */
static void websvr_read_users(struct websvr_act *wsa){
	char *data, *ptr, *eoln, *sep1, *sep2, *id;
	int line, len;
	void **p;
	struct websvr_user *wsu;

	/* Make sure it's initialized properly.
	 */
	wsa->users = 0;

	/* Read the entire file.
	 */
	if ((data = file_read(wsa->db)) == 0) {
		return;
	}
	len = strlen(data);

	/* Parse each line.
	 */
	for (ptr = data, line = 1; *ptr != 0; line++) {
		/* Skip empty lines.
		 */
		if (*ptr == '\n') {
			ptr++;
			len--;
			continue;
		}

		/* Find the end of line, or eof.
		 */
		if ((eoln = memchr(ptr, '\n', len)) == 0) {
			eoln = &ptr[len];
		}

		/* Then find the separators.
		 */
		if ((sep1 = memchr(ptr, '^', eoln - ptr)) == 0) {
			err_fatal("websvr_read_users: no separator in line %d", line);
		}
		if ((sep2 = memchr(sep1 + 1, '^', eoln - (sep1 + 1))) == 0) {
			err_fatal("websvr_read_users: no 2nd separator in line %d", line);
		}

		/* Get the id.
		 */
		id = mem_string_copy(ptr, sep1 - ptr);

		/* Create in the internal database.
		 */
		p = tst_find(&wsa->users, id, 1);
		if (*p != 0) {
			err_fatal("websvr_read_users: duplicate user '%s' in line %d", id, line);
		}
		free(id);
		
		/* Allocate the user record.
		 */
		wsu = (struct websvr_user *) calloc(1, sizeof(*wsu));
		*p = wsu;

		/* Get the password and redirect string.
		 */
		sep1++;
		wsu->passwd = mem_string_copy(sep1, sep2 - sep1);
		sep2++;
		if (eoln[-1] == '\r') {
			wsu->redirect = mem_string_copy(sep2, eoln - sep2 - 1);
		}
		else {
			wsu->redirect = mem_string_copy(sep2, eoln - sep2);
		}

		/* Move pointer to end of line.
		 */
		len -= eoln - ptr;
		ptr = eoln;
	}

	free(data);
}

/* Return whether any of the characters in 'bad' are in 's'.
 */
static bool_t has_badchars(char *s, char *bad){
	while (*bad != 0) {
		if (strchr(s, *bad) != 0) {
			return true;
		}
		bad++;
	}
	return false;
}

/* An HTTP POST "register.cgi" request was received to register a new user.
 */
static void websvr_register(struct websvr_client *wsc){
	struct websvr_act *wsa = wsc->activity;
	map_t map = url_decode(wsc->body, wsc->size);
	void **p;
	char *id, *passwd;
	struct websvr_user *wsu;

	/* Get the parameters.
	 */
	p = tst_find(&map, "ID", 0);
	if (p == 0 || (id = *p) == 0 || *id == 0) {
		websvr_respond(wsc, "400 no ID specified", "no ID specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	if (has_badchars(id, " \t\r\n:")) {
		websvr_respond(wsc, "400 bad ID",
				"bad characters in ID --- use another id", -1, 0);
		
		tst_release(&map, mem_release, 0);
		return;
	}
	p = tst_find(&map, "passwd", 0);
	if (p == 0 || (passwd = *p) == 0 || *passwd == 0) {
		websvr_respond(wsc, "400 no password", "no password specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	if (has_badchars(id, "\r\n")) {
		websvr_respond(wsc, "400 bad password",
				"bad characters in password --- try again", -1, 0);
		
		tst_release(&map, mem_release, 0);
		return;
	}

	/* Create a new user entry.  Make sure it didn't exist already.
	 */
	p = tst_find(&wsa->users, id, 1);
	if (*p != 0) {
		websvr_respond(wsc, "400 ID already taken",
				"that ID has already been taken.  Try again...", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}

	/* Allocate a new user record.
	 */
	wsu = (struct websvr_user *) calloc(1, sizeof(*wsu));
	wsu->passwd = mem_string_copy(passwd, strlen(passwd));
	wsu->redirect = mem_print("");
	*p = wsu;

	/* Store on file.
	 */
	websvr_store_users(wsa);

	/* Done.
	 */
	websvr_respond(wsc, "200 OK", "registration successful", -1, 0);
	tst_release(&map, mem_release, 0);
}

/* An HTTP POST "recv.cgi" request was received to receive a message.
 */
static void websvr_recv(struct websvr_client *wsc){
	map_t map = url_decode(wsc->body, wsc->size);
	void **p;
	char *id, *passwd;
	struct websvr_user *wsu;

	/* Get the parameters.
	 */
	p = tst_find(&map, "ID", 0);
	if (p == 0 || (id = *p) == 0 || *id == 0) {
		websvr_respond(wsc, "400 no ID specified", "no ID specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	p = tst_find(&map, "passwd", 0);
	if (p == 0 || (passwd = *p) == 0 || *passwd == 0) {
		websvr_respond(wsc, "400 no password specified",
								"no password specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}

	/* Find the user entry.
	 */
	p = tst_find(&wsc->activity->users, id, 0);
	if (p == 0 || (wsu = *p) == 0) {
		err_warning("websvr_recv: no user '%s'", id);
		websvr_respond(wsc, "400 no such user", "no such user", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	if (strcmp(passwd, wsu->passwd) != 0) {
		websvr_respond(wsc, "400 bad password", "bad password", -1, 0);
		return;
	}

	switch (wsu->tag) {
	case USER_RECV:
		/* Somebody else was doing a receive already.  Add to list.
		 */
		wsc->next = wsu->u.recv.clients;
		wsu->u.recv.clients = wsc;
		break;
	case USER_IDLE:
		/* Save all the state, but don't respond yet.
		 */
		wsu->tag = USER_RECV;
		wsu->u.recv.clients = wsc;
		break;
	case USER_SEND:
		/* A message is already waiting.
		 */
		websvr_respond(wsc, "200 OK",
				wsu->u.send.body, wsu->u.send.size, wsu->u.send.type);
		free(wsu->u.send.body);
		free(wsu->u.send.type);
		wsu->tag = USER_IDLE;
	}
	tst_release(&map, mem_release, 0);
}

/* Send a message of the given document type to the identified user.
 */
static void websvr_do_send(struct websvr_client *wsc, char *id,
							char *type, char *body, int size){
	void **p;
	struct websvr_user *wsu;
	struct websvr_client *rcver;

	/* Find the user entry.
	 */
	p = tst_find(&wsc->activity->users, id, 0);
	if (p == 0 || (wsu = *p) == 0) {
		websvr_respond(wsc, "400 no such user", "no such user", -1, 0);
		return;
	}

	/* See if the user's being redirected.
	 */
	if (*wsu->redirect != 0) {
		websvr_redirect(wsc, wsu);
		return;
	}

	/* See what's up with this user.
	 */
	switch (wsu->tag) {
	case USER_SEND:
		/* A message is buffered already.  Overwrite.
		 */
		free(wsu->u.send.body);
		free(wsu->u.send.type);
		/* FALL THROUGH */
	case USER_IDLE:
		/* Buffer the message.
		 */
		wsu->tag = USER_SEND;
		wsu->u.send.type = mem_string_copy(type, strlen(type));
		wsu->u.send.body = malloc(size);
		memcpy(wsu->u.send.body, body, size);
		wsu->u.send.size = size;
		break;
	case USER_RECV:
		/* There are one or more receivers waiting already.
		 */
		while ((rcver = wsu->u.recv.clients) != 0) {
			websvr_respond(rcver, "200 OK", body, size, type);
			wsu->u.recv.clients = rcver->next;
		}
		wsu->tag = USER_IDLE;
		break;
	}
	websvr_respond(wsc, "200 OK", "thanks for sending", -1, 0);
}

/* An HTTP POST "send.cgi" request was received to send a message.
 */
static void websvr_send(struct websvr_client *wsc){
	map_t map = url_decode(wsc->body, wsc->size);
	void **p;
	char *id, *msg, *type;

	/* Get the parameters.
	 */
	p = tst_find(&map, "ID", 0);
	if (p == 0 || (id = *p) == 0 || *id == 0) {
		websvr_respond(wsc, "400 no ID", "no ID specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	p = tst_find(&map, "msg", 0);
	if (p == 0 || (msg = *p) == 0 || *msg == 0) {
		websvr_respond(wsc, "400 no message", "no message specified", -1, 0);
		tst_release(&map, mem_release, 0);
		return;
	}
	p = tst_find(&map, "type", 0);
	if (p == 0) {
		type = "text/plain";
	}
	else {
		type = *p;
	}

	/* This function takes care of the actual sending.
	 */
	websvr_do_send(wsc, id, type, msg, strlen(msg));
	tst_release(&map, mem_release, 0);
}

/* An HTTP POST "/direct/<id>" request was received to send a message to the given user.
 */
static void websvr_direct(struct websvr_client *wsc, char *id){
	void **p;
	char *type;

	/* Get the content type.
	 */
	p = tst_find(&wsc->map, "content-type", 0);
	if (p == 0) {
		type = "application/octetstream";
	}
	else {
		type = *p;
	}

	/* This function takes care of the actual sending.
	 */
	websvr_do_send(wsc, id, type, wsc->body, wsc->size);
}

/* It's an HTTP POST request.  Dispatch based on uri.
 */
static void websvr_post(struct websvr_client *wsc){
	if (strcmp(wsc->uri, "/register.cgi") == 0) {
		websvr_register(wsc);
	}
	else if (strcmp(wsc->uri, "/recv.cgi") == 0) {
		websvr_recv(wsc);
	}
	else if (strcmp(wsc->uri, "/send.cgi") == 0) {
		websvr_send(wsc);
	}
	else if (strncmp(wsc->uri, "/direct/", 8) == 0) {
		websvr_direct(wsc, wsc->uri + 8);
	}
	else {
		websvr_respond(wsc, "404 Not Found", "no such script", -1, 0);
	}
}

static struct {
	char *suffix;
	char *type;
} mime[] = {
	{ "html",	"text/html" },
	{ "htm",	"text/html" },
	{ "txt",	"text/plain" },
	{ "jpg",	"image/jpeg" },
	{ "jpeg",	"image/jpeg" },
	{ "gif",	"image/gif" },
	{ "pdf",	"application/pdf" },
	{ "js",		"application/x-javascript" },
	{ 0, 0 }
};

/* Try to determine the MIME type of this file.
 */
static char *mime_type(char *file){
	char *suffix = 0;
	int i;

	/* Find suffix.
	 */
	while (*file != 0)
		if (*file++ == '.')
			suffix = file;

	/* If there's no suffix, assume that it's an html file.
	 */
	if (suffix == 0 || *suffix == 0)
		return "text/html";
	
	/* Convert suffix to lower case.
	 */
	suffix = mem_string_copy(suffix, strlen(suffix));
	for (i = 0; suffix[i] != 0; i++)
		if ('A' <= suffix[i] && suffix[i] <= 'Z')
			suffix[i] = suffix[i] - 'A' + 'a';
	
	/* Find the suffix in the mime table.
	 */
	for (i = 0; mime[i].suffix != 0; i++)
		if (strcmp(mime[i].suffix, suffix) == 0) {
			free(suffix);
			return mime[i].type;
		}

	/* Assume that it's a plain text file.
	 */
	free(suffix);
	return "text/plain";
}

static int dump_visit(void *env, char *user, void **v){
	struct print_channel *pc = env;
	struct websvr_user *wsu = *v;

	pc_print(pc, "   %s:", user);
	if (wsu->redirect != 0 && *wsu->redirect != 0) {
		pc_print(pc, " (redirect = '%s')", wsu->redirect);
	}
	switch (wsu->tag) {
	case USER_IDLE:
		pc_print(pc, " IDLE\n");
		break;
	case USER_RECV:
		pc_print(pc, " RECV\n");
		break;
	case USER_SEND:
		pc_print(pc, " SEND (%s, %d)\n",
						wsu->u.send.type, wsu->u.send.size);
		break;
	default:
		pc_print(pc, " BAD TAG???\n");
	}

	return 1;
}

static void websvr_dump_wsa(struct websvr_act *wsa, struct print_channel *pc){
	pc_print(pc, "users:\n");
	tst_travel(&wsa->users, dump_visit, pc);
}

static char *websvr_dump_wsc(struct websvr_client *wsc, int *len){
	struct print_channel *pc = mc_open();

	pc_print(pc, "<pre>\n");
	pc_print(pc, "'%s'", wsc->cmd == 0 ? "<null>" : wsc->cmd);
	pc_print(pc, " '%s'", wsc->uri == 0 ? "<null>" : wsc->uri);
	pc_print(pc, " '%s'", wsc->version == 0 ? "<null>" : wsc->version);
	pc_print(pc, " (%d bytes)\n", wsc->size);
	websvr_dump_wsa(wsc->activity, pc);
	pc_print(pc, "</pre>\n");

	*len = mc_size(pc);
	return mc_close(pc);
}

/* Got a GET request.  Check the "files" directory and return the file if found.
 */
static void websvr_get(struct websvr_client *wsc){
	char *uri, *filename;
	int len;
	char *data;

	if (strcmp(wsc->uri, "/") == 0) {
		uri = "/index.html";
	}
	else {
		uri = wsc->uri;
	}
	if (strcmp(uri, "/state.html") == 0) {
		data = websvr_dump_wsc(wsc, &len);
	}
	else {
		filename = mem_print("files%s", uri);
		data = file_raw_read(filename, &len);
		free(filename);
	}
	if (data == 0) {
		websvr_respond(wsc, "404 Not Found", "no such page", -1, 0);
	}
	else {
		websvr_respond(wsc, "200 OK", data, len, mime_type(uri));
		free(data);
	}
}

/* Dispatch to the right routine based on the command.
 */
static void websvr_dispatch(struct websvr_client *wsc){
	if (strcmp(wsc->cmd, "GET") == 0 || strcmp(wsc->cmd, "HEAD") == 0) {
		websvr_get(wsc);
	}
	else if (strcmp(wsc->cmd, "POST") == 0) {
		websvr_post(wsc);
	}
	else {
		websvr_respond(wsc, "400 Bad Request",
						"don't understand command", -1, 0);
	}
}

/* Invoked for every user.  If there's a client, and it's the client that's closing down,
 * clean up the record.
 */
static int websvr_cleanup(void *env, char *id, void **v){
	struct websvr_client *wsc = env, **p;
	struct websvr_user *wsu = *v;

	if (wsu->tag == USER_RECV) {
		for ((p = &wsu->u.recv.clients); *p != 0; p = &(*p)->next) {
			if (*p == wsc) {
				*p = wsc->next;
				log_printf("websvr_cleanup: cleaned up recv record");
				break;
			}
		}
		if (wsu->u.recv.clients == 0) {
			wsu->tag = USER_IDLE;
		}
	}
	return 1;
}

/* When we receive a close event on a client stream, we forward one along the chain,
 * and clean up all the resources related to this event.
 */
static void websvr_close(struct websvr_client *wsc){
	struct websvr_act *wsa = wsc->activity;

	/* Send a close event to the output channel.
	 */
	wsa->refcnt++;
	ev_send(wsa->release, wsc->output, ev_create(evt_close));

	tst_release(&wsc->map, mem_release, 0);
	if (wsc->cmd != 0) {
		free(wsc->cmd);
	}
	if (wsc->uri != 0) {
		free(wsc->uri);
	}
	if (wsc->version != 0) {
		free(wsc->version);
	}
	if (wsc->body != 0) {
		free(wsc->body);
	}

	/* Clean up any lingering references to this client.
	 */
	tst_travel(&wsa->users, websvr_cleanup, wsc);

	ev_channel_release(wsc->in);
	wsc->in = 0;

	websvr_release(wsc->activity);

	free((char *) wsc);
}

/* This routine takes uri as input, removes redundant slashes, and processes "." and
 * "..".  The simplified uri is returned.  Everything after a question mark is
 * returned in *options (unless options == 0).
 */
static char *simplify(char *uri, char **options){
	char *file = malloc(strlen(uri) + 2);
	char *p = uri, *q, *r = file;

	do {
		/* Skip slashes to find start of component.
		 */
		while (*p == '/')
			p++;

		/* If no next component, it's a directory access.
		 */
		if (*p == 0 || *p == '?') {
			*r++ = '/';
			q = p;
			break;
		}

		/* Find end of component.
		 */
		for (q = p; *q != 0 && *q != '/' && *q != '?'; q++)
			;

		/* If next component is '.', skip it.
		 */
		if (strncmp(p, ".", q - p) == 0) {
			p = q;
			continue;
		}

		/* If the next component is "..", we have to strip one from the output.
		 */
		if (strncmp(p, "..", q - p) == 0) {
			if (r > file)
				while (*--r != '/')
					;
			p = q;
			continue;
		}

		/* Otherwise add the next component to the file name.
		 */
		*r++ = '/';
		while (p < q)
			*r++ = *p++;
	} while (*q != 0 && *q != '?');
	*r = 0;

	/* Return options
	 */
	if (options != 0)
		*options = *q == 0 ? "" : q + 1;

	return file;
}

/* Web requests arrive here.
 */
static void websvr_requests(void *env, struct event *ev){
	struct websvr_client *wsc = env;
	char *start, *uri;
	int len;
	struct chunk *chunk;

	/* See if it's an HTTP header.
	 */
	if (ev->type == evt_http) {
		/* Sanity check.  See if we were awaiting more data first.
		 */
		if (wsc->expected != 0) {
			err_fatal("websvr_requests: expected data, not header");
		}

		/* Get the "start", and parse it.
		 *
		 * TODO.  Put in a sanity check on the length here.
		 */
		start = ev_cast(ev, evt_http)->start;
		len = strlen(start);
		wsc->cmd = malloc(len + 1);
		uri = malloc(len + 1);
		wsc->version = malloc(len + 1);
		sscanf(start, "%s %s %s", wsc->cmd, uri, wsc->version);

		/* Remove ".."'s and the like from the uri.
		 */
		wsc->uri = simplify(uri, 0);
		free(uri);

		/* Steal the map.
		 */
		wsc->map = ev_cast(ev, evt_http)->map;
		ev_cast(ev, evt_http)->map = 0;

		/* See how much more data to be expected.
		 */
		wsc->expected = ev_cast(ev, evt_http)->size;

		/* If no more data expected, dispatch now.
		 */
		if (wsc->expected == 0) {
			websvr_dispatch(wsc);
		}
	}

	/* Maybe it's data...
	 */
	else if (ev->type == evt_data) {
		chunk = ev_cast(ev, evt_data)->chunk;

		/* Sanity check.  See if we were expecting this data.
		 */
		if (chunk->size > wsc->expected) {
			err_fatal("websvr_requests: didn't expect that much data");
		}

		/* Add this data to the buffer.
		 */
		if (wsc->size == 0) {
			wsc->body = malloc(chunk->size);
		}
		else {
			wsc->body = realloc(wsc->body, wsc->size + chunk->size);
		}
		memcpy(&wsc->body[wsc->size], chunk->data, chunk->size);
		wsc->size += chunk->size;

		/* See if this was all we waited for.
		 */
		wsc->expected -= chunk->size;
		if (wsc->expected == 0) {
			websvr_dispatch(wsc);
		}
	}

	/* If not, perhaps it's a close event.
	 */
	else if (ev->type == evt_close) {
		websvr_close(wsc);
	}

	/* We ignore error and other events.
	 */
	else {
		err_warning("websvr_requests: unknown event type");
	}

	/* Done.
	 */
	ev_handled(ev);
}

/* New client notifications arrive here.
 */
static void websvr_handler(void *env, struct event *ev){
	struct websvr_act *wsa = env;
	struct ev_channel *parser;
	struct websvr_client *wsc;
	struct ev_channel *http_create(struct ev_channel *dst);

	/* See if it's a new client notification.
	 */
	if (ev->type == evt_client) {
		/* Create a new websvr client record.
		 */
		wsc = (struct websvr_client *) calloc(1, sizeof(*wsc));
		wsc->activity = wsa;
		wsc->in = ev_channel_alloc(wsa->activity, websvr_requests, wsc);
		wsa->refcnt++;

		/* Create a TCP output activity for the client socket.
		 */
		wsc->output = tcp_output_create(ev_cast(ev, evt_client)->si);

		/* Create an HTTP request parser handler, which reads from the client socket
		 * and sends http events to the web request event handler.
		 */
		parser = http_create(wsc->in);

		/* Create a TCP input activity on the client socket, and send
		 * input events to the HTTP request parser.
		 */
		tcp_input_create(ev_cast(ev, evt_client)->si, parser);
	}

	/* If it's a close event, clean up resources.
	 */
	else if (ev->type == evt_close) {
		ev_channel_release(wsa->in);
		wsa->in = 0;
		websvr_release(wsa);
	}

	/* We ignore error and other events.
	 */
	else {
		err_warning("websvr_handler: unknown event type");
	}

	ev_handled(ev);
}

static void websvr_event_release(void *env, ev_t ev){
	struct websvr_act *wsa = env;

	ev_release(ev);
	websvr_release(wsa);
}

/* Create a new web server activity.
 */
struct ev_channel *websvr_create(char *db){
	struct websvr_act *wsa = (struct websvr_act *) calloc(1, sizeof(*wsa));

	wsa->refcnt = 1;
	wsa->activity = act_alloc(wsa);
	wsa->in = ev_channel_alloc(wsa->activity, websvr_handler, wsa);
	wsa->release = ev_channel_alloc(wsa->activity, websvr_event_release, wsa);
	wsa->db = mem_string_copy(db, strlen(db));

	/* Read the user database.
	 */
	websvr_read_users(wsa);

	return wsa->in;
}

/* Initialize this module.
 */
void websvr_init(void){
	evt_data_data = evt_index(evt_data, "data");
	evt_error_descr = evt_index(evt_error, "descr");
	evt_client_socket = evt_index(evt_client, "socket");
	evt_http_start = evt_index(evt_http, "start");
	evt_http_map = evt_index(evt_http, "map");
	evt_http_size = evt_index(evt_http, "size");
}

void websvr_done(void){
}
