/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* This event handler processes data events coming from an XML stream.  It produces the
 * following types of events:
 *
 *	root/xml/start:	a start element
 *	root/data:		a data element
 *	root/xml/end:	an end element
 *	root/close:		stream has been closed
 */
#include <stdlib.h>
#include <stdio.h>
#include "oslib.h"
#include "xmlparse/xmlparse.h"

/* Flow control window.
 */
#define LO_WATER			2
#define HI_WATER			4

/* This module creates a new event type.
 */
evt_t evt_xml;
evt_t evt_xml_start;
evt_t evt_xml_end;

/* Various event indices go here.
 */
static int evt_xml_start_name;
static int evt_xml_start_attrs;
static int evt_xml_end_name;
static int evt_data_data;
static int evt_error_descr;

struct xml_act {
	struct activity *activity;
	struct ev_channel *in;			/* input event channel */
	struct ev_channel *release;		/* event release channel */
	struct ev_channel *dst;			/* output event channel */
	XML_Parser parser;				/* instance of expat parser */
};

/* Clean up a request parser activity.  Sends a close event on the output channel.
 */
static void xml_act_release(struct xml_act *xa){
	ev_channel_release(xa->release);
	act_release(xa->activity);
	free((char *) xa);
}

/* Invoked when a parse error occurs.  Outputs an error event and puts the parser in an
 * error state in which nothing more can be parsed.
 */
static void xml_act_error(struct xml_act *xa, char *descr){
	ev_t ev;

	ev = ev_create(evt_error);
	ev_set_ptr(ev, evt_error_descr, "xml_act_error");
	ev_send(xa->release, xa->dst, ev);
}

static void xml_act_ev_send(struct xml_act *xa, ev_t ev){
	/* Send the event.
	 */
	ev_send(xa->release, xa->dst, ev);

	/* Flow control:  if too many outstanding events, block input.  This
	 * flag gets unset in the event release handler.
	 */
	if (act_outstanding(xa->activity) > HI_WATER) {
		ev_set_blocked(xa->in, true);
	}
}

/* An XML start tag has been parsed.
 */
void xa_startElement(void *userData, const char *name, const char **attrs)
{
	struct xml_act *xa = userData;
	ev_t ev = ev_create(evt_xml_start);
	map_t map = 0;
	int i;
	const char *attr, *v;
	void **p;

	/* Make a map of the attributes.
	 */
	for (i = 0; (attr = attrs[i]) != 0; i++) {
		if ((v = attrs[++i]) == 0) {
			err_warning("xa_startElement: bad attribute");
			break;
		}
		p = tst_find(&map, attr, 1);
		if (*p != 0) {
			err_warning("xa_startElement: duplicate attribute");
		}
		else {
			*p = mem_string_copy(v, strlen(v));
		}
	}

	/* Add the stuff to the output event, and send it.
	 */
	ev_set_ptr(ev, evt_xml_start_name, mem_string_copy(name, strlen(name)));
	ev_set_ptr(ev, evt_xml_start_attrs, map);
	xml_act_ev_send(xa, ev);
}

/* The end of an element has been reached.
 */
void xa_endElement(void *userData, const char *name)
{
	struct xml_act *xa = userData;
	ev_t ev = ev_create(evt_xml_end);

	/* Add the stuff to the output event, and send it.
	 */
	ev_set_ptr(ev, evt_xml_end_name, mem_string_copy(name, strlen(name)));
	xml_act_ev_send(xa, ev);
}

void xa_charData(void *userData, const XML_Char *s, int len) {
	struct xml_act *xa = userData;
	ev_t ev = ev_create(evt_data);
	struct chunk *chunk;

	/* Create the chunk.
	 */
	chunk = (struct chunk *) calloc(1, sizeof(*chunk));
	chunk->data = malloc(len);
	chunk->size = len;
	memcpy(chunk->data, s, len);

	/* Send the event.
	 */
	ev_set_ptr(ev, evt_data_data, chunk);
	xml_act_ev_send(xa, ev);
}

/* This function deals with HTTP request input.
 */
void xml_act_handler(void *env, struct event *ev){
	struct xml_act *xa = env;
	struct chunk *chunk;

	/* If it's data, parse it.
	 */
	if (ev->type == evt_data) {
		chunk = ev_get_ptr(ev, evt_data_data);
		if (!XML_Parse(xa->parser, chunk->data, chunk->size, 0)) {
			log_printf("xml_act_handler: XML_Parse: %s at line %d",
				XML_ErrorString(XML_GetErrorCode(xa->parser)),
				XML_GetCurrentLineNumber(xa->parser));
		}
	}

	/* If it's a close event, clean up everything.
	 */
	else if (ev->type == evt_close) {
		ev_send(xa->release, xa->dst, ev_create(evt_close));
		ev_channel_release(xa->in);
		xa->in = 0;
	}

	/* If we get a bad input event, we output an error event.
	 */
	else {
		xml_act_error(xa, "bad input event");
	}

	/* Done with the input event.
	 */
	ev_handled(ev);
}

/* The event release handler.
 */
void xml_act_event_release(void *env, struct event *ev){
	struct xml_act *xa = env;

	ev_release(ev);

	/* If the input channel is closed, clean up.
	 */
	if (xa->in == 0) {
		if (act_outstanding(xa->activity) == 0) {
			xml_act_release(xa);
		}
	}

	/* Otherwise see if we can open up our input channel.
	 */
	else if (act_outstanding(xa->activity) < LO_WATER) {
		ev_set_blocked(xa->in, false);
	}
}

/* Create a new XML parser activity.
 */
struct ev_channel *xml_act_create(struct ev_channel *dst){
	struct xml_act *xa = (struct xml_act *) calloc(1, sizeof(*xa));

	xa->activity = act_alloc(xa);
	xa->in = ev_channel_alloc(xa->activity, xml_act_handler, xa);
	xa->release = ev_channel_alloc(xa->activity, xml_act_event_release, xa);
	xa->dst = dst;
	xa->parser = XML_ParserCreate(NULL);
	XML_SetUserData(xa->parser, xa);
	XML_SetElementHandler(xa->parser, xa_startElement, xa_endElement);
	XML_SetCharacterDataHandler(xa->parser, xa_charData);
	return xa->in;
}

/* Used to release ternary search tree maps.
 */
static void map_release(void *map){
	tst_release((map_t *) &map, mem_release, 0);
}

static void do_free(void *p){
	free(p);
}

/* Initialize this module.
 */
void xml_act_init(void){
	extern evt_t evt_root, evt_error, evt_data;

	evt_data_data = evt_index(evt_data, "data");
	evt_error_descr = evt_index(evt_error, "descr");

	evt_xml = evt_create(evt_root, "xml");
	evt_finalize(evt_xml);

	evt_xml_start = evt_create(evt_xml, "start");
	evt_ptr_property(evt_xml_start, "name", do_free);
	evt_ptr_property(evt_xml_start, "attrs", map_release);
	evt_finalize(evt_xml_start);
	evt_xml_start_name = evt_index(evt_xml_start, "name");
	evt_xml_start_attrs = evt_index(evt_xml_start, "attrs");

	evt_xml_end = evt_create(evt_xml, "end");
	evt_ptr_property(evt_xml_end, "name", do_free);
	evt_finalize(evt_xml_end);
	evt_xml_end_name = evt_index(evt_xml_end, "name");
}

void xml_act_done(void){
	evt_release(evt_xml);
	evt_release(evt_xml_start);
	evt_release(evt_xml_end);
}
