/* This file is part of the MediaNet Project.
   Copyright (C) 2002-2004 Michael Hicks, Robbert van Renesse

   MediaNet is free software; you can redistribute it and/or it
   under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   MediaNet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place, Suite
   330, Boston, MA 02111-1307 USA. */

/* This is the XML parser.  It is based on the EXPAT package, wrapped however with something that
 * makes it a bit easier to figure out what to do with all the pieces that EXPAT identifies.
 * The basic idea is to have a stack.  Each time a start tag is identified, a new entry is pushed
 * onto the stack (including the attributes), but no other action is taken at this point.  The
 * action is taken when an end tag is encountered.  Just before the entry is popped, the action
 * that is taken depends entirely on what's on the stack.  All the data between the tags will be
 * stored, concatenated, in the "data" field of the stack entry.
 *
 * To make things even simpler, you can associate callbacks with particular stacks.  For example:
 *
 *			xps_upcall(xps, "/root/mib/entry", upcall, env)
 *
 * will cause "(*upcall)(env, stack, n)" to be invoked each time <root><mib><entry> and all the
 * corresponding closing tags have been found.  "stack" is an array with entries containing the
 * attributes and the data, starting with the root first.  "n" is the size of the stack.  You can
 * call xp_upcall with a null path, in which case the corresponding upcall is invoked for
 * everything that didn't match.
 *
 * The other methods are:
 *
 *		xps_t xps_create()
 *			Create a new "syntax".
 *
 *		void xps_release(xps)
 *			Release a syntax handle.
 *
 *		xp_t xp_create(xps)
 *			Create a new parser for the given syntax, and return an xp handle.
 *
 *		void xp_data(xp, char *data, int size)
 *			Offer a bunch of data to the XML parser.
 *
 *		void xp_free(xp)
 *			Release the parser.
 */

#include "oslib.h"
#include "xmlparse/xmlparse.h"

/* Maximum stack depth.
 *
 * TODO.  Check for overflow or grow dynamically.
 */
#define MAX_DEPTH	10

struct xsyntax {
	map_t map;				/* map of syntax children */
	void (*upcall)(void *env, struct stack_entry *stack, int n);
	void *env;
};

/* One of these is maintained per server connection.
 */
struct xparse {
	xps_t syntax;				/* syntax of XML language */
	XML_Parser parser;			/* XML parser */
	struct stack_entry stack[MAX_DEPTH];		/* XML parsing stack */
	int sp;						/* points to top of stack (initially -1) */
};

xps_t xps_create(void){
	xps_t xps;

	xps = (xps_t) calloc(1, sizeof(*xps));
	return xps;
}

/* Invoke upcall when the given path has been recognized.
 */
void xps_upcall(xps_t xps, char *path,
		void (*upcall)(void *env, struct stack_entry *, int), void *env){
	char *tag;
	void **p;

	if (path != 0) {
		while (*path == '/') {
			/* Figure out what the next tag is.
			 */
			tag = ++path;
			while (*path != 0 && *path != '/') {
				path++;
			}
			tag = mem_string_copy(tag, path - tag);

			/* Create a new entry, or use the existing one.
			 */
			p = tst_find(&xps->map, tag, 1);
			if ((xps = *p) == 0) {
				*p = xps = xps_create();
			}

			free(tag);
		}
	}
	xps->upcall = upcall;
	xps->env = env;
}

static void xps_map_release(void *env, void *v){
	xps_release(v);
}

void xps_release(xps_t xps){
	tst_release(&xps->map, xps_map_release, 0);
	free((char *) xps);
}

/* An XML start tag has been parsed.  Currently we are only interested in level 1 elements, and ignore
 * the rest.
 */
static void start_element(void *env, const char *name, const char **attrs){
	xp_t xp = env;
	int i;
	const char *attr, *v;
	struct stack_entry *se;
	void **p;

	/* Push this entry onto the stack.
	 */
	se = &xp->stack[++xp->sp];
	se->name = mem_string_copy(name, strlen(name));
	for (i = 0; (attr = attrs[i]) != 0; i++) {
		p = tst_find(&se->attrs, attr, 1);
		if (*p != 0) {
			err_warning("start_element: already have this attribute");
			free(*p);
		}
		if ((v = attrs[++i]) == 0) {
			err_warning("start_element: null attribute??");
			*p = 0;
		}
		else {
			*p = mem_string_copy(v, strlen(v));
		}
	}

	/* Don't gather top-level data.
	 */
	se->pc = xp->sp == 0 ? 0 : mc_open();
}

/* The end of an element has been reached.  Here's where we take action.
 */
static void end_element(void *env, const char *name){
	xp_t xp = env;
	xps_t xps = xp->syntax;
	struct stack_entry *se = &xp->stack[xp->sp];
	void **p;
	int i;

	/* Close the print_channel to recover the data between the tags.
	 */
	se->data = se->pc == 0 ? 0 : mc_close(se->pc);

	/* Look up the corresponding record in the syntax tree.
	 */
	for (i = 0; i <= xp->sp; i++) {
		p = tst_find(&xps->map, xp->stack[i].name, 0);
		if (p == 0) {
			/* It's not there.  Invoke the top-level upcall.
			 */
			xps = xp->syntax;
			break;
		}
		xps = *p;
	}

	/* Invoke the upcall.
	 */
	if (xps->upcall != 0) {
		(*xps->upcall)(xps->env, xp->stack, xp->sp + 1);
	}

	/* Pop element of the stack.
	 */
	free(se->name);
	tst_release(&se->attrs, mem_release, 0);
	if (se->data != 0) {
		free(se->data);
	}
	xp->sp--;
}

/* Collect the pieces of data in the current element.
 */
static void char_data(void *env, const XML_Char *s, int len){
	xp_t xp = env;
	struct stack_entry *se = &xp->stack[xp->sp];

	if (se->pc != 0)
		pc_print(se->pc, "%.*s", len, s);
}

void xp_free(xp_t xp){
	XML_ParserFree(xp->parser);
	free((char *) xp);
}

void xp_data(xp_t xp, char *data, int size){
	if (!XML_Parse(xp->parser, data, size, 0)) {
		log_printf("xp_data: XML_Parse: %s at line %d",
			XML_ErrorString(XML_GetErrorCode(xp->parser)),
			XML_GetCurrentLineNumber(xp->parser));
	}
}

/* Create a new parser.  xps contains the "syntax".
 */
xp_t xp_create(xps_t xps){
	xp_t xp;

	xp = (xp_t) calloc(1, sizeof(*xp));
	xp->syntax = xps;
	xp->parser = XML_ParserCreate(NULL);
	XML_SetUserData(xp->parser, xp);
	XML_SetElementHandler(xp->parser, start_element, end_element);
	XML_SetCharacterDataHandler(xp->parser, char_data);
	xp->sp = -1;
	return xp;
}
