/*
 *
 *		    Copyright (c) 1991
 *		    Jeff Hollingsworth
 *
 *  Permission to use, copy, modify, and distribute this
 *  software and its documentation for any purpose and without
 *  fee is hereby granted, provided that the above copyright
 *  notice appear in all copies and that both that copyright
 *  notice and this permission notice appear in supporting
 *  documentation, and that the name of principal(s) not be used
 *  in any advertising or publicity relating to this software
 *  without specific, written prior authorization.  No representations
 *  are made about the suitability of this software for any purpose.
 *  It is provided "as is" without express or implied warranty.
 */

#ifndef lint
static char rcsid[] = "@(#) $Header: /home/poona/hollings/src/original/xweather/RCS/callbacks.c,v 1.9 1992/12/17 17:40:40 hollings Exp $";
#endif

/*
 * $Log: callbacks.c,v $
 * Revision 1.9  1992/12/17  17:40:40  hollings
 * fixed type of 3rd arf to XtAddEventHandler.
 *
 * Revision 1.8  1992/12/17  01:17:20  hollings
 * SYSV port.
 *
 * Revision 1.7  1992/12/16  19:24:33  hollings
 * fixes to compile with gcc.
 *
 * Revision 1.6  1992/07/30  19:36:52  hollings
 * fixed making area too small.
 *
 * Revision 1.5  1992/07/02  19:39:23  hollings
 * Color codeded watch boxes.
 *
 * Revision 1.4  1992/05/05  23:27:40  hollings
 * Version 1.0.
 *
 * Revision 1.3  1992/04/24  20:45:46  hollings
 * user preferences.
 *
 * Revision 1.2  1992/01/22  20:42:28  hollings
 * man page uses ScrollByL widget.
 *
 * Revision 1.1  1991/12/10  16:51:29  hollings
 * Initial revision
 *
 */

#include <stdio.h>
#include <strings.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/List.h>
#include <math.h>
#include <sys/time.h>
#include <memory.h>
#include <unistd.h>

#include "ScrollByL.h"
#include "windows.h"
#include "sadecode.h"
#include "map.h"
#include "radar.h"
#include "rubberband.h"
#include "cache.h"
#include "io.h"

extern char *doprod();
extern char *report();
extern Pixmap icon_pixmap;
extern Widget top;
extern double maptop;
extern double mapbottom;
extern double mapleft;
extern double mapright;
extern double radc, rdc, drc, a;
extern void XtSetValue();
extern void  end_rubber_band();
extern void  start_rubber_band();
extern struct report *FindStationByNameOrId();
extern struct report *FindNearestStationWithProduct();
char tempDispOptions[NR_OPTIONS];
struct report *nearestStation;

#define SEC_PER_DAY	(3600 * 24)

/* ARGSUSED */
void help_window(w, text, event)
Widget w;
char *text;
XEvent *event;
{
    extern  Widget  help_widget;
    Arg             help_arg;

    XtSetArg(help_arg, XtNlabel, text);
    XtSetValues(help_widget, &help_arg, 1);
    XFlush(XtDisplay(help_widget));
}

extern struct timeval timeAtFetch;
extern struct timeval lastUaTime;

void makeTitleLine()
{
    time_t lastHour;
    struct tm *tim;
    struct timeval timeUsed;

    if (pref.height == 0) {
	timeUsed = timeAtFetch;
    } else {
	timeUsed = lastUaTime;
    }

    if (pref.epochTime) {
	strcpy(title, pref.epochTime);
	strcat(title, " UTC (Historic) ");
    } else if (pref.gmtTime) {
	tim = gmtime(&timeUsed.tv_sec);
	tim->tm_min = 0;
	tim->tm_sec = 0;
	strcpy(title, asctime(tim));
	title[24] = '\0';
	strcat(title, " (UTC) ");
    } else {
	tim = gmtime(&timeUsed.tv_sec);
	lastHour = timeUsed.tv_sec - tim->tm_min * 60 - tim->tm_sec;
	strcpy(title, ctime(&lastHour));
	title[24] = '\0';
    }

    if (pref.height == 0) {
	strcat(title, "- Surface Analysis ");
    } else { 
	int height;
	char temp[40];
	switch (pref.height) {
	    case 8: height = 850; break;
	    case 7: height = 700; break;
	    case 5: height = 500; break;
	    case 3: height = 300; break;
	    case 2: height = 200; break;
	}
	sprintf(temp, "- %d mili-bar Analysis ", height);
	strcat(title, temp);
    }

    if ((pref.disp_list[SHO_ARS]) || (pref.disp_list[SHO_RADSUM])) 
	strcat(title, "and Radar Summary");

    XtSetValue(info.TB, XtNlabel, title);
}

/*
 * Called by pressing the button, or a SIGALRM.
 *
 */
void refetch_func()
{
    char *emptyList[1];
    struct radarObject *tmpObject;
    extern struct radarObject *radarObjects;

    XDefineCursor(d, xwin, watch); 

    setPictureTime();

    /* empty old list */
    emptyList[0] = "";
    XawListChange(info.notes, emptyList, 1, -1, False);

    /* empty all the old cached data */
    purgeCache();

    /* get the new data */
    load_reports(RELOAD);

    /* dump existing radar objects */
    while (radarObjects) {
	tmpObject = radarObjects->next;
	if (radarObjects->type == RADAR_MDR) 
	    free(radarObjects->info.region.data);
	free(radarObjects);
	radarObjects = tmpObject;
    }

    /* update the screen too */
    resizePixmap();

    /* update data */
    makeTitleLine();
    XawListChange(info.notes, note, nnotes, -1, True);

    /* reset to default cursor */
    XDefineCursor(d, xwin, crosshair); 
}

/* ARGSUSED */
void quit_func(w, cdata, call_data)
Widget w;
caddr_t cdata;
caddr_t call_data;
{
    char buf[80];

    requestServerData("EXIT\n");
    readServerData(buf, sizeof buf);
    printf("\n%s", &buf[2]);
    exit(0);
}


/* ARGSUSED */
void change_func(w, cdata, call_data)
Widget w;
caddr_t cdata;
caddr_t call_data;
{
    Position x, y;

    CreateDisplayPopup();
#ifndef hp
    XtTranslateCoords(w, 10, 10, &x, &y);
    XtMoveWidget(info.displaySheet, x, y);
#endif
    XtPopupSpringLoaded(info.displaySheet);
    memcpy(tempDispOptions, pref.disp_list, NR_OPTIONS);
}

/* ARGSUSED */
void pref_func(w, cdata, call_data)
Widget w;
caddr_t cdata;
caddr_t call_data;
{
    Position x, y;
    extern struct preference prefBu;

    CreatePrefPopup();
#ifndef hp
    XtTranslateCoords(w, 10, 10, &x, &y);
    XtMoveWidget(info.prefSheet, x, y);
#endif
    XtPopupSpringLoaded(info.prefSheet);
    memcpy( &prefBu, &pref, sizeof(pref));
}

/* ARGSUSED */
void display_func(buttonWidget, cdata, call_data)
Widget buttonWidget;
caddr_t cdata;
caddr_t call_data;
{
    /* Force it to use new data */
    resizePixmap();
}

/* ARGSUSED */
void redraw_handler(mapWidget, c_data, event)
Widget mapWidget;
caddr_t c_data;
XExposeEvent *event;
{
    XEvent extraEvent;

    /*
     * Eat expose events.
     *
     */
    XSync(d, False);
    while (XCheckWindowEvent(d, w, ExposureMask, &extraEvent) == True);

    XGetWindowAttributes(d, xwin, &xwa);
    resizePixmap();
}

/* ARGSUSED */
void resize_handler(mapWidget, c_data, event)
Widget mapWidget;
caddr_t c_data;
XEvent *event;
{

    XWindowAttributes xwa2;

    XGetWindowAttributes(d, xwin, &xwa2);
    /* see if we changed size */
    if ((xwa.width > 0) && 
	((xwa2.width != xwa.width) || (xwa2.height != xwa.height))) {
	if (xwa2.width < xwa.width) XClearArea(d, xwin, 0, 0, 0, 0, True);
    }
}

/* ARGSUSED */
void station_func(buttonWidget, c_data, rest)
Widget buttonWidget;
caddr_t c_data;
caddr_t rest;
{
    CreateStationInfoPopup();
    XtPopupSpringLoaded(info.dialog);
}

/* ARGSUSED */
void full_func(buttonWidget, c_data, rest)
Widget buttonWidget;
caddr_t c_data;
caddr_t rest;
{
    XtSetValue(priorityWidgets[maxprio-1], XtNleftBitmap, None);
    maxprio = 2;
    XtSetValue(priorityWidgets[maxprio-1], XtNleftBitmap, (char *)check_pixmap);
    center_lat = 39.0;
    center_lon = 97.0;
    scale = 2.3;

    /* Draw new map */
    resizePixmap();
}

/* ARGSUSED */
void ToggleDisplayItem(Widget menuWidget, int id, caddr_t cdata)
{
    if (tempDispOptions[id]) {
	/* Disable it */
	tempDispOptions[id] = False;
    } else {
	/* enable it */
	/* XtSetValue(menuWidget, XtNleftBitmap, check_pixmap); */
	tempDispOptions[id] = True;
    }
}

/* ARGSUSED */
void ChangePriority(Widget menuWidget, int newPriority, caddr_t cdata)
{
    XtSetValue(priorityWidgets[maxprio-1], XtNleftBitmap, (char *) None);
    XtSetValue(menuWidget, XtNleftBitmap, (char *) check_pixmap);
    maxprio = newPriority;

    /* Draw new map */
    resizePixmap();
}

struct regionalDef {
    int id;
    double center_lat;
    double center_lon;
    double scale;
};

struct regionalDef regionalList[] = {
/* id		lat	long	scale */
{ REGION_NE,	42.0,	76.0,	0.9 },
{ REGION_AT,	37.0,	82.0,	0.9 },
{ REGION_SE,	31.0,	88.0,	1.1 },
{ REGION_MW,	43.0,	93.0,	1.1 },
{ REGION_SP,	32.5,	100.0,	1.2 },
{ REGION_NW,	44.0,	112.0,	1.0 },
{ REGION_SW,	37.0,	112.0,	1.0 },
{ (int) NULL,	0.0,	0.0,	0.0 },
};

/* ARGSUSED */
void ShowRegion(Widget menuWidget, int region, caddr_t cdata)
{
    int i;

    XtSetValue(priorityWidgets[maxprio-1], XtNleftBitmap, None);
    maxprio = 3;
    XtSetValue(priorityWidgets[maxprio-1], XtNleftBitmap, (char*) check_pixmap);
    for (i=0; regionalList[i].id; i++) {
	if (regionalList[i].id == region) {
	    center_lat = regionalList[i].center_lat;
	    center_lon = regionalList[i].center_lon;
	    scale = regionalList[i].scale;
	}
    }
    /*
     * Force redraw.
     */
    resizePixmap();
}

struct upperAir {
    char id[4];
    int temp, dpdp;
    int dir, vel, ht;
    struct report *station;
} ua;

void doCurrentUaConditions(int lat, int lon)
{
    float x;
    char s[256];
    char buf[20];
    struct tm *tp;
    time_t sec, hour;
    double round();
    char cond[256];
    long lat2, lon2;
    char repTime[200];
    struct timeval tv;
    struct serverDataList *isoData;
    double newDist, dist, gc_distance();

    sprintf(buf, "U%d %s\n", pref.height, upperairtime);
    isoData = getCacheableData(buf, "*** Isometric data not available");
    memset(&ua, '\0', sizeof(ua));
    /* skip the first item - its a header */
    ua.id[0] = '\0';	/* make it empty for min check */
    for (isoData = isoData->next; isoData; isoData = isoData->next) {
	if (!strcmp(isoData->data[0], "ZZZ")) continue;
	lat2 = 3600 * atof(isoData->data[1]); 
	lon2 = 3600 * atof(isoData->data[2]);
	newDist = gc_distance(lat, lon, lat2, lon2);
	if ((newDist < dist) || !ua.id[0]) {
	    strcpy(ua.id, isoData->data[0]);
	    /* arrives in meters, convert to feet */
	    ua.ht = atoi(isoData->data[3]) * 3.280840;
	    ua.temp = round(atof(isoData->data[4]));
	    ua.dpdp = round(atof(isoData->data[5]));
	    ua.dir = atoi(isoData->data[6]);
	    ua.vel = atoi(isoData->data[7]);
	    dist = newDist;
	}
    }
    ua.station = FindStationByNameOrId(ua.id);
    nearestStation = ua.station;
    if (!ua.station) abort();

    x = 1.15 * (float)ua.vel;
    sprintf(cond, "%d F, %d dpdp, wind %s %2d mph at %d ft.", 
	ua.temp, ua.dpdp, winddir(ua.dir), (int)(.5+x), ua.ht);

    /* now time of report */
    hour = upperairtime[6] * 10 + upperairtime[7];
    sec = (tv.tv_sec / (SEC_PER_DAY)) * SEC_PER_DAY + hour * 3600;
    if (pref.gmtTime) {
	tp = gmtime(&sec);
    } else {
	tp = localtime(&sec);
    }
    strftime(repTime, sizeof(repTime), "%H:%M %Z", tp);
    sprintf(s, "%19s, %s (%s) %s", ua.station->name, 
	 ua.station->state, repTime, cond);
    help_window(NULL, s, NULL);
    sprintf(s, "%s, %s", ua.station->name, ua.station->state);
    XtSetValue(info.mapMenuLabel, XtNlabel, s);
}

void doCurrentSurfaceConditions(int lat, int lon)
{
    int i;
    char *sky;
    char s[256];
    struct tm *tp;
    char cond[256];
    long lat2, lon2;
    struct report *r;
    char repTime[200];
    struct timeval tv;
    struct timezone tz;
    time_t sec, hour, min;
    char *curwx;
    static char *cldprio = " sSbBoOxX";
    double newDist, dist, gc_distance();

    /* find nearest station */
    nearestStation = NULL;
    dist = 100000.0;
    for (i = 0, r = rr; i < nrr; i++, r++) {
	if (r->prio > maxprio) continue;
	lat2 = r->lat * 3600;
	lon2 = r->lon * 3600;
	newDist = gc_distance(lat, lon, lat2, lon2);
	if ((newDist < dist) || !nearestStation) {
	    nearestStation = r;
	    dist = newDist;
	}
    }
    if (pref.rawConditions) {
	strcpy(s, nearestStation->raw);
	help_window(NULL, s, NULL);
	sprintf(s, "%s, %s", nearestStation->name, 
	    nearestStation->state);
	XtSetValue(info.mapMenuLabel, XtNlabel, s);
	return;
    }

    if (nearestStation->time[0] == 'M') {
	sprintf(s, "Current condition for %s, %s not available", 
	    nearestStation->name, nearestStation->state);
	help_window(NULL, s, NULL);
	sprintf(s, "%s, %s", nearestStation->name, 
	    nearestStation->state);
	XtSetValue(info.mapMenuLabel, XtNlabel, s);
	return;
    }
    if (nearestStation->nlayers == MISSING_I)
	curwx = NULL;
    else
	curwx = "Clear";
    sky = cldprio;
    for (i = 0; i < nearestStation->nlayers; i++) {
	char *s = index(cldprio, nearestStation->cover[i]);
	assert (s != NULL);
	if (s > sky) sky = s;
	}
    if (*sky == 's' || *sky == 'S') curwx = "Partly cloudy";
    else if (*sky == 'b' || *sky == 'B') curwx = "Mostly cloudy";
    else if (*sky == 'o' || *sky == 'O') curwx = "Cloudy";
    if (nearestStation->wx[0]) curwx = wxdecode(nearestStation->wx);

    if (nearestStation->temp != -9999) {
	sprintf(cond, "%d F, ", nearestStation->temp);
    } else {
	sprintf(cond, "temp NA, ");
    }
    if (nearestStation->rh != -9999) {
	sprintf(s, "%d%% humidity, ", nearestStation->rh);
    } else {
	sprintf(s, "humidity NA, ");
    }
    strcat(cond, s);
    if (curwx) strcat(cond, curwx);
    if (nearestStation->vis > -0.01) {
	sprintf(s, ", visibiliy %2.0f miles", nearestStation->vis);
    } else {
	sprintf(s, ", visibiliy NA");
    }
    strcat(cond, s);
    if (nearestStation->wdir != MISSING_I) {
	if (nearestStation->wspd == 0)
	    sprintf(s, ", wind calm");
	else {
	    float x;
	    x = 1.15 * (float)nearestStation->wspd;
	    sprintf(s, ", wind %s %2d mph", 
		winddir(nearestStation->wdir), (int)(.5+x));
	    if (nearestStation->wgust != MISSING_I) {
		strcat(cond, s);
		x = 1.15 * (float)nearestStation->wgust;
		sprintf(s, " gusts to %d", (int)(.5+x));
	    }
	}
    }
    strcat(cond, s);
    gettimeofday(&tv, &tz);
    sscanf(nearestStation->time, "%2d%2d", &hour, &min);
    sec = (tv.tv_sec / (SEC_PER_DAY)) * SEC_PER_DAY + hour * 3600 + 
	min * 60;
    if (pref.gmtTime) {
	tp = gmtime(&sec);
    } else {
	tp = localtime(&sec);
    }
    strftime(repTime, sizeof(repTime), "%H:%M %Z", tp);
    sprintf(s, "%19s, %s (%s) %s", nearestStation->name, 
	 nearestStation->state, repTime, cond);
    help_window(NULL, s, NULL);
    sprintf(s, "%s, %s", nearestStation->name, nearestStation->state);
    XtSetValue(info.mapMenuLabel, XtNlabel, s);
}

void xyToLatLon(int x, int y, double *lat, double *lon)
{

    double x1,y1;
    float rho, ang;

    x1 = (((double)x)*(mapright-mapleft)/xwa.width)+mapleft;
    y1 = (((double)y)*(mapbottom-maptop)/xwa.height)+maptop;
    rho = sqrt(x1*x1+y1*y1);
    ang = atan(-(x1/y1));
    *lat = 90.-(atan(rho/1.866/a)*2./drc);
    *lon = center_lon - (((double)ang)/drc);
}

/* ARGSUSED */
void button_handler(w, data, event)
Widget w;
caddr_t data;
XButtonEvent *event;
{
    int lat, lon;
    double dlat1, dlon1;


    switch (event->button) {
	case Button1:
	    xyToLatLon(event->x, event->y, &dlat1, &dlon1);
	    lat = 3600 * dlat1;
	    lon = 3600 * dlon1;
	    if (pref.height == 0) {
		doCurrentSurfaceConditions(lat, lon);
	    } else {
		doCurrentUaConditions(lat, lon);
	    }
	    break;

	case Button2:
	    start_rubber_band(w, (rubber_band_data *) data, event);
	    break;

	case Button3:
	    xyToLatLon(event->x, event->y, &center_lat, &center_lon);
	    resizePixmap();
	    break;
    }
}


extern char stationName[];

/* ARGSUSED */
void DumpProduct(Widget continueButton, Widget shell, caddr_t cdata)
{
    XtPopdown(shell);
    XtUnmapWidget(shell);
    XtDestroyWidget(shell);
}

/*
 * Display a product in a window.
 *
 */
void DisplayProduct(char *fileName)
{
    int acount;
    Widget cont;
    Arg args[100];
    XEvent event;
    XSizeHints hints;
    Widget shell, box, data;

    acount = 0;
    XtSetArg(args[acount], XtNallowShellResize, True); acount++;
    XtSetArg(args[acount], XtNx, xwa.width/2); acount++;
    XtSetArg(args[acount], XtNy, xwa.height/2); acount++;
    XtSetArg(args[acount], XtNsaveUnder, True); acount++;
    XtSetArg(args[acount], XtNiconPixmap, icon_pixmap); acount++;
    shell = XtCreatePopupShell("Product", topLevelShellWidgetClass, 
	top, args, acount);


    acount = 0;
    XtSetArg(args[acount], XtNresizable, True); acount++;
    box = XtCreateManagedWidget("form", formWidgetClass, shell, args, acount);

    acount = 0;
    XtSetArg(args[acount], XtNresizable, True); acount++;
    XtSetArg(args[acount], XtNleft, XtChainLeft); acount++;
    XtSetArg(args[acount], XtNright, XtChainRight); acount++;
    XtSetArg(args[acount], XtNtop, XtChainTop); acount++;
    XtSetArg(args[acount], XtNbottom, XtChainBottom); acount++;
    XtSetArg(args[acount], XtNheight, 400); acount++;

    XtSetArg(args[acount], XtNscrollVertical, 
	XawtextScrollWhenNeeded); acount++;
    XtSetArg(args[acount], XtNstring, fileName); acount++;
    XtSetArg(args[acount], XtNtype, XawAsciiFile); acount++;
    XtSetArg(args[acount], XtNdisplayCaret, False); acount++;
    XtSetArg(args[acount], XtNresize, XawtextResizeWidth); acount++;
    data = XtCreateManagedWidget("data", asciiTextWidgetClass, box, 
	args, acount);
	
    acount = 0;
    XtSetArg(args[acount], XtNfromVert, data); acount++;
    XtSetArg(args[acount], XtNhorizDistance, 200); acount++;
    XtSetArg(args[acount], XtNtop, XtChainBottom); acount++;
    XtSetArg(args[acount], XtNbottom, XtChainBottom); acount++;
    XtSetArg(args[acount], XtNshapeStyle, XawShapeOval); acount++;
    cont =  XtCreateManagedWidget("Continue", commandWidgetClass, box,
	args, acount);


    XtAddCallback(cont, XtNcallback, (XtCallbackProc) DumpProduct, shell);
    
    XtRealizeWidget(shell);
    hints.x = xwa.width/2;
    hints.y = xwa.height/2;
    hints.flags = USPosition;
    /* odd ? */
    XtCallActionProc(data, "redraw-display", &event, NULL, 0);
    XSetNormalHints(d, XtWindow(shell), &hints);
    XtPopup(shell, XtGrabNone);
}

/* ARGSUSED */
void DisplayStationCallBack(Widget buttonWidget, int display, caddr_t cdata)
{
    char *file;
    char str[80];
    char *product;
    struct report *r;
    char request[80];
    extern char *XawToggleGetCurrent();

    XtPopdown(info.dialog);
    if (display) {
	if (index(stationName, '-')) {
	    /* We got a real product request */
	    file = doprod(stationName);
	    strcpy(request, stationName);
	} else {
	    r = (struct report *) FindStationByNameOrId(stationName);
	    if (!r) return;
	    product = (char *) XawToggleGetCurrent(info.toggle);
	    if (!product || !strcmp(product, "CurrentConditions")) {
		sprintf(request, "Current conditions for %s, %s", 
		    r->name, r->state);
		file = (char *) report(r->id);
	    } else {
		/* see if we know a better station for this product */
		r = (struct report *) FindNearestStationWithProduct(r, product);
		sprintf(request, "%s-k%s", product, r->id);
		file = doprod(request);
	    }
	}
        if (file) {
	   DisplayProduct(file);
	   (void) unlink(file);
	} else {
	   sprintf(str, "%s not available\n", request);
	   ErrorWindow(str);
	}
    }
}

/* ARGSUSED */
void pointProductCallBack(Widget w, char *product, caddr_t cdata)
{
    char *file;
    char str[80];
    struct report *r;
    char request[80];

    r = nearestStation;
    if (!product || !strcmp(product, "CurrentConditions")) {
	sprintf(request, "Current conditions for %s, %s",
	    r->name, r->state);
	file = (char *) report(r->id);
    } else {
	/* see if we know a better station for this product */
	r = (struct report *) FindNearestStationWithProduct(r, product);
	sprintf(request, "%s-k%s", product, r->id);
	file = doprod(request);
    }
    if (file) {
       DisplayProduct(file);
       (void) unlink(file);
    } else {
       sprintf(str, "%s not available\n", request);
       ErrorWindow(str);
    }
}

/* ARGSUSED */
void ChangeDisplayCallBack(Widget buttonWidget, int change, caddr_t cdata)
{

    XtPopdown(info.displaySheet);
    if (change) {
	memcpy(pref.disp_list, tempDispOptions, NR_OPTIONS);
	resizePixmap(); 
    } 
}

/* ARGSUSED */
void CloseHelp(Widget continueButton, Widget shell, caddr_t cdata)
{
    XtPopdown(shell);
    XtUnmapWidget(shell);
}

void button_release(Widget w, rubber_band_data *data, XButtonEvent *event)
{
    double dlat1, dlon1;
    double dlat2, dlon2;
    double scaleX, scaleY;

    if (event->button != Button2) return;

    end_rubber_band(w, data, event);

    xyToLatLon(data->start_x, data->start_y, &dlat1, &dlon1);
    xyToLatLon(event->x, event->y, &dlat2, &dlon2);

    if ((fabs((double) dlon2 - (double) dlon1) < 1.0) ||
        (fabs((double) dlat2 - (double) dlat1) < 1.0)) {
	/* too small to display */
	help_window(NULL, "Area too small", NULL);
	return;
    }
    help_window(NULL, "", NULL);

    center_lon = (dlon1 + dlon2)/2;
    center_lat = (dlat1 + dlat2)/2;

    scaleX = fabs((double) (data->start_x - event->x))/xwa.width * scale;
    scaleY = fabs((double) (data->start_y - event->y))/xwa.height * scale;
    if (scaleX > scaleY) 
	scale = scaleX;
    else
	scale = scaleY;

    radc = 1.866 * a * tan((90. - center_lat) * drc / 2.);
    resizePixmap();
}

void help_func()
{

    FILE *fp;
    int acount;
    Widget cont;
    char str[80];
    Arg args[100];
    Widget box, data;
    XSizeHints hints;
    static int manInit;
    static Widget manShell;


    if (!manInit) {
	acount = 0;
	XtSetArg(args[acount], XtNallowShellResize, True); acount++;
	XtSetArg(args[acount], XtNx, xwa.width/2); acount++;
	XtSetArg(args[acount], XtNy, xwa.height/2); acount++;
	XtSetArg(args[acount], XtNsaveUnder, True); acount++;
	XtSetArg(args[acount], XtNiconPixmap, icon_pixmap); acount++;
	manShell = XtCreatePopupShell("Manual", topLevelShellWidgetClass, 
	    top, args, acount);


	acount = 0;
	XtSetArg(args[acount], XtNresizable, True); acount++;
	box = XtCreateManagedWidget("form", formWidgetClass, manShell, args, 
	    acount);

	acount = 0;
	XtSetArg(args[acount], XtNresizable, True); acount++;
	XtSetArg(args[acount], XtNleft, XtChainLeft); acount++;
	XtSetArg(args[acount], XtNright, XtChainRight); acount++;
	XtSetArg(args[acount], XtNtop, XtChainTop); acount++;
	XtSetArg(args[acount], XtNbottom, XtChainBottom); acount++;
	XtSetArg(args[acount], XtNheight, 400); acount++;

	fp = fopen(pref.helpFile, "r");
	if (!fp) {
	    sprintf(str, "Manual page %s not found\n", pref.helpFile);
	    ErrorWindow(str);
	    return;
	}
	XtSetArg(args[acount], XtNfile, fp); acount++;
	data = XtCreateManagedWidget("manual", scrollByLineWidgetClass, box,
	    args, acount);
	
	acount = 0;
	XtSetArg(args[acount], XtNfromVert, data); acount++;
	XtSetArg(args[acount], XtNhorizDistance, 200); acount++;
	XtSetArg(args[acount], XtNtop, XtChainBottom); acount++;
	XtSetArg(args[acount], XtNbottom, XtChainBottom); acount++;
	XtSetArg(args[acount], XtNshapeStyle, XawShapeOval); acount++;
	cont =  XtCreateManagedWidget("Continue", commandWidgetClass, box,
	    args, acount);


	XtAddCallback(cont, XtNcallback, (XtCallbackProc) CloseHelp, manShell);
	XtRealizeWidget(manShell);

	manInit = TRUE;

	hints.x = xwa.width/2;
	hints.y = xwa.height/2;
	hints.flags = USPosition;
	XSetNormalHints(d, XtWindow(manShell), &hints);
    }
    
    XtPopup(manShell, XtGrabNone);
}

/* map height parameter back to heightWidgets index */
int heightToIndex[] = { 0, -1, 5, 4, -1, 3, -1, 2, 1 };

/* ARGSUSED */
void SetHeight(Widget menuWidget, int height, caddr_t cdata)
{
    int change;
    int prevHeight;
    static struct preference prev;

    /* update check */
    XtSetValue(heightWidgets[heightToIndex[pref.height]], XtNleftBitmap, 
	(char *) None);
    XtSetValue(menuWidget, XtNleftBitmap, (char *) check_pixmap);

    prevHeight = pref.height;
    change = pref.height !=  height;
    pref.height = height;

    if (change) {
	if (prevHeight == 0) {
	    /* if going to UA data enable iso data */
	    prev = pref;
	    pref.disp_list[SHO_ISOBAR] = TRUE;
	    pref.disp_list[SHO_ISOTHERM] = TRUE;
	} else if (height == 0) {
	    /* going back to surface, restore previous iso data */
	    pref.disp_list[SHO_ISOBAR] = prev.disp_list[SHO_ISOBAR];
	    pref.disp_list[SHO_ISOTHERM] = prev.disp_list[SHO_ISOTHERM];
	}
	resizePixmap();
    }
}

/* ARGSUSED */
void DisplaySummary(Widget menuWidget, char *product, caddr_t cdata)
{
    char *file;
    char str[80];

    file = doprod(product);
    if (file) {
       DisplayProduct(file);
       (void) unlink(file);
    } else {
       sprintf(str, "%s not available\n", product);
       ErrorWindow(str);
    }
}
