/*

		    Copyright (c) 1988 	
    Jeff Hollingsworth, KJ Pires, UC Berkeley XCF

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of UC Berkeley not be used
in any advertising or publicity relating to this software
without specific, written prior authorization.  No representations 
are made about the suitability of this software for any purpose.  
It is provided "as is" without express or implied warranty.

*/

#ifndef lint
static char rcsid[] = "@(#) $Header: /home/poona/hollings/src/nwxmap/RCS/distance.c,v 1.1 1991/12/10 16:51:29 hollings Exp $";
#endif

/*
 * $Log: distance.c,v $
 * Revision 1.1  1991/12/10  16:51:29  hollings
 * Initial revision
 *
 */


/* 
 * 
 * by: J.K. Hollingsworth (4/16/88).
 *
 * distance.c:  This file contains two routines to compute the 
 *   distance between two points given their latitude, and longitude.
 *  
 * gc_distance computes the distance using the great circle model
 *   of the earth.
 *
 * distance compute the distance using a more accurate model.
 *
 * Both routines expect the input to be a set of four longs 
 *  lat1, long1, lat2, long2 in that oerder.  These numbers are
 *  fixed point reals.  The floating point value can be computed
 *  by dividing the number by the constant FIXED_TO_FLOAT.
 *
 * The return value is a double which is the distance in meters
 *  between the two points.
*/
#include <math.h>

#ifndef M_PI
#define M_PI 3.1415925
#define M_PI_4 M_PI * 4
#endif


#define	FIXED_TO_FLOAT	10000.0
#define AVG_DEGREE	111300 	/* kilometers that is */
#define CONV_TO_RAD	M_PI / (FIXED_TO_FLOAT * 180.0)

/* gc_distance uses the great circle formula to compute the distance
   between two points. 
*/
double gc_distance(l_lat1, l_long1, l_lat2, l_long2)
    long l_lat1, l_long1, l_lat2, l_long2;

{
    register double diff_long, cos_d, d, r_lat1, r_long1;
    register double r_lat2, r_long2;

    r_lat1 = l_lat1 * CONV_TO_RAD;
    r_long1 = l_long1 / FIXED_TO_FLOAT;
    r_lat2 = l_lat2 * CONV_TO_RAD;
    r_long2 = l_long2 / FIXED_TO_FLOAT;

    diff_long = (r_long2 - r_long1) * M_PI / 180.0;
    cos_d = sin(r_lat1) * sin(r_lat2) + 
	    cos(r_lat1) * cos(r_lat2) * cos(diff_long);

    d = acos(cos_d) * 180.0 / M_PI;
    return(d * AVG_DEGREE);
}

/* This version of the distance between two points on the earth
   is based on the formula given in Appendix A of Sodano, E.M.,
   "General non-iterative solution of the inverse and direct geodetic
   problems", Bulletin Geodesique, vol 75 (March 1965), pp 69-84.
*/
#define a0	6378388.000
#define b0      6356911.946
#define f	(1 - (b0 / a0))
#define cot(x)  (1.0 / tan(x))
#define acot(x) (atan(1.0 / (x)))
#define csc(x)  (1.0 / sin(x))
#define sqr(x) ((x) * (x))

double distance(l_lat1, l_long1, l_lat2, l_long2)
    long l_lat1, l_long1, l_lat2, l_long2;

{
    double r_lat1, r_long1;
    double beta1, beta2, r_lat2, r_long2;
    double l, a, b, phi, sin_phi, c, m, temp, S;

    r_lat1 = l_lat1 * CONV_TO_RAD;
    r_long1 = l_long1 / FIXED_TO_FLOAT;
    r_lat2 = l_lat2 * CONV_TO_RAD;
    r_long2 = l_long2 / FIXED_TO_FLOAT;

    l = (r_long2 - r_long1) * M_PI / 180.0;
    
    if (fabs(r_lat2) <= M_PI_4)
	beta2 = atan(tan(r_lat2) * (1 - f));
    else {
	temp = cot(r_lat2) / (1 - f);
	beta2 = acot(temp);
    }

    if (fabs(r_lat1) <= M_PI_4)
	beta1 = atan(tan(r_lat1) * (1 - f));
    else {
	temp = cot(r_lat1) / (1 - f);
	beta1 = acot(temp);
    }
	
    /* some intermediate */
    a = sin(beta1) * sin(beta2);
    b = cos(beta1) * cos(beta2);
    phi = acos(a + b * cos(l));
    /* note the fabs(sin(phi)) is to make sure we get the shorter
       arc */
    sin_phi = fabs(sqrt(sqr(sin(l) * cos(beta2)) +
			sqr(sin(beta2) * cos(beta1) - 
			    sin(beta1) * cos(beta2) * cos(l))));
    c = (b * sin(l)) / sin_phi;
    m = 1 - sqr(c);

    /* temp == S / b0 */
    temp = ((1 + f + sqr(f)) * phi) +
	   a * ((f + sqr(f)) * sin_phi - 
		(sqr(f) / 2.0) * sqr(phi) * csc(phi)) +
	   m * (-1.0 * ((f + sqr(f)) / 2.0) * phi - 
	       (f + sqr(f))/ 2.0 * sin_phi * cos(phi) + 
	       (sqr(f) / 2.0 * sqr(phi) * cot(phi))) +
	   sqr(a) * (-0.5 * sqr(f) * sin_phi * cos(phi)) +
	   sqr(m) * (sqr(f) / 16.0 * phi + 
		     sqr(f) / 16.0 * sin_phi * cos(phi) -
		     sqr(f) / 2.0 * sqr(phi) * cot(phi) -
		     sqr(f) / 8.0 * sin_phi * pow(cos(phi), 3.0)) +
	   a * m * (sqr(f) / 2.0 * sqr(phi) * csc(phi) +
		    sqr(f) / 2.0 * sin_phi * sqr(cos(phi)));
    S = b0 * temp;
    return(S);
}
