/*
** WXMAP - Draw surface weather maps on an X display.
**
** Charley Kline, University of Illinois Computing Services Office.
** 2 Mar 1990.
** 3 Oct 1990.
**
** Charley Kline, University of Illinois Computing Services Office.
** Some code contributed by William Kucharski.
** 14 Feb 1991.
**
** Wxmap reads a file containing station location and current weather
** conditions from a weather server machine (garcon.cso.uiuc.edu) into
** memory and then allows this information to be displayed both on the
** terminal and in the form of a NWS Surface Analysis Map on an X
** workstation.
**
** Maps can be printed using xwd and xpr.
**
** This program is under serious construction. Next:
**    Keep the keyboard alive during map display.
**    More interaction with the mouse and the map.
**
 * Copyright (C) 1990 by the University of Illinois Board of Trustees.
 * 
 * This code is distributed in the hope that it will be useful,
 * but without any warranty.  No author or distributor accepts
 * responsibility to anyone for the consequences of using it or for
 * whether it serves any particular purpose or works at all, unless
 * s/he says so in writing.
 * 
 * Everyone is granted permission to copy, modify and redistribute
 * this code under the following conditions:
 * 
 *    Permission is granted to anyone to make or distribute copies
 *    of program source code, either as received or modified, in any
 *    medium, provided that all copyright notices, permission and
 *    nonwarranty notices are preserved, and that the distributor
 *    grants the recipient permission for further redistribution as
 *    permitted by this document, and gives him and points out to
 *    him an exact copy of this document to inform him of his rights.
 * 
 *    Permission is granted to distribute this code in compiled
 *    or executable form under the same conditions applying for
 *    source code, provided that either
 *    A. it is accompanied by the corresponding machine-readable
 *       source code, or
 *    B. it is accompanied by a written offer, with no time limit,
 *       to give anyone a machine-readable copy of the corresponding
 *       source code in return for reimbursement of the cost of
 *       distribution.  This written offer must permit verbatim
 *       duplication by anyone.
 *    C. it is distributed by someone who received only the
 *       executable form, and is accompanied by a copy of the
 *       written offer of source code which he received along with it.
 * 
 * In other words, you are welcome to use, share and improve this
 * code.  You are forbidden to forbid anyone else to use, share
 * and improve what you give them.   Help stamp out software-hoarding!
 */

#ifndef lint
static char rcsid[] = "@(#) $Header: /usr/home/hollings/src/original/xweather/RCS/main.c,v 1.10 1992/12/17 01:17:20 hollings Exp hollings $";
#endif

/*
 * $Log: main.c,v $
 * Revision 1.10  1992/12/17  01:17:20  hollings
 * SYSV port.
 *
 * Revision 1.9  1992/12/16  19:24:33  hollings
 * fixes to compile with gcc.
 *
 * Revision 1.8  1992/08/27  15:15:07  hollings
 * made it version 1.1.
 *
 * Revision 1.7  1992/08/10  21:31:44  hollings
 * supress printing specials for historic data.
 *
 * Revision 1.6  1992/07/02  19:39:23  hollings
 * Color codeded watch boxes.
 *
 * Revision 1.5  1992/05/05  23:27:40  hollings
 * Version 1.0.
 *
 * Revision 1.4  1992/02/11  17:56:49  hollings
 * historic fix (again)
 *
 * Revision 1.3  1992/01/22  20:42:49  hollings
 * historic data support.
 *
 * Revision 1.2  1991/12/13  00:30:48  hollings
 * Made LIBDIR a resource.
 *
 * Revision 1.1  1991/12/10  16:51:29  hollings
 * Initial revision
 *
 */


#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include <stdlib.h>
#include <X11/cursorfont.h>
#include <X11/Intrinsic.h>
#include <errno.h>
#include <unistd.h>

#include "sadecode.h"
#include "map.h"
#include "patchlevel.h"
#include "io.h"

#ifndef isgraph
#define isgraph(x) ((x) != ' ')
#endif

#define CVT(var, offset)	((var[offset] - '0') * 10 + var[offset+1] - '0')
#define BASE_TIME	16
#define RANDOM_INTERVAL	6

XWindowAttributes xwa;
char programName[80];
char serverVersion[80];
GC gc = (GC) 0;
GC copygc0 = (GC) 0;
GC copygc1 = (GC) 0;
XFontStruct *fs, *tfs, *ifs, *lfs;
Cursor crosshair = (Cursor) 0;
Cursor watch = (Cursor) 0;
Display *d;
Pixmap w;
Window rw = (Window ) 0;
Window xwin;
FILE *netin, *netout;

char *convertTime();

/* 
** Defaults.
*/
char *display_name = NULL;			/* X display name */

/*
** Command list structure.
*/
extern void cmd_defregion(bool echo);
extern void cmd_watch(bool echo);
extern void cmd_decode(bool echo);
extern void cmd_title(bool echo);
extern void cmd_note(bool echo);
extern void cmd_title(bool echo);
extern void cmd_xyzzy(bool echo);
extern void cmd_root(bool echo);
extern void cmd_help(bool echo);

struct cmdentry {
    char *c_name;		/* ASCII command name */
    void (*c_func)(bool);	/* Pointer to handler, NULL if unimplemented */
};


static struct cmdentry cmdtab[] =  {
    {"DECODE",		cmd_decode},
    {"DEFREGION",	cmd_defregion},
    {"NOTE",            cmd_note},
    {"HELP",		cmd_help},
    {"TITLE",		cmd_title},
    {"WATCHAREA",	cmd_watch},
    {"ROOT",		cmd_root},
    {"XYZZY",		cmd_xyzzy},

    {NULL, NULL},
};




/*
** Array to hold the SA database.
*/
#define MAXSTATIONS 2000
char title[128];			/* Title string from database */
char *note[20];				/* Notes to draw on map legend */
int nnotes;				/* Number of notes */
struct report rr[MAXSTATIONS];		/* Array of reports for all stations */
int nrr;				/* Number of reports in database */
extern int nregions;
extern char *bisearch();
extern struct report *sadecode();

/*
** Map parameters.
*/
double scale,			/* Map scale factor */
       center_lon, center_lat;	/* Coordinates of center of map */
int maxprio;			/* Maximum WMO station priority to display */

int on_root = 0;		/* Draw on root window? */


/*
 * alarm for auto-rescan has gone off.
 *
 */
void reScanAlarm()
{
    /* schedule alarm for next 1/2 hour */
    alarm(1800);
    refetch_func();
}

/*
 * return a random number between 0 and 60 * RANDOM_INTERVAL.
 *
 */
int randomTime()
{
    int n;

    srand(getpid());
    n = rand() % (RANDOM_INTERVAL * 60);
    return(n);
}

int Argc;
char **Argv;
XtAppContext appConn;
Widget top;
extern int errorHandler();
struct timeval timeAtFetch;
struct timeval lastUaTime;


void load_reports(int detail) 
{
    FILE *fd;
    time_t time;
    int nrpt = 0;
    char stid[6];
    char buf[4096];
    struct tm *tim;
    char request[80];
    struct report *r = rr;
    struct report *findstn();
    struct cmdentry *cmdptr, *wordsym();


    /*
     * initialize variables.
     *
     */
    nnotes = 0;
    nregions = 0;
    if (detail == FIRST_LOAD) {
	nrr = 0;
	sprintf(buf, "%s/city.sort", pref.libDirectory);
	if ((fd = fopen(buf, "r")) == NULL) {
	    fprintf(stderr, "unable to open file %s/city.sort\n", 
		pref.libDirectory);
	    exit(1);
	    }
	printf("Loading from disk... ");
	fflush(stdout);
	while (fgets(buf, sizeof buf, fd) != NULL) {
	    char *q, *p = buf;
	    q = strchr(p, ':'); *q = '\0';
	    strcpy(r->id, p);
	    p = q + 1; q = strchr(p, ':'); *q = '\0';
	    strcpy(r->name, p);
	    p = q + 1; q = strchr(p, ':'); *q = '\0';
	    strcpy(r->state, p);
	    p = q + 1; q = strchr(p, ':'); *q = '\0';
	    r->prio = atoi(p);
	    p = q + 1; q = strchr(p, ':'); *q = '\0';
	    r->lat = atof(p);
	    p = q + 1; q = strchr(p, ':'); *q = '\0';
	    r->lon = atof(p);
	    p = q + 1; q = strchr(p, ':'); if (q) *q = '\0';
	    r->elev = atoi(p);
	    if (q) {
		p = q + 1;
		q = strchr(p, '\n'); if (q) *q = '\0';
		strcpy(r->wmo, p);
		}
	    else r->wmo[0] = '\0';
	    r->time[0] = 'M';   /* Assume missing report */
	    r->raw[0] = '\0';
	    r++, nrr++;
	    }
	fclose(fd);
        connectToServer();
    }

    /*
    ** Request the hourly sequence reports.
    */
    sprintf(request, "SA %s\n", picturetime);
    requestServerData(request);
    readServerData(buf, sizeof buf);
    if (buf[0] != '1') {
	/* try previous hour just in case */
	if (!pref.epochTime) {
	    time = timeAtFetch.tv_sec-3600;
	    tim = gmtime(&time);
	    sprintf(picturetime, "%02d%02d%02d%02d",
		tim->tm_year,tim->tm_mon+1,tim->tm_mday, tim->tm_hour);
	    sprintf(request, "SA %s\n", picturetime);
	    requestServerData(request);
	    readServerData(buf, sizeof buf);
	    if (buf[0] != '1') {
		fprintf(stderr, "\n!!! Not available: %s", buf);
		exit(1);
	    }
	} else {
	    fprintf(stderr, "\n!!! Not available: %s", buf);
	    exit(1);
	}
    }
    
    /*
    ** Load and convert the sequence reports.
    */
    if (detail == FIRST_LOAD) {
	printf("Loading from network...");
    }
    fflush(stdout);
    while (readServerData(buf, sizeof buf) != NULL) {
	int i, l = strlen(buf);
	if (buf[0] == '.' && buf[1] == '\n') break;
	while (strchr("\r\n ", buf[--l])) ;
	buf[++l] = '\0';
	stid[3] = '\0';
	for (i = 0; i < 3; i++) stid[i] = buf[i];
	if ((r = findstn(stid)) == NULL) continue;
	strncpy(r->raw, buf, 79); r->raw[79] = '\0';
	if (sadecode(buf, r) == NULL) {
	    r->time[0] = 'M';   /* decode failed, so make it missing again */
	    continue;
	    }
	nrpt++;
    }
    if (buf[0] != '.') {
	fprintf(stderr, "\n!!! Connection severed.\n");
	exit(1);
	}
    if (detail == FIRST_LOAD) {
	/* printf("\n*** %d stations (%d reporting).\n", nrr, nrpt); */
    }

    /*
    ** Request special information.
    */
    if (pref.epochTime) {
	note[nnotes++] = "*** No Special Statements available for past weather";
    } else {
	requestServerData("SPECIALS\n");
	readServerData(buf, sizeof buf);
	if (buf[0] == '1') {
	    while (1) {
		if (getsyms() == EOF) break;
		if (symv[0][0] == '.') break;
		if ((cmdptr = wordsym(symv[0])) == NULL) continue;
		if (cmdptr->c_func)
		    (*cmdptr->c_func)(FALSE);
	    }
	    if (symv[0][0] != '.') {
		fprintf(stderr, "!!! Connection severed.\n");
		exit(1);
	    }
	}
    }

	/* 
	 * load directory of where the product are available.
	 *
	 */
    if (detail == FIRST_LOAD) {
	loadProductDirectory("fe", FE_MASK);
	loadProductDirectory("fq", FQ_MASK);
	loadProductDirectory("fp", FP_MASK);
    }
}

void setPictureTime()
{
    struct tm *tim;

    gettimeofday(&timeAtFetch, 0);
    /* reports don't come in until BASE_TIME */
    timeAtFetch.tv_sec -= BASE_TIME * 60;

    /* check to see if a specific time was requested */
    if (pref.epochTime) {
	printf("Using historic data %s\n", pref.epochTime);
	if (strlen(pref.epochTime) != 8) {
	    XtWarning("Invalid Epoch Time");
	} else {
	    strcpy(picturetime, pref.epochTime);
	}
    } else {
	tim = gmtime(&timeAtFetch.tv_sec);
	sprintf(picturetime, "%02d%02d%02d%02d",
	    tim->tm_year,tim->tm_mon+1,tim->tm_mday, tim->tm_hour);
	/*
	** Upper air data only comes in every twelve hours, so create
	** the picturetime for upper air data from the hourly picturetime
	** by rounding down to the nearest 00Z or 12Z boundary.
	*/
	lastUaTime.tv_sec = timeAtFetch.tv_sec;
	/* subtract time from last 12 hour inteval */
	lastUaTime.tv_sec -= timeAtFetch.tv_sec % (3600 * 12);
	tim = gmtime(&lastUaTime.tv_sec);
	sprintf(upperairtime, "%02d%02d%02d%02d",
	    tim->tm_year,tim->tm_mon+1,tim->tm_mday, tim->tm_hour);
    }
}

void main(int argc, char *argv[]) 
{
    int alarmTime;
    Widget init_x();
    struct cmdentry *wordsym();

    Argc = argc;
    Argv = argv;

    /* ignore sigpipe, since we check all writes */
    signal(SIGPIPE, SIG_IGN);
    strcpy(programName, argv[0]);

    printf("Weather disseminator. **Version 1.2**\n");
    puts("Please send comments to hollings@cs.wisc.edu.");

    /* Default inital map position */
    maxprio = 2;
    center_lat = 39.;
    center_lon = 97.;
    scale = 2.3;

    /*
    ** Initialize the X window system. This also ends up parsing
    ** the command line and the user's defaults. It may fail to
    ** open the display, in which case it prints its own error messages
    ** and leaves _d_ NULL so that we know we are running without X.
    */
    (void) init_x();


    setPictureTime();

    /* printf("picture time = %s\n", picturetime);
    printf("upperairtime time = %s\n", upperairtime); */

    /* Define the Error handler */
    XSetErrorHandler(errorHandler);
    XSetIOErrorHandler(errorHandler);

    /*
     * Must do this after init_x incase serverHost, or serverPort resource is
     *   set.
     */
    load_reports(FIRST_LOAD);

    buildWidgets(top);
    if (crosshair == (Cursor) 0)
	crosshair = XCreateFontCursor(d, XC_crosshair);

    if (watch == (Cursor) 0)
	watch = XCreateFontCursor(d, XC_watch);

    xwa.height = 0;
    xwa.width = 0;

    /*
     * Set auto-rescan if desired.
     */
    if (pref.autoRescan) {
	signal(SIGALRM, reScanAlarm);
	alarmTime = 60 * 60 - timeAtFetch.tv_sec % 3600 + randomTime();
	alarm(alarmTime);
    }

    makeTitleLine();

    /*
    ** A short and sweet main loop.
    */
    XtAppMainLoop(appConn);
}




/*
** This only works because the _id_ field comes first in struct report!
**
** And because the city.sort file is sorted, so crdb writes the reports
** out in order, so we read them in in order, so they end up in the _rr_
** array in order.
*/
struct report *findstn(s) 
char *s;
{
    return (struct report *) bisearch(s, (char *)rr, nrr,
	sizeof (struct report), strcmp);
    }


/* _pat_ is a string of slash-terminated alternatives. Return true
** if _s_ matches one of the alternatives, false otherwise.
*/ 
int oneof(char *s, char *pat) 
{
    register char *p = pat, *q;
    register int slen = strlen(s);

    while (q = strchr(p, '/')) {
	if (q - p == slen && strncmp(s, p, slen) == 0) return(TRUE);
	p = q + 1;
	}
    return (FALSE);
}


int loadProductDirectory(name, code)
char *name;
int code;
{
    FILE *fp;
    char buf[256];
    char station[40];
    struct report *st;

    sprintf(buf, "%s/%s.dir", pref.libDirectory, name);
    fp = fopen(buf, "r");
    if (!fp) {
	perror(buf);
	return(-1);
    }
    while (!feof(fp)) {
	fscanf(fp, "%s\n", station);
	if (station[0] == '.') break;
	st = findstn(station);
	st->products |= code;
    }
    fclose(fp);
    return(0);
}

int parmscan(char **p)
{
    char *q;

    q = *p;
    while (*q && isgraph(*q)) q++;
    if (*q == '\0') return 0;
    *q++ = '\0';
    while (*q && isspace(*q)) q++;
    if (*q == '\0') return 0;
    *p = q;
    return 1;
}




/* Find the command table entry whose name is _s_ and return a pointer
** to it. The command in _s_ is modified by converting it to lower case
** first. Unambiguous abbreviations of valid commands are recognized.
** Wordsym() prints its own error messages.
*/
struct cmdentry *wordsym(char *s) 
{
    struct cmdentry *p, *q;
    int matches = 0;
    int slen = strlen(s);

    for (p = cmdtab; p->c_name; p++)
	if (strncmp(p->c_name, s, slen) == 0) {
	    q = p;
	    matches++;
	    }

    switch (matches) {
	case 0:
	    printf("%s Command unrecognized.\r\n", s);
	    return NULL;
	
	case 1:
	    return q;
	
	default:
	    printf("%s: Ambiguous.\r\n", s);
	    return NULL;
	}
}


void cmd_title(bool echo)
{
    int i;

    title[0] = '\0';
    for (i = 1; i < symc; i++) {
	strcat(title, symv[i]);
	strcat(title, " ");
	}
}


void cmd_note(bool echo) 
{
    int i, sl;
    char buf[128];

    buf[0] = '\0';
    for (i = 1; i < symc; i++) {
        strcat(buf, " ");
	if (isdigit(*symv[i])) {
	    strcat(buf, convertTime(symv[i]));
	} else {
	    strcat(buf, symv[i]);
	}
    }
    /* dump version info string */
    if (!strncmp(buf, " Wxmap", 6)) {
	strcpy(serverVersion, buf);
	return;
    }
    sl = strlen(buf);
    note[nnotes] = (char *) malloc(sl);
    strcpy(note[nnotes++], buf + 1);
}

char *convertTime(char *zulu)
{
    char *t2;
    char time1[80];
    char time2[80];
    static char outTime1[80];
    static char outTime2[80];
    struct timeval tv;
    struct timezone tz;

    if (pref.gmtTime) return(zulu);
    gettimeofday(&tv, &tz);
    if (t2 = strchr(zulu, '-')) {
	/* time range */
	strcpy(time1, &zulu[2]);
	strcpy(time2, t2+2);
	dolocaltime(time1, &tv, &tz, NULL, outTime1);
	dolocaltime(time2, &tv, &tz, NULL, outTime2);
	if (outTime2[0] > '2') {
	    /* expired */
	    strcpy(outTime1, "(EXPIRED)");
	} else {
	    strcat(outTime1, " to ");
	    strcat(outTime1, outTime2);
	}
    } else {
	strcpy(time1, &zulu[2]);
	dolocaltime(time1, &tv, &tz, NULL, outTime1);
	if (outTime1[0] > '2') {
	    /* expired */
	    strcpy(outTime1, "(EXPIRED)");
	}
    }
    return(outTime1);
}



void cmd_help(bool echo)
{
    char *pager, *getenv();
    char buf[64];

    if ((pager = getenv("PAGER")) == NULL)
#ifdef hpux
	pager = "/usr/bin/more";
#else
	pager = "/usr/ucb/more";
#endif
    sprintf(buf, "%s %s/helpfile", pager, pref.libDirectory);
    system(buf);
}


void cmd_xyzzy(bool echo)
{
    if (echo) puts("*** Nothing happens.");
}


void cmd_decode(bool echo)
{
    struct report *r;
    int i;

    if (symc < 2) {
	puts("*** Usage: decode STN [...]");
	return;
	}

    for (i = 1; i < symc; i++) {
	if ((r = findstn(symv[i])) == NULL)
	    printf("*** %s not found amongst reports.\n", symv[i]);
	else {
	    printf("*** %s is %s, %s.   Latitude %.2fN, Longitude %.2fW.\n",
		r->id, r->name, r->state, r->lat, r->lon);
	    printf("*** Elevation %dft.", (int)(0.5 + r->elev * 3.280));
	    if (r->wmo[0]) printf("  WMO ID %s", r->wmo);
	    putchar('\n');
	    }
	}
}


struct report *FindStationByNameOrId(char *name)
{
    char *ch;
    char error[80];
    char search[80];
    struct report *ret;
    struct report *r = rr;
    int i, match = 0, slen;

    strcpy(search, name);
    for (ch = search; *ch; ch++) *ch = toupper(*ch);

    r = findstn(search);
    if (r) return(r);

    /*
     * Try by station name now.
     */
    slen = strlen(name);

    for (i = 0, r = rr, ret = NULL; i < nrr; i++, r++) {
	if (!strncasecmp(name, r->name, slen)) {
	    match++;
	    ret = r;
	}
    }
    if (match > 1) {
	sprintf(error, "%s matches more than one station", name);
	ErrorWindow(error);
	return(NULL);
    } else if (match == 0) {
	sprintf(error, "Station %s does not exist", name);
	ErrorWindow(error);
	return(NULL);
    } else {
	return(ret);
    }
}

/*
 * Find the nearest station in the same state that has the currently 
 *    requested product. 
 */
struct report *FindNearestStationWithProduct(struct report *st, char *product)
{
    int i;
    int code;
    struct report *r;
    struct report *ret;
    double newDist, dist, distance();
    long lat1, lat2, lon1, lon2;

    if (!strcmp(product, "fpus1") || 
        !strcmp(product, "wwus35") ||
	!strcmp(product, "fpus5")) {
	code = FP_MASK;
    } else if (!strcmp(product, "feus1") || 
	       !strcmp(product, "sxus90")) {
	code = FE_MASK;
    } else if (!strcmp(product, "fqus1")) {
	code = FQ_MASK;
    } else {
	/* unknown use passed station */
	return(st);
    }

    /* if we have the data don't look elsewhere */
    if (st->products & code) return(st);

    ret = NULL;
    lat1 = st->lat * 3600;
    lon1 = st->lon * 3600;
    for (i = 0, r = rr; i < nrr; i++, r++) {
        if ((r->products & code) && !strcmp(r->state, st->state)) {
	    lat2 = r->lat * 3600;
	    lon2 = r->lon * 3600;
	    newDist = distance(lat1, lon1, lat2, lon2);
	    if ((newDist < dist) || !ret) {
		ret = r;
		dist = newDist;
	    }
	}
    }
    if (ret) {
	return(ret);
    } else {
	return(st);
    }
}

void cmd_root(bool echo)
{
    extern void reset_root_window();

    if (symc > 2) {
	printf("*** Usage: %s [ON|OFF]\n", symv[0]);
	return;
	}

    if (symc == 2) {
	if (!strcmp(symv[1], "ON")) {
	    on_root = 1;
	    reset_root_window();
	} else if (!strcmp(symv[1], "OFF")) {
	    if (rw) {
		reset_root_window();
		rw = (Window) 0;
	    }
	    on_root = 0;
	} else {
	    printf("*** Usage: %s [ON|OFF]\n", symv[0]);
	    return;
	    }
	}

    if (!echo) return;

    if (on_root)
	puts(">>> Drawing maps on the root window.");
    else
	puts(">>> Drawing maps in separate windows.");
    }
    

#ifndef HAS_STRNCASECMP

/*
 * This array is designed for mapping upper and lower case letter
 * together for a case independent comparison.  The mappings are
 * based upon ascii character sequences.
 */
static u_char charmap[] = {
	'\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
	'\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
	'\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
	'\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
	'\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
	'\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
	'\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
	'\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
	'\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
	'\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
	'\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
	'\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
	'\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
	'\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
	'\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
	'\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
	'\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
	'\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
	'\300', '\301', '\302', '\303', '\304', '\305', '\306', '\307',
	'\310', '\311', '\312', '\313', '\314', '\315', '\316', '\317',
	'\320', '\321', '\322', '\323', '\324', '\325', '\326', '\327',
	'\330', '\331', '\332', '\333', '\334', '\335', '\336', '\337',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};


strncasecmp(s1, s2, n) char *s1, *s2; register int n; {
    register u_char	*cm = charmap,
		    *us1 = (u_char *)s1,
		    *us2 = (u_char *)s2;

    while (--n >= 0 && cm[*us1] == cm[*us2++])
	if (*us1++ == '\0') return(0);
    return(n < 0 ? 0 : cm[*us1] - cm[*--us2]);
}

#endif

errorHandler(display, event)
Display *display;
XErrorEvent *event;
{
    char data[80];

    XGetErrorText(display, event->error_code, data, sizeof(data));
    printf("\n\n ***Xerror: %s\n", data);
    printf("using data from %s\n", picturetime);
    printf("dumping core\n");
    abort();
}

