/* 
   drm - Generates .mid drum backing from annotated lyric or chord files

   Copyright (C) 2001 Neil Spring

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

#include <termios.h>
#include <grp.h>
#include <pwd.h>
*/

/* get asprintf */
#define __USE_GNU

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>
#include <assert.h>
#include <string.h>
#include <netinet/in.h>
#include "system.h"
#include "midiout.h"
#include "config.h"
#include "nscommon.h"

void *xmalloc (size_t n);
void *xrealloc (void *p, size_t n);
char *xstrdup (char *p);

/* after xmalloc ... so that the defines in dmalloc.h don't
   break much */
#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

static void usage (int status);

/* The name the program was run with, stripped of any leading path. */
char *program_name;

/* getopt_long return codes */
enum {DUMMY_CODE=129
};

/* Option flags and variables */
static struct option const long_options[] =
{
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'V'},
  {"verbose", no_argument, 0, 'v'},
  {"output", required_argument, 0, 'o'},
  {"input",  required_argument, 0, 'i'},
  {"chordoutput",  required_argument, 0, 'c'},
  {NULL, 0, NULL, 0}
};

static int decode_switches (int argc, char **argv);

/* not static, used by lexer for debug messages */
int verbose;
int embedlyrics;
/* not static, used by lexer for error messages */
char *input_filename = NULL;
FILE *input_file;
static char *output_filename = NULL;
FILE *output_file;
static const char *chord_filename = NULL;
FILE *fchordata;

struct file_header {
  char id[4];
  unsigned int length; /* following the header MSB */
  unsigned short format;
  unsigned short ntrks;
  unsigned short division;
} hdr;


/* text event: FF 01 len text */
/* FF 05 len text Lyric 
   A lyric to be sung. Generally, each syllable will be a
   separate lyric event which begins Bt the event's time.
*/
/* FF 2F 00 End of Track */
/* FF 51 03 tttttt Set Tempo, in microseconds per MIDI quarter-note */

/* this is static enough. */
/* leading zero is the time */
const char end_of_track[] = { 0x00, 0xff, 0x2f, 0x00 };

/* from http://ourworld.compuserve.com/homepages/mark_clay/midi.htm */


extern FILE *yyin;
extern int yyparse(void);
int
main (int argc, char **argv)
{
  int i;
  unsigned int sizepos;
  unsigned int numbytes;

  program_name = argv[0];
  i = decode_switches (argc, argv);

  input_file = fopen(input_filename, "r");
  if(!input_file) {
    perror("couldn't open input file");
    exit(EXIT_FAILURE);
  }

  output_file = fopen(output_filename, "w");
  if(!output_file) {
    perror("couldn't open output file");
    exit(EXIT_FAILURE);
  }
  
  strncpy(hdr.id, "MThd", 4);
  hdr.length = htonl(6);
  hdr.format = htons(0);
  hdr.ntrks  = htons(1);
  hdr.division = htons(120); /* 120 ticks per 1/4 note*/
  assert(fwrite(&hdr, 1, 14, output_file));
  
  fwrite("MTrk", 1, 4, output_file);
  sizepos = ftell(output_file);
  fwrite("XXXX", 4, 1, output_file);


  midi_set_filep(output_file);
  yyin = input_file;
  if(yyparse()) {
    fprintf(stderr, "input parsing failed... or so yyparse would have us believe.\n");
  }

  assert(fwrite(end_of_track, sizeof(end_of_track), 1, output_file));

  /* cleanup the length of the file */
  numbytes = htonl(ftell(output_file) - sizepos - 4);
  if(verbose) printf("numbytes = %u\n", ntohl(numbytes));
  fseek(output_file, sizepos, SEEK_SET);
  fwrite(&numbytes, 4, 1, output_file);

  fclose(output_file);
  fclose(input_file);

  free(output_filename);
  free(input_filename);
  exit (0);
}

/* Set all the option flags according to the switches specified.
   Return the index of the first non-option argument.  */

static int
decode_switches (int argc, char **argv)
{
  int c;

  while ((c = getopt_long (argc, argv, 
                           "h"	        /* help */
                           "V"  	/* version */
                           "v" 	        /* be verbious */
                           "l" 	        /* add lyrics */
                           "i:" 	/* input file name */
                           "o:" 	/* output file name */
                           "c:",	/* chord output file name */
                           long_options, (int *) 0)) != EOF) {
    switch (c) {
    case 'V':
      printf ("drm %s\n", VERSION);
      exit (0);
    case 'h':
      usage (0);
    case 'i':
      input_filename = xstrdup(optarg);
      break;
    case 'o':
      output_filename = xstrdup(optarg);
      break;
    case 'c':
      chord_filename = xstrdup(optarg);
      break;
    case 'l':
      embedlyrics = TRUE;
      break;
    case 'v':
      verbose = 1;
      break;
    default:
      usage (EXIT_FAILURE);
    }
  }
  
  if (optind < argc) {
    if(input_filename == NULL) 
      input_filename = xstrdup(argv[optind++]);
    while (optind < argc) {
      printf ("non-option ARGV-element: ");
      printf ("%s \n", argv[optind++]);
    }
  }

  if(output_filename == NULL) {
    if(input_filename == NULL) {
      usage(EXIT_FAILURE);

    }
    output_filename = malloc(strlen(input_filename)+5);
    sprintf((char *)output_filename, "%s.mid", input_filename);
  } 
  if(chord_filename != NULL) {
    fchordata = fopen(chord_filename, "w");
  }
  return optind;
}


static void
usage (int status) {
  printf (_("%s - \
Generates .mid drum backing from annotated lyric or chord files\n"), program_name);
  printf (_("Usage: %s [OPTION]... [FILE]...\n"), program_name);
  printf (_("\
Options:\n\
  -h, --help                 display this help and exit\n\
  -V, --version              output version information and exit\n\
  -i, --input                file to read\n\
  -o, --output               .mid file to write\n\
  -c, --chordoutput          .crd file to write\n\
"));
  exit (status);
  }
