#include <config.h>
#include "sounds.h"
#include "midiout.h"
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

#define MAX_VELOCITY (127U)

extern int embedlyrics;
int bpm_has_been_set;
extern boolean verbose;
boolean unstuck; /* do we have to re-initialize ourselves after raw midi */

static FILE *fdout;
static unsigned long leftover_delta_t;
int s_volume = 60;

void midi_set_filep(FILE *output) {
  fdout = output;
}

void midi_write_var(unsigned long value) {
  unsigned long buffer;

  assert(fdout!=NULL);

  buffer = value & 0x7f;
  while ((value >>= 7) > 0) {
    buffer <<= 8;
    buffer |= 0x80;
    buffer += (value & 0x7f);
  }
  while (TRUE) {
    (void)fputc((int)buffer,fdout);
    if ((buffer & 0x80) != 0)
      buffer >>= 8;
    else
      break;
  }
}

void midi_set_volume(int volume) {
  s_volume = volume;
}

void midi_set_bpm(int bpm) {
  unsigned int usec_per_beat;
  /*@+charint@*/
  char set_tempo[] = { 0xff, 0x51, 0x03, 0x00, 0x00, 0x00 };
  /*@-charint@*/
  assert(fdout!=NULL);

  midi_write_var(leftover_delta_t);
  leftover_delta_t = 0;
  usec_per_beat = 60000000U/(bpm);
  set_tempo[3] = (char)((usec_per_beat & 0xff0000) >> 16);
  set_tempo[4] = (char)((usec_per_beat & 0x00ff00) >> 8);
  set_tempo[5] = (char)((usec_per_beat & 0x0000ff));

  bpm_has_been_set = TRUE;
  (void)fwrite(set_tempo, 6, 1, fdout);
  midi_zero_volume_note();
}

/* http://www.indiana.edu/~emusic/chvcmess.html */
/* midi channel 10, establish running status is default. */
/* running status means we don't have to say which channel anymore */
/*@+charint@*/
const char zero_volume_note[] = { 0x00, /* time */
                                  0x99, /* note on (9), channel 10 (9) */
                                  0x11, /* data byte 1, k=17 */
                                  0x00 };  /* velocity = 0 */
/*@-charint@*/
void midi_zero_volume_note(void) {
  (void)fwrite(zero_volume_note, 4, 1, fdout);
}

unsigned char midi_lookup_instrument(const char *inst) {
  size_t l = strlen(inst);
  struct sound *s = sounds;
  for(s=sounds; s->name!=NULL && (strncmp(inst,s->name,l) != 0); s++);
  return((s->note) ? s->note : atoi(inst));
}

void midi_writenote(unsigned int delta_t, 
                    unsigned char instrument_number, 
                    unsigned int velocity) {
  midi_write_var(delta_t);
  if(unstuck) {
    fputc(0x99, fdout); /* re-attach to the drum channel, note on events */
    unstuck = FALSE; /* we're no longer unstuck */
  }
  midi_write_var((unsigned long int)instrument_number);
  midi_write_var(velocity);
}
void midi_writenotetext(unsigned int delta_t, 
                        const char *inst, 
                        unsigned int velocity) {
  unsigned char instrument_number = midi_lookup_instrument(inst);
  if(instrument_number != 0) {
    if(verbose) {
      printf("writing: dt=%u(0x%x), inst=%u(%s)(0x%x), vel=0x%x\n",
             delta_t, delta_t,
             instrument_number, inst, instrument_number,
             velocity);
    }
    midi_writenote(delta_t, instrument_number, velocity);
  } else if(inst[0] == '0' && inst[1] == 'x') {
    unsigned int c;
    const char *p;
    midi_write_var(delta_t);
    for(p = &inst[2]; *p!='\0'; p++) {
      while(isspace(*p)) { p++; }
      if(isdigit(*p)) {
        c = (int)(*p-'0') << 4;
      } else {
        c = (int)(toupper(*p) - 'A' + 10 ) << 4;
      }
      p++; /* no nibbles */
      if(*p == '\0') {
        printf("ERROR: invalid hex midi '%s'", inst);
        return;
      }
      if(isdigit(*p)) {
        c += (*p-'0');
      } else {
        c += (toupper(*p) - 'A' + 10 );
      }
      if(verbose) {
        printf("writing: char %x\n", c);
      }
      (void)fputc((unsigned int)c,fdout);
    }
    unstuck = TRUE;
  } else {
    printf("undefined instrument: %s\n", inst);
  }
}

static boolean write_beats(const struct beatpair *bp, void *user) {
  unsigned int *now = (unsigned int *)user;
  unsigned int absolute_offset = 
    (120U * 4U) * (bp->beat) / bp->parent->measure_beats;
  unsigned int relative_offset = absolute_offset - (*now);
  /* base volume + base * emph/5, but at least 1. */
  unsigned int good_volume = max(1U, s_volume 
    + (unsigned int)((float)bp->emph * s_volume / 5.0));

  relative_offset += leftover_delta_t;
  leftover_delta_t = 0;
  midi_writenotetext(relative_offset, 
                     bp->inst, 
                     min(MAX_VELOCITY, good_volume));
  (*now) = absolute_offset;
  return TRUE;
}

void midi_writepattern(const beatpairlist bpl) {
  int now = 0;
  if(!bpm_has_been_set) {
    printf("error: @set bpm [x] annotation is required before patterns can be used\n");
    exit(EXIT_FAILURE);
  }
  assert(bpl != NULL);
  (void)bp_q_iterate(bpl->q, write_beats, &now);
  leftover_delta_t += 120*4 - now;
}

void midi_writetext(unsigned char type, const char *annot) {
  /*@+charint@*/
  unsigned char prefix[] = { 0xff, 0x01 };
  /*@-charint@*/
  prefix[1]=type;
  if(annot!=NULL && annot[0]!='\0') {
    midi_write_var(leftover_delta_t);
    leftover_delta_t=0;
    (void)fwrite(prefix, 2, 1, fdout);
    midi_write_var(strlen(annot));
    (void)fwrite(annot, strlen(annot), 1, fdout);
  }
}

void midi_writelyric(const char *annot) {
  if(embedlyrics && annot) {
    midi_writetext(0x01, annot);
  }
}

void midi_writetitle(const char *title) {
  if(title)
    midi_writetext(0x03, title);
}

static int bpl_compare(const struct beatpair *a, const struct beatpair *b) {
  if(a->beat < b->beat ) {
    return -1;
  } else if(a->beat > b->beat) {
    return 1;
  } else {
    return 0;
  }
}
beatpairlist bpl_new(void) {
  NEWPTRX(struct beatpairlist, bpl);
  bpl->q = bp_q_new(bpl_compare, free);
  bpl->measure_beats = 0;
  assert(bp_q_length(bpl->q) == 0);
  return (bpl);
}

static boolean make_beatpairs(int beat, void *tmpl) {
  struct beatpair *template = (struct beatpair *)tmpl;
  NEWPTRX(struct beatpair, newbp);

  assert(template != NULL);
  memcpy(newbp, template, sizeof(struct beatpair));
  newbp->beat = beat;
  bp_q_insert(template->parent->q, newbp);
  return TRUE;
}

static boolean merge_helper(const struct beatpair *bp, 
                            void *a) {
  queue dstq = (queue) a;
  NEWPTRX(struct beatpair, newbp);
  memcpy(newbp, bp, sizeof(struct beatpair));
  q_insert(dstq, newbp);
  return TRUE;
}

boolean bpl_merge(beatpairlist dst,
                  beatpairlist src) {
  if(dst->measure_beats != 0 && dst->measure_beats != src->measure_beats) {
    printf("ERROR: unable to merge pattern in %d beats with new pattern in %d beats.", 
           src->measure_beats, dst->measure_beats);
    return FALSE;
  }
  dst->measure_beats = src->measure_beats;
  bp_q_iterate(src->q, merge_helper, dst->q);
  return TRUE;
}

void bpl_insert(beatpairlist bpl, 
                sorted_intlist beats, 
                const char *inst, 
                int emph) {
  struct beatpair template;
  template.beat = 0;
  template.emph = emph;
  template.inst = inst;
  if(midi_lookup_instrument(inst) || (inst[0] == '0' && inst[1] == 'x')) {
    template.parent = bpl;
    (void)si_iterate(beats, make_beatpairs, &template);
  } else {
    struct sound *s;
    printf("ERROR: unknown instrument %s, attempting to recover, next time try:\n", inst);
    for(s=sounds; s->name!=NULL; s++) {
      printf("%s, ", s->name);
    }
    printf("\n");
  }
}

void bpl_set_measure_beats(beatpairlist bpl, int mb) {
  if( bpl->measure_beats != 0 && bpl->measure_beats != mb ) {
    printf("ERROR: tried to merge measures of different beats. bad behavior will result.\n");
  }
  bpl->measure_beats = mb;
}

static boolean bp_print(const struct beatpair *bp, void *fileptr) {
  FILE *fpout = (FILE *)fileptr;
  fprintf(fpout, " %d %s\n", bp->beat, bp->inst);
  return TRUE;
}

void bpl_print(FILE *fpout, beatpairlist bpl) {
  assert(bpl->q != NULL);
  fprintf(fpout, "beatpairlist:\n");
  (void)bp_q_iterate(bpl->q, bp_print, fpout);
}
