function [R, p, E] = chold(A)
%CHOLD   Computes a Cholesky decomposition.
%   CHOLD(A) uses only the diagonal and upper triangle of A.
%   The lower triangular is assumed to be the (complex conjugate)
%   transpose of the upper.  If A is positive definite, then
%   R = CHOL(A) produces an upper triangular R so that R'*R = A.
%   If A is not positive definite, an error message is printed.
%
%   [R, p] = CHOLD(A) returns the Cholesky factor R and p = 0
%   when A is positive definite.  If A is not positive definite,
%   it returns a partial Cholesky decomposition and a Schur
%   complement.  Specifically,
%
%      R(1:p-1,1:p-1)'*R(1:p-1,:) = A(1:p-1,:)             (*)
%
%   and
%
%      R(p:n, p:n) = triu(A(p:n,p:n) - R(1:p,:)*R(1:p,:))  (**)
%
%   is the upper half of the Schur complement of A(1:p-1,1:p-1)
%   in A.
%
%   [R, p, E] = CHOLD(A) returns a pivoted Cholesky factorization;
%   i.e., an upper triangular matrix R and a permutation matrix
%   E such that R'*R = E'*A*E.  The permutation is chosen so
%   that
%
%        R(k,k) >= norm(R(k:n,j)) (k=1:n-1, j=k+1:n)
%   
%   If p > 0, a pivoted partial decomposition is returned that
%   satisfies (*) and (**) when A is replaced by E'*A*E.

R = flap(A);
R = triu(R.d);

[m,n] = size(R);

if m ~= n
   error('Flap: The matrix must be square.')
end

if any(imag(diag(R)))
   error('Flap: The matrix must have real diagonals.')
end

t = zeros(n,1);

pivot = false;
if nargout == 3
   E = eye(n);
   pivot = true;
end

for k=1:n

   if pivot
      [val,  pvt] = max(diag(R(k:n,k:n)));
      pvt = pvt+k-1;
   else
      val = R(k,k);
   end

   if val <= 0
      if nargout == 1
         error('Flap: The matrix is not positive definite.')
      else
         p = k;
         return;
      end
   end

   if pivot
      temp = R(1:k-1,k);
      R(1:k-1,k) = R(1:k-1,pvt);
      R(1:k-1,pvt) = temp;
      temp = R(k,k);
      R(k,k) = R(pvt,pvt);
      R(pvt,pvt) = temp;
      R(k,pvt)= conj(R(k,pvt));
      for j=k+1:pvt-1
          temp = conj(R(k,j));
          R(k,j) = conj(R(j,pvt));
          R(j,pvt) = temp;
      end
      temp = R(k,pvt+1:n);
      R(k,pvt+1:n) = R(pvt,pvt+1:n);
      R(pvt,pvt+1:n) =  temp;
      temp = E(1:n,k);
      E(1:n,k) = E(1:n,pvt);
      E(1:n,pvt) = temp;
   end

   R(k,k:n) = FlapRound(R(k,k:n)/FlapRound(sqrt(R(k,k))));
   
   for  j= k+1:n
      t(j) = conj(R(k,j));
      R(k+1:j,j) =  FlapMinus(R(k+1:j,j), FlapRound(t(k+1:j)*R(k,j)));
   end
end
if nargout > 0;
   p = 0;
end
R = flap(R);



